/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import java.util.*;
import java.io.*;
import java.beans.*;

/**
 * Abstract class for all the 3D objects (even for non-graphical 3D objects).
 * 
 * @author Artiste on the Web
 */

public abstract class Element3D extends AbstractBean implements NamedObject, Lockable {
  private static final long serialVersionUID = -8135113902270590169l;
  
  /**
   * Creates a new element.
   */
  public Element3D() {  }
  /**
   * Creates a new element, with the given name.
   */
  public Element3D(String newName) { name = newName; }

  protected String name;
  /**
   * Gets this element's name.
   * @return the name
   */
  public final String getName() { return name; }
  /**
   * Checks if this element has a name.
   * @return true if the name property is non-null and has a length > 0
   */
  public final synchronized boolean hasName() { return (name != null && !name.equals("")); }
  /**
   * Sets this element's name.
   * @param n the new name
   */
  public void setName(String n) {
    String oldName;
    synchronized(this) {
      oldName = name;
      name = n;
    }
    firePropertyChange("name", oldName, name);
  }

  /**
   * This element's parent. You should not change this field !
   */
  protected World3D parent;
  /**
   * Called when the element is added into a world. You can override this method if needed;
   * don't forget to call the super implementation.
   * @param into the world
   */
  protected synchronized void added(World3D into) { parent = into; }
  /**
   * Called when the element is removed from a world. You can override this method if needed;
   * don't forget to call the super implementation.
   * @param from the world
   */
  protected synchronized void removed(World3D from) { parent = null; }

  /**
   * Remove this element from its parent (if it has one).
   */
  public final synchronized void remove() {
    World3D parent2;
    synchronized(this) { parent2 = parent; }
    if(parent2 != null) parent2.remove(this);
  }

  /**
   * Gets this element's parent.
   * @return the parent world
   */
  public final World3D getParent() { return parent; }
  /**
   * Gets this element's root parent. Its root parent is a world that has no parent, and that
   * contains this element.
   * @return the root parent world
   */
  public final World3D getRootParent() {
    World3D parent2;
    synchronized(this) {
      if(parent == null) return null;
      if(parent.parent == null) return parent;
      parent2 = parent; // Avoid deadlock !
    }
    return parent2.getRootParent();
  }
  /**
   * Gets this element's environment.
   * @return the environment
   */
  public Environment3D getEnvironment() {
    World3D parent2;
    synchronized(this) {
      if(parent instanceof Environment3D) return (Environment3D) parent;
      if(parent == null) return null;
      parent2 = parent;
    }
    return parent2.getEnvironment();
  }
  /**
   * Checks if this element is inside the given world. The element is inside if the world is
   * its parent, or if its parent is inside the world.
   * @param w the world
   * @return true if inside
   */
  public final boolean isInside(World3D w) {
    World3D parent2;
    synchronized(this) {
      if(parent == w   ) return true ;
      if(parent == null) return false;
      parent2 = parent;
    }
    return parent2.isInside(w);
  }
  
  // Overrides :
  /**
   * Checks if it is really necessary to fire an event. This method returns false if there is
   * no listener or if the element is lock, else it returns true.
   * @see opale.soya.soya3d.Element3D#lock
   * @return true if firing an event if necessary
   */
  protected boolean isWorthFiringEvent() {
    if(lockLevel > 0) return false;
    return super.isWorthFiringEvent();
  }

  protected transient int lockLevel;
  /**
   * Checks if this element is locked.
   * @see opale.soya.soya3d.Element3D#lock
   * @return true if locked
   */
  public boolean isLocked() { return lockLevel > 0; }
  /**
   * Locks this element. A locked element will send no event until it will be unlocked. If you
   * intend to change, move, rotate, resize or animate many time in succession the same element,
   * you should lock it, change it and unlock it. Only a single event will be sent, when
   * unlocking, so there may be a performance boost (in particular if the event treatment is
   * complex). But locking element for nothing can waste time.
   * Notice that locking is cumulative : if you lock twice an object, you need to unlock it
   * twice before it can be refreshed or rebuilt.
   * The lock-state is not saved in serialization.
   */
  public synchronized void lock() { lockLevel++; }
  /**
   * Unlocks this element.
   * @see opale.soya.soya3d.Element3D#lock
   */
  public void unlock() {
    synchronized(this) {
      if(lockLevel == 0) return;
      lockLevel--;
    }
    if(lockLevel == 0) firePropertyChange();
  }
  
  /**
   * Clones this element.
   * @return the clone.
   */
  public synchronized Object clone() {
    Element3D el = null;
    try { el = (Element3D) getClass().newInstance(); }
    catch(Exception e) { System.out.println("Cannot create a new object : " + getClass().getName()); return null; }
    el.name = name;
    return el;
  }
  
  /**
   * Gets a short description of this element (shorter that toString()).
   * @return the description
   */
  public synchronized String getDescription() {
    if(hasName()) return name  +  " (" + getClass().getName() + ")";
    else          return "(no name) (" + getClass().getName() + ")";
  }
  /**
   * Gets a string that describes all the properties of this element.
   * @return the string
   */
  public synchronized String propertiesString() {
    if(parent == null) return "name : " + name + "\n" + "parent : null\n";
    else return "name : " + name + "\n" + "parent : " + parent.getDescription();
  }
  /**
   * Gets a complete description of this element.
   * @return the complete description
   */
  public final String toString() { return getClass().getName() + " {\n" + propertiesString() + "\n}"; }
}
