/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.Matrix;
import opale.soya.soya2d.*;
import opale.soya.soya3d.model.*;
import gl4java.*;
import java.util.*;

/**
 * Collects fragments and sort them by material ; can make a significative speed boost (if no/low
 * video memory).
 * 
 * @author Artiste on the Web
 */

// Optimizable by re-using Record and float[16] object (instead creating new ones each frame).

public class DrawablesCollectorByMaterial extends Object implements DrawablesCollector {
  public DrawablesCollectorByMaterial(Renderer r) { renderer = r; }
  public void reset() { // Before a new use.
    //fs  = new HashMap(lastSize );
    //fsa = new HashMap(lastSizea);
    fs .clear();
    fsa.clear();
    specialEffects.clear();
  }

  private transient int lastSize, lastSizea; // "a" stands for "alpha".
  //private transient Map fs, fsa;
  private transient Map fs = new HashMap(), fsa = new HashMap();
  
  private transient java.util.Collection specialEffects = new java.util.Vector();;
  public void addSpecialEffect(SpecialEffect s) { specialEffects.add(s); }
  public void removeSpecialEffect(SpecialEffect s) { specialEffects.remove(s); }

  private transient Renderer renderer;
  public Renderer getRenderer() { return renderer; }

  transient GLFunc  gl ;
  transient GLUFunc glu;
  public void setGLandGLU(GLFunc gl, GLUFunc glu) {
    this.gl  = gl ;
    this.glu = glu;
  }

  public void collectAll(java.util.Collection c, float[] mat) {
    for(Iterator i = c.iterator(); i.hasNext(); ) collect((Fragment) i.next(), mat);
  }
  public void collect(Drawable f, float[] mat) { // Sort by alpha-use and then by material.
    Material m = f.getMaterial();

    if(f.getUseAlpha()) {
      java.util.Collection materialCol = (java.util.Collection) fsa.get(m);
      if(materialCol == null) {
        materialCol = new java.util.Vector();
        fsa.put(m, materialCol);
      }
      materialCol.add(new Record(f, mat));
    }
    else {
      java.util.Collection materialCol = (java.util.Collection) fs.get(m);
      if(materialCol == null) {
        materialCol = new java.util.Vector();
        fs.put(m, materialCol);
      }
      materialCol.add(new Record(f, mat));
    }
  }

  public void draw() {
    Record r;
    for(Iterator i = fs.keySet().iterator(); i.hasNext(); ) {
      Material m = (Material) i.next();
      java.util.Collection materialCol = (java.util.Collection) fs.get(m);
      
      m.makeCurrent(gl, glu);
      for(Iterator j = materialCol.iterator(); j.hasNext(); ) ((Record) j.next()).draw();
      m.unmakeCurrent(gl, glu);
    }
    lastSize  =  fs.size();
  }
  public void drawAlpha() {
    Record r;
    for(Iterator i = fsa.keySet().iterator(); i.hasNext(); ) {
      Material m = (Material) i.next();
      java.util.Collection materialCol = (java.util.Collection) fsa.get(m);
      
      m.makeCurrent(gl, glu);
      for(Iterator j = materialCol.iterator(); j.hasNext(); ) ((Record) j.next()).draw();
      m.unmakeCurrent(gl, glu);
    }
    lastSizea = fsa.size();
  }
  
  public void drawZOnly() {
    Material.WHITE_MATERIAL.makeCurrent(gl, glu);
    for(Iterator i = fs.keySet().iterator(); i.hasNext(); ) {
      Material m = (Material) i.next();
      java.util.Collection materialSet = (java.util.Collection) fs.get(m);

      for(Iterator j = materialSet.iterator(); j.hasNext(); ) ((Record) j.next()).drawZOnly();
    }
    Material.WHITE_MATERIAL.unmakeCurrent(gl, glu);
    lastSize  =  fs.size();
  }
  public void drawAlphaZOnly() {
    for(Iterator i = fsa.keySet().iterator(); i.hasNext(); ) {
      Material m = (Material) i.next();
      java.util.Collection materialSet = (java.util.Collection) fsa.get(m);

      m.makeCurrent(gl, glu);
      for(Iterator j = materialSet.iterator(); j.hasNext(); ) ((Record) j.next()).drawZOnly();
      m.unmakeCurrent(gl, glu);
    }
    lastSizea = fsa.size();
  }

  public String toString() {
    String s = getClass().getName() + "@ {\n" + "Non-alpha part :\n";
    for(Iterator i = fs.values().iterator(); i.hasNext(); ) {
      Set materialSet = (Set) i.next();
      for(Iterator j = materialSet.iterator(); j.hasNext(); ) s = s + j.next().toString() + "\n";
    }
    s = s + "Alpha part :\n";
    for(Iterator i = fsa.values().iterator(); i.hasNext(); ) {
      Set materialSet = (Set) i.next();
      for(Iterator j = materialSet.iterator(); j.hasNext(); ) s = s + j.next().toString() + "\n";
    }
    s = s + "}";
    return s;
  }
  
  // ConfigurationChanger :
  private transient boolean inverted;
	public boolean isLeftHanded () { return  inverted; }
	public boolean isRightHanded() { return !inverted; }
  public void setLeftHanded () { inverted = true ; }
  public void setRightHanded() { inverted = false; }
	public void invertConfiguration() { inverted = !inverted; }
  
  private final class Record {
    public Record(Drawable frag, float[] mat) {
      f = frag;
      m = mat;
      inverted = isLeftHanded();
      if(!specialEffects.isEmpty())
        mySpecialEffects = (SpecialEffect[]) specialEffects.toArray(new SpecialEffect[0]);
    }
    private Drawable f;
    private float[] m;
    private boolean inverted;
    private SpecialEffect[] mySpecialEffects;
    public void setFragment(Drawable frag, float[] mat) {
      f = frag;
      //if(m != null) Matrix.matrixRelease(m); // Release the former matrix.
      m = mat;
      inverted = isLeftHanded();
    }
    
    public void draw() {
      if(inverted) gl.glFrontFace(GLEnum.GL_CW);
      gl.glLoadMatrixf(m);
      if(mySpecialEffects == null) f.draw(renderer, gl, glu);
      else                         f.draw(renderer, gl, glu, mySpecialEffects);
      if(inverted) gl.glFrontFace(GLEnum.GL_CCW);
    }
    public void drawZOnly() {
      if(inverted) gl.glFrontFace(GLEnum.GL_CW);
      gl.glLoadMatrixf(m);
      if(mySpecialEffects == null) f.draw(renderer, gl, glu);
      else                         f.draw(renderer, gl, glu, mySpecialEffects);
      if(inverted) gl.glFrontFace(GLEnum.GL_CCW);
    }
    public String toString() {
      String s = getClass().getName() + "@ {\n";
      s = s + "Fragment :\n" + f.toString() + "\n";
      s = s + "Matrix :\n" + Matrix.matrixToString(m) + "/n";
      s = s + "}";
      return s;
    }
  }
}
