/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import gl4java.*;

/**
 * A coordinates system is a frame defined by 3 vectors x, y and z.
 * 
 * The x direction is the right, the y the up and the z the back, so -z is the front.
 * 
 * By default, a coordinates system is direct; you can define an indirect coordinates system
 * by setting one or all of its axis to a negative value.
 * 
 * @author Artiste on the Web
 */

public interface CoordSyst {
  /**
   * Converts the given point from this coordinates system to the root coordinates system.
   * @param p the point (float[3])
   */
  void convertPointFrom (float[] p);
  /**
   * Converts the given point from the root coordinates system to this coordinates system.
   * @param p the point (float[3])
   */
  void convertPointTo   (float[] p);
  /**
   * Converts the given vector from this coordinates system to the root coordinates system.
   * @param v the vector (float[3])
   */
  void convertVectorFrom(float[] v);
  /**
   * Converts the given vector from the root coordinates system to this coordinates system.
   * @param v the vector (float[3])
   */
  void convertVectorTo  (float[] v);
  
  /**
   * Makes this coordinates system current for futher GL drawing.
   * @param r the renderer
   * @param gl  the gl
   * @param glu the glu
   */
  //void makeCurrent(Renderer r, GLFunc gl, GLUFunc glu);
  /**
   * Unmakes this coordinates system current.
   * @param r the renderer
   * @param gl  the gl
   * @param glu the glu
   */
  //void unmakeCurrent(Renderer r, GLFunc gl, GLUFunc glu);
  
  /**
   * Gets the matrix that defines this coordinates system.
   * @return the matrix (float[16])
   */
  public float[] getMatrix();
  /**
   * Gets the product of all matrixes : ... X (this.getCoordSyst().getMatrix()) X (this.getMatrix).
   * @return the root matrix (float[16])
   */
  public float[] getRootMatrix();
  /**
   * Gets the inversion of the root matrix.
   * @return the inverted root matrix (float[16])
   */
  public float[] getInvertedRootMatrix();
  
  /**
   * Checks if this coordinates system is left-handed.
   * @return true if left-handed
   */
  public boolean isLeftHanded();
  /**
   * Checks if this coordinates system is right-handed.
   * @return true if right-handed
   */
  public boolean isRightHanded();
  
  /**
   * Gets the x axis of this coordinates system.
   * Warning : this method is not a property; calling it twice may not return the same
   * object, and moving the returned vector may not change the coordinates system.
   */
  public Vector x();
  /**
   * Gets the y axis of this coordinates system.
   * Warning : this method is not a property; calling it twice may not return the same
   * object, and moving the returned vector may not change the coordinates system.
   */
  public Vector y();
  /**
   * Gets the z axis of this coordinates system.
   * Warning : this method is not a property; calling it twice may not return the same
   * object, and moving the returned vector may not change the coordinates system.
   */
  public Vector z();
  /**
   * Gets the origin of the coordinates system.
   * Warning : this method is not a property; calling it twice may not return the same
   * object, and moving the returned position may not change the coordinates system.
   */
  public Position origin();
  
  /**
   * gets the parent coordinates system of this coordinates system.
   * @return the parent coordinates system.
   */
  public CoordSyst getCoordSyst();
  
  /**
   * gets the root coordinates system of this coordinates system.
   * @return the root coordinates system.
   */
  public CoordSyst getRootCoordSyst();
}
