/*
 * Soya3D
 * Copyright (C) 1999  Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.util.*;
import java.util.Arrays;
import java.util.Iterator;
import java.beans.*;

/**
 * A box is a DimensionWrapper that wrap all object into a parallelepiped (= a box).
 * 
 * @see opale.soya.soya3d.DimensionWrapper
 * 
 * @author Artiste on the Web
 */

public class Box implements DimensionWrapper {
  public Box(Position newMin, Position newMax) { this(newMin, newMax, new Point(0f, 0f, 0f, newMin.getCoordSyst())); }
  public Box(Position newMin, Position newMax, Position newOrigin) {
    imin = newMin;
    imax = newMax;
    iorigin = newOrigin;
    min = imin;
    max = imax;
    origin = iorigin;
  }
  public Box(Dimension d) { // Box this Dimension :).
    DimensionWrapper b = d.wrapper();
    imin = new Point(b.getMin());
    imax = new Point(b.getMax());
    iorigin = new Point(b.getOrigin());
    min = imin;
    max = imax;
    origin = iorigin;
  }
  public Box(java.util.Collection dimensions, CoordSyst f) { this(dimensions.iterator(), f); }
  public Box(java.util.Collection dimensions, CoordSyst f, Position newOrigin) { this(dimensions.iterator(), f, newOrigin); }
  public Box(Iterator dimensions, CoordSyst f) { this(dimensions, f, new Point(0f, 0f, 0f, f)); }
  public Box(Iterator dimensions, CoordSyst f, Position newOrigin) {
    DimensionWrapper b;
    imin = new Point(Float.POSITIVE_INFINITY, Float.POSITIVE_INFINITY, Float.POSITIVE_INFINITY, f);
    imax = new Point(Float.NEGATIVE_INFINITY, Float.NEGATIVE_INFINITY, Float.NEGATIVE_INFINITY, f);
    for( ; dimensions.hasNext(); ) {
      b = ((Dimension) dimensions.next()).wrapper();
      b.setCoordSyst(f);
      b.minimize(imin);
      b.maximize(imax);
    }
    iorigin = newOrigin;
    min = imin;
    max = imax;
    origin = iorigin;
  }
  
  public Object clone() {
    return new Box((Position) imin.clone(), (Position) imax.clone(), (Position) iorigin.clone());
  }
  
  public void addWrapper(DimensionWrapper w) {
    w.setCoordSyst(imin.getCoordSyst());
    w.minimize(imin);
    w.maximize(imax);
  }
  
  private final Position imin; // i for initial.
  private final Position imax;
  private final Position iorigin;

  private Position origin;
  private Position min   ;
  private Position max   ;
  public Position getOrigin() { return origin; }
  public Position getMin   () { return min   ; }
  public Position getMax   () { return max   ; }

  public DimensionWrapper wrapper() { return this; } // Box is a Box :-).

  public CoordSyst getInitialCoordSyst() { return imin.getCoordSyst(); }
  public CoordSyst getCoordSyst() { return min.getCoordSyst(); }
  public void setCoordSyst(CoordSyst f) {
    if((f == null) || (min.getCoordSyst() == null)) { // No conversion.
      min.setCoordSyst(f);
      max.setCoordSyst(f);
    }
    else {
      if(f == getInitialCoordSyst()) {
        min = imin;
        max = imax;
        origin = iorigin;
      }
      else convert(f);
    }
  }
  private void convert(CoordSyst f) {
    origin = iorigin.clone(f);
    
    CoordSyst oldF = imin.getCoordSyst(); // should assume that imin and imax have the same coordSyst.
    Position lowLeftFrontCorner = imin.clone(f);
    min = lowLeftFrontCorner;
    max = new Point(lowLeftFrontCorner);
    
    Vector v = new Vector(imax.getX() - imin.getX(), 0f, 0f, oldF);
    v.setCoordSyst(f);
    v.minimize(min); v.maximize(max);
    
    v.setCoordSyst(null); v.setCoordSyst(oldF); // Avoid useless coordSyst conversion.
    v.move(0f, imax.getY() - imin.getY(), 0f);
    v.setCoordSyst(f);
    v.minimize(min); v.maximize(max);
    
    v.setCoordSyst(null); v.setCoordSyst(oldF); // Avoid useless coordSyst conversion.
    v.move(0f, 0f, imax.getZ() - imin.getZ());
    v.setCoordSyst(f);
    v.minimize(min); v.maximize(max);
  }

  public float getWidth () { return max.getX() - min.getX(); }
  public float getHeight() { return max.getY() - min.getY(); }
  public float getDepth () { return max.getZ() - min.getZ(); }
  
  // No factor here.
  public float getXFactor() { return 1f; }
  public float getYFactor() { return 1f; }
  public float getZFactor() { return 1f; }
  public void setXFactor(float f) {  }
  public void setYFactor(float f) {  }
  public void setZFactor(float f) {  }
  
  // Temporary
  /*
  public float[] getFactors() {
    float[] fs = { 1f, 1f, 1f };
    return fs;
  }
  public float[] getDims() {
    float[] fs = { getWidth(), getHeight(), getDepth() };
    return fs;
  }
  */
  
  public void scale(float f) { scale(f, f, f); }
  public void scale(float x, float y, float z) {
    Vector v = new Vector(iorigin, imin);
    imin.move(iorigin.getX() + x * v.getX(), iorigin.getY() + y * v.getY(), iorigin.getZ() + z * v.getZ());
           v = new Vector(iorigin, imax);
    imax.move(iorigin.getX() + x * v.getX(), iorigin.getY() + y * v.getY(), iorigin.getZ() + z * v.getZ());
    if(getCoordSyst() != getInitialCoordSyst()) convert(getCoordSyst());
  }
  public void setWidth (float f) { scale(f / getWidth (), 1f, 1f); }
  public void setHeight(float f) { scale(1f, f / getHeight(), 1f); }
  public void setDepth (float f) { scale(1f, 1f, f / getDepth() ); }
  public void setDims(float w, float h, float d) { scale(w / getWidth(), h / getHeight(), d / getDepth()); }
  
  public void minimize(Position p) {
    //if((p.getCoordSyst() == min.getCoordSyst()) || (p.getCoordSyst() == null) || (min.getCoordSyst() == null))
    min.minimize(p);
  }
  public void maximize(Position p) {
    max.maximize(p);
  }
  
  public float getSquareRadius() {
    return Matrix.pow2(Math.max(Math.abs(min.getX()), Math.abs(max.getX()))) +
           Matrix.pow2(Math.max(Math.abs(min.getY()), Math.abs(max.getY()))) +
           Matrix.pow2(Math.max(Math.abs(min.getZ()), Math.abs(max.getZ())));
  }
  
  // No event : never change!
  public void addPropertyChangeListener   (PropertyChangeListener listener) {  }
  public void removePropertyChangeListener(PropertyChangeListener listener) {  }
  public java.util.Collection propertyChangeListeners() { return null;  }
  public void firePropertyChange() {  }
  public void firePropertyChange(String propertyName) {  }
  public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {  }
  public void firePropertyChange(PropertyChangeEvent e) {  }
  public void fireResize() {  }
  public void fireResize(String propertyName) {  }
  public void fireResize(String propertyName, Object oldValue, Object newValue) {  }
}
