/*
 * Soya3D
 * Copyright (C) 1999  Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya2d;

import opale.soya.*;
import opale.soya.awt.*;
import opale.soya.util.AbstractBean;
import gl4java.*;
import java.util.*;
import java.awt.*;
import java.awt.image.*;
import java.io.*;

/**
 * Abstract class for all kind of image.
 * 
 * @author Artiste on the Web
 */

public abstract class Image extends AbstractBean {
  private static final long serialVersionUID = 2332159127765675165l;

  /**
   * Cheks if this image use the alpha channel.
   * @return true if use alpha blending.
   */
  public abstract boolean getUseAlpha();
  
  /**
   * Cheks if this image is transparent or semi-transparent.
   * @return true if transparent.
   */
  public abstract boolean isTransparent();
  
  private int width, height;
  /**
   * Gets the width of this image.
   * @return the width in pixel.
   */
  public int getWidth() { return width; }
  /**
   * Gets the height of this image.
   * @return the height in pixel.
   */
  public int getHeight() { return height; }

  private byte[] data; // Optimizable (Don't keep it in memory if they are in the GL).
  /**
   * Gets the image data.
   * @return the image data, stored in an array of byte.
   */
  public byte[] getImageData() { return data; }
  
  protected String imageFileName = "";
  /**
   * Gets the image file name, or null if it has no attached file name. It is not the file
   * where this object is stored, but the file name of the image file (.gif,...) from
   * witch it was created.
   * @return the image file name.
   */
  public String getImageFileName() { return imageFileName; }
  /**
   * Sets the image file name. The image will be loaded into this image.
   * @param fileName the image file name.
   */
  public synchronized void setImageFileName(String fileName) throws InterruptedException {
    try {
      imageFileName = fileName;
      update();
      firePropertyChange("imageFileName");
    }
    catch(Exception e) {
      e.printStackTrace();
      throw new InterruptedException();
    }
  }

  /**
   * Gets an awt.image that correspond to this opengl image. The result id null if no file
   * name is associated with this image.
   * @return the image or null
   */
  public synchronized java.awt.Image getImage() {
    if(imageFileName == null || imageFileName == "") return null;
    java.awt.Image i;
    try {
      i = Toolkit.getDefaultToolkit().getImage(imageFileName);
      MediaTracker mt = new MediaTracker(new Canvas());
      mt.addImage(i, 0);
      mt.waitForAll();
    }
    catch(Exception e) {
      e.printStackTrace();
      return null;
    }
    return i;
  }
  /**
   * Defines this opengl image from an awt.Image.
   * If the image is loaded from a file, use rather setImageFileName(String) .
   * @param an awt.Image
   */
  public synchronized void setImage(java.awt.Image i) throws InterruptedException {
    width  = i.getWidth (null);
    height = i.getHeight(null);
    data = computeImageToData(i);
    firePropertyChange("image");
  }
  /**
   * Computes the given awt.Image into an array of byte, according to the properties of 
   * this image.
   * @param i the awt image
   * @return the image data
   */
  protected abstract byte[] computeImageToData(java.awt.Image i) throws InterruptedException;
  
  /**
   * Updates this image from the image filename. Call this if the image file has change and
   * you want to update the image.
   */
  public synchronized void update() throws InterruptedException { setImage(getImage()); }

  /**
   * Draws the image woth no scaling.
   * @param gl  the gl for drawing
   * @param glu the glu for drawing
   * @param x the x coordinate in pixel
   * @param y the y coordinate in pixel
   */
  public abstract void draw(GLFunc gl, GLUFunc glu, int x, int y);
  /**
   * Draws the image, stretched.
   * @param gl  the gl for drawing
   * @param glu the glu for drawing
   * @param x the x coordinate in pixel
   * @param y the y coordinate in pixel
   * @param w the width in pixel
   * @param h the height in pixel
   */
  public abstract void draw(GLFunc gl, GLUFunc glu, int x, int y, int w, int h);

  // static stuff:
  /**
   * Converts an awt.image to an array of byte, with 3 color components for each pixel.
   * @param i the awt image
   * @return the image data
   */
  protected static byte[] dataFromImageRGB(java.awt.Image i) throws InterruptedException {
    int width  = i.getWidth (null);
    int height = i.getHeight(null);
    PixelGrabber pg = new PixelGrabber(i, 0, 0, width, height, true);
    pg.startGrabbing();
    pg.grabPixels();
    Object o = pg.getPixels();
    if(o instanceof int[]) {
      byte[] data = new byte[width * height * 3];
      int[] p = (int[]) o;
      int j = 0;
      for(int k = 0; k < width * height; k++) {
        data[j++] = getRed  (p[k]);
        data[j++] = getGreen(p[k]);
        data[j++] = getBlue (p[k]);
      }
      return data;
    }
    return (byte[]) o;
  }
  /**
   * Converts an awt.image to an array of byte, with 4 color components for each pixel
   * (= with an alpha channel).
   * @param i the awt image
   * @return the image data
   */
  protected static byte[] dataFromImageRGBA(java.awt.Image i) throws InterruptedException {
    int width  = i.getWidth (null);
    int height = i.getHeight(null);
    PixelGrabber pg = new PixelGrabber(i, 0, 0, width, height, true);
    pg.startGrabbing();
    pg.grabPixels();
    Object o = pg.getPixels();
    if(o instanceof int[]) {
      byte[] data = new byte[width * height * 4];
      int[] p = (int[]) o;
      int j = 0;
      for(int k = 0; k < width * height; k++) {
        data[j++] = getRed  (p[k]);
        data[j++] = getGreen(p[k]);
        data[j++] = getBlue (p[k]);
        data[j++] = getAlpha(p[k]);
      }
      return data;
    }
    return (byte[]) o;
  }
  /**
   * Gets the red components of a pixel color value.
   * @param pixel the pixel color
   * @return the red component
   */
  protected static byte getRed  (int pixel) { return (byte) ((pixel >> 16) & 0xff); }
  /**
   * Gets the green components of a pixel color value.
   * @param pixel the pixel color
   * @return the green component
   */
  protected static byte getGreen(int pixel) { return (byte) ((pixel >>  8) & 0xff); }
  /**
   * Gets the blue components of a pixel color value.
   * @param pixel the pixel color
   * @return the blue component
   */
  protected static byte getBlue (int pixel) { return (byte) ((pixel      ) & 0xff); }
  /**
   * Gets the alpha components of a pixel color value.
   * @param pixel the pixel color
   * @return the alpha component
   */
  protected static byte getAlpha(int pixel) { return (byte) ((pixel >> 24) & 0xff); }
}
