/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.editor;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import java.awt.event.*;

/**
 * An abstract class for 3D handles used by the GUIEditor.
 * 
 * You must override the doSelection, doClick and doMovement method.
 *
 * Add the handle into your cursor's handles collection (soya.editor.Handles), if you want
 * the handle to be functionnal.
 * 
 * A handle can have different color, we suggest to choose this color according to its
 * functionnality. Selected handles are always green.
 * 
 * @author Artiste on the Web
 */

public abstract class Handle3D extends Volume3D {
  /**
   * Creates a new handle.
   */
  public Handle3D() {  }
  /**
   * Creates a new handle, with the given color.
   */
  public Handle3D(int newNaturelColor) {
    this();
    setNaturelColor(newNaturelColor);
    setColorToNaturelColor();
  }

  /**
   * Gets the object that this handle can move/modify.
   */
  public abstract Object getEditedObject();

  private boolean selected;
  /**
   * Checks if this handle is selected. A handle is selected if the cursor is over it. Selected
   * handles become green.
   * @return true if selected
   */
  public boolean isSelected() { return selected; }
  /**
   * Sets if this handle is selected. A handle is selected if the cursor is over it. Selected
   * handles become green. If you unselect a clicked handle, it will be unclicked first.
   * @param b true if selected
   */
  public final synchronized void setSelected(boolean b) {
    if((selected != b) && (visible)) {
      if(b == false && isClicked() == true) setClicked(false); // Unclick first.
      if(b) {
        setColor(GREEN);
        if(showCoordinates == SHOW_COORDINATES_IF_SELECTED) showCoordinates();
      }
      else {
        setColorToNaturelColor();
        if(showCoordinates == SHOW_COORDINATES_IF_SELECTED) hideCoordinates();
      }
      selected = b;
      doSelection(selected);
      firePropertyChange("selected"); 
    }
  }
  /**
   * Called when the handle is selected or unselected. You must override ths method.
   * @param selected true if selected
   */
  protected abstract void doSelection(boolean selected);

  private boolean clicked;
  /**
   * Checks if this handle is Clicked. A handle is clicked if the cursor was pressed when it
   * was selected.
   * @return true if clicked
   */
  public boolean isClicked() { return clicked; }
  /**
   * Sets if this handle is clicked. A handle is clicked if the cursor was pressed when it
   * was selected. If you click an unselected handle, it will be selected first.
   * @param b true if clicked
   */
  public final synchronized void setClicked(boolean b) { 
    if((clicked != b) && (visible)) {
      if(b == true && isSelected() == false) setSelected(true); // Select first.
      doClick(b);
      clicked = b;
      realPosition = new Point(this); // reset it. (?).
      firePropertyChange("clicked"); 
    }
  }
  /**
   * Called when the handle is clicked or unclicked. You must override ths method.
   * @param clicked true if clicked
   */
  protected abstract void doClick(boolean clicked);

  private float step = .1f;
  /**
   * Gets the step of this handle. It represent its size.
   * @return the step
   */
  public float getSpet() { return step; }
  /**
   * Sets the step of this handle. It represent its size.
   * @param f the new step
   */
  public void setStep(float f) {
    step = f;
    firePropertyChange("step");
  }
  
  /** Useless for you. Called by the 3D cursor. */
  public final void cursorPress(Cursor3D p, MouseEvent e) {
    if(!isVisible()) return;
    if(p.distanceTo(this) < step) setClicked(true);
    else {
      if(!e.isShiftDown()) setClicked(false);
    }
  }
  /** Useless for you. Called by the 3D cursor. */
  public final void cursorRelease(Cursor3D p, MouseEvent e) {  }

  protected Position realPosition;
  /** Useless for you. Called by the 3D cursor. */
  public final synchronized void cursorMove(Cursor3D p, Vector translation, MouseEvent e) {
    if(!isVisible()) return;
    if(!isClicked()) setSelected(p.distanceTo(this) < step);
    else {
      //if((e.getModifiers() & e.BUTTON1_MASK) == e.BUTTON1_MASK) {
      if(p.isPressed()) {
        realPosition.addVector(translation);
        if(p.isMagnetic()) {
          Position magnetizedPosition = new Point(realPosition);
          p.magnetize(magnetizedPosition);
          move(magnetizedPosition);
        }
        else move(realPosition);
        if(areCoordinatesShown()) computeCoordinates();
        doMovement(p, this, e); // Clicked and moved, so should do something.
      }
    }
  }
  /**
   * Called when the handle is moved. You must override ths method.
   * @param cursor the 3D cursor
   * @param position the position suggered by the cursor (including grid-magnetization,...)
   * @param e the associated mouse event
   */
  protected abstract void doMovement(Cursor3D cursor, Position position, MouseEvent e);
  
  //Overrides :
  /**
   * Sets if the handle is visible. Invisible handles are inactive.
   * @param b true if visible and active
   */
  public void setVisible(boolean b) {
    if((b == false) && selected) setSelected(false); // Unselect invisible handles
    super.setVisible(b); // Should be at the end, for raising the event after unselect.
  }
  
  /**
   * Update the position of the handle. Default implementation.
   */
  public void update() {
    //if(realPosition != null) realPosition.move(this);
  }
  
  // Color stuff:
  private int naturelColor;
  /**
   * Gets the natural color of this handle. The natural color is the color when the handle is
   * not selected (selected handles are green).
   * @return a constant that represent the natural color
   */
  public int getNaturelColor() { return naturelColor; }
  /**
   * Sets the natural color of this handle. The natural color is the color when the handle is
   * not selected (selected handles are green).
   * @param color a constant that represent the new natural color
   */
  public void setNaturelColor(int color) { 
    naturelColor = color;
    firePropertyChange("naturelColor", null, null); 
  }

  private int color;
  /**
   * Gets the color of this handle.
   * @return a constant that represent the color
   */
  public int getColor() { return color; }
  /**
   * Sets the color of this handle.
   * @param color a constant that represent the new color
   */
  public synchronized void setColor (int color) { 
    switch(color) {
      case RED   : setRed   (); break;
      case GREEN : setGreen (); break;
      case YELLOW: setYellow(); break;
      case BLUE  : setBlue  (); break;
    }
    this.color = color;
    firePropertyChange("color", null, null); 
  }
  /**
   * Sets this handle to its natural color. The natural color is the color when the handle is
   * not selected (selected handles are green).
   */
  public void setColorToNaturelColor() {
    setColor(getNaturelColor());
  }

  /** Sets this handle to red.    */ public void setRed   () { setShape(getRedCube   ()); }
  /** Sets this handle to green.  */ public void setGreen () { setShape(getGreenCube ()); }
  /** Sets this handle to yellow. */ public void setYellow() { setShape(getYellowCube()); }
  /** Sets this handle to blue.   */ public void setBlue  () { setShape(getBlueCube  ()); }

  /** Constant for handles' colors. */ public static final int RED    = 0;
  /** Constant for handles' colors. */ public static final int GREEN  = 1;
  /** Constant for handles' colors. */ public static final int YELLOW = 2;
  /** Constant for handles' colors. */ public static final int BLUE   = 3;

  private static Shape redCube   ;
  private static Shape greenCube ;
  private static Shape yellowCube;
  private static Shape blueCube  ;
  /**
   * Gets a red cube.
   * @return a red cube shape
   */
  public static Shape getRedCube() { 
    if(redCube == null) {
      try {
        String s = opale.soya.editor.Editor.class.getResource("redcube.shape").getFile();
        redCube = Shape.load(s);
      }
      catch(Exception e) { System.out.println("Can find red cube" + e); e.printStackTrace(); }
    }
    return redCube;
  }
  /**
   * Gets a green cube.
   * @return a green cube shape
   */
  public static Shape getGreenCube() { 
    if(greenCube == null) {
      try {
        String s = opale.soya.editor.Editor.class.getResource("greencube.shape").getFile();
        greenCube = Shape.load(s);
      }
      catch(Exception e) { System.out.println("Can find green cube"); e.printStackTrace(); }
    }
    return greenCube;
  }
  /**
   * Gets a yellow cube.
   * @return a yellow cube shape
   */
  public static Shape getYellowCube() { 
    if(yellowCube == null) {
      try {
        String s = opale.soya.editor.Editor.class.getResource("yellowcube.shape").getFile();
        yellowCube = Shape.load(s);
      }
      catch(Exception e) { System.out.println("Can find yellow cube" + e); e.printStackTrace(); }
    }
    return yellowCube;
  }
  /**
   * Gets a blue cube.
   * @return a blue cube shape
   */
  public static Shape getBlueCube() { 
    if(blueCube == null) {
      try {
        String s = opale.soya.editor.Editor.class.getResource("bluecube.shape").getFile();
        blueCube = Shape.load(s);
      }
      catch(Exception e) { System.out.println("Can find blue cube" + e); e.printStackTrace(); }
    }
    return blueCube;
  }


  
  /**
   * Constant for showCoordinates property. Shows the handle's coordinates.
   */
  public static final int SHOW_COORDINATES = 0;
  /**
   * Constant for showCoordinates property. Does not show the handle's coordinates.
   */
  public static final int HIDE_COORDINATES = 1;
  /**
   * Constant for showCoordinates property. Shows the handle's coordinates only if selected.
   */
  public static final int SHOW_COORDINATES_IF_SELECTED = 2;
  private int showCoordinates = SHOW_COORDINATES_IF_SELECTED;
  //private TextShape coordinatesShape;
  //private Volume3D coordinates;
  private Label3D coordinates;
  /**
   * Gets if this handle shows its coordinates.
   * @return a symbolic constant
   */
  public int getShowCoordinates() { return showCoordinates; }
  /**
   * Sets if this handle shows its coordinates.
   * @param i a symbolic constant
   */
  public void setShowCoordinates(int i) {
    if(i != showCoordinates) {
      showCoordinates = i;
      switch(showCoordinates) {
      case SHOW_COORDINATES: showCoordinates(); break;
      case HIDE_COORDINATES: hideCoordinates(); break;
      case SHOW_COORDINATES_IF_SELECTED:
        if(isSelected()) showCoordinates();
        else             hideCoordinates();
        break;
      }
      firePropertyChange("showCoordinate");
    }
  }
  private boolean areCoordinatesShown() { return coordinates != null; }
  private void showCoordinates() {
    if(areCoordinatesShown()) return;
    coordinates = new Label3D();
    coordinates.setOverlayed(true );
    coordinates.setLit      (false);
    /*
    coordinatesShape = new TextShape();
    coordinatesShape.setOverlayed(true );
    coordinatesShape.setLit      (false);
    coordinates = new Volume3D(coordinatesShape);
    */
    World3D parent = getParent();
    if(parent != null) parent.add(coordinates);
    computeCoordinates();
  }
  private void hideCoordinates() {
    //coordinatesShape = null;
    coordinates.remove();
    coordinates = null;
  }
  private void computeCoordinates() {
    coordinates.setText(getX() + ", " + getY() + ", " + getZ());
    coordinates.move(this);
  }
  
  // Overrides :
  protected void added(World3D into) {
    super.added(into);
    if(coordinates != null) {
      into.add(coordinates);
      computeCoordinates();
    }
  }
  protected void removed(World3D from) {
    super.removed(from);
    if(coordinates != null) from.remove(coordinates);
  }
}
