/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.editor;

import opale.soya.*;
import opale.soya.util.*;
import java.beans.*;
import java.lang.reflect.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Map;
import java.util.HashMap;

/**
 * A panel bean editor.
 * 
 * This bean editor also edits write only properties.
 * 
 * @author Artiste on the Web
 */

public class BeanEditorPane extends ScrollPane implements PropertyChangeListener {
  public BeanEditorPane() {
    super(ScrollPane.SCROLLBARS_ALWAYS);
    
    addComponentListener(new ComponentAdapter() {
      public void componentShown(ComponentEvent e) {
        doLayout();
        repaint();
      }
    });
  }
  public BeanEditorPane(Object bean) {
    this();
    setBean(bean);
  }
  
  public void doLayout() { if(panel != null) panel.doLayout(); }
  
  //public Dimension getDefaultSize() {
  public Dimension getPreferredSize() {
    if(panel != null) {
      Insets i = getInsets();
      Insets i2 = panel.getInsets();
      Dimension p = panel.getPreferredSize();
      
      int width  = p.width  + i.left + i.right  + i2.left + i2.right  + 4;
      int height = p.height + i.top  + i.bottom + i2.top  + i2.bottom;
      
      height = height + getHScrollbarHeight();
      width = width + getVScrollbarWidth();
      return new Dimension(width, height);
    }
    else return new Dimension(0, 0);
  }
  
  private PropertySheetPanel panel;
  private Map oldPanels = new HashMap();
  
  public void setBean(Object bean) {
    BeanInfo bi;
    try { bi = Introspector.getBeanInfo(bean.getClass()); }
    catch (IntrospectionException e) {
      System.out.println("PropertySheet: Couldn't introspect."); e.printStackTrace();
      return;
    }
    setBean(bean, bi);
  }
  public void setBean(Object bean, BeanInfo bi) {
    if((panel == null) || (panel.getBean().getClass() != bean.getClass())) {
      if(panel != null) {
        remove(panel);
        oldPanels.put(panel.getBean().getClass(), panel);
      }
      panel = (PropertySheetPanel) oldPanels.get(bean.getClass());
      if(panel == null) panel = new PropertySheetPanel();
      
      add(panel);
    }
    panel.setBean(bean, bi);
    
    if(!AbstractBean.addPropertyChangeListenerToBean(bean, this)) System.out.println("Warning : can't add a property change listener to this bean");
  }
  public void finalize() throws java.lang.Throwable {
    super.finalize();
    if(!AbstractBean.removePropertyChangeListenerToBean(panel.getBean(), this)) System.out.println("Warning : can't remove a property change listener to this bean");
  }
  
  public void propertyChange(PropertyChangeEvent e) { if(panel != null) panel.propertyChange(e); }
}

class EditedProperty {
  public EditedProperty(Object bean, PropertyDescriptor pd) throws Exception {
    descriptor = pd;
    
    String name = descriptor.getDisplayName();
    if(descriptor.isHidden() || descriptor.isExpert()) throw new IntrospectionException(name + " is hidden or expert"); //return;
    Method setter = descriptor.getWriteMethod();
    if((setter == null) && (descriptor.getPropertyEditorClass() == null)) throw new IntrospectionException(name + " has no setter");
    Method getter = descriptor.getReadMethod();
    Class type = descriptor.getPropertyType();

    try {
      Class pec = descriptor.getPropertyEditorClass();
      if (pec != null) {
        try { editor = (PropertyEditor) pec.newInstance(); }
        catch (Exception ex) { System.out.println("Can't create the specified Editor! Try create default..."); ex.printStackTrace(); }
      }
      if(editor == null) editor = PropertyEditorManager.findEditor(type);
      if(editor == null) throw new IntrospectionException(name + " has no editor");

      if(getter != null) {
        value = getter.invoke(bean, NO_ARG);
        editor.setValue(value);
      }

      if (editor.supportsCustomEditor())    view = editor.getCustomEditor();
      else {
        if (getter == null) { // write only.
          if (editor.getTags() != null)     view = new PropertySelectorWriteOnly(editor); // Tagged
          else                              view = new PropertyTextWriteOnly(editor); // Text/Number
        }
        else {                // read and write.
          if (editor.getTags() != null)     view = new PropertySelector(editor); // Tagged
          else                              view = new PropertyText(editor); // Text/number
        }
      }
    } 
    catch (Exception e) { throw e; }
    label = new Label(name, Label.RIGHT);
  }
  private static final Object[] NO_ARG = {  };
  
  public PropertyDescriptor descriptor;
  public PropertyEditor editor;
  public Object value;
  public Component view;
  public Label label;
  
  public boolean isDisplayable() { return ((label != null) && (view != null)); }
}

class PropertySheetPanel extends Panel implements PropertyChangeListener {
  PropertySheetPanel() { super(null); }

  private static final Object[] NO_ARG = {  };

  private Object bean;
  public Object getBean() { return bean; }
  private EditedProperty[] properties;

  private boolean processEvents;
  private static int hPad = 4, vPad = 4;

  synchronized void setBean(Object bean) {
    BeanInfo bi;
    try { bi = Introspector.getBeanInfo(bean.getClass()); }
    catch (IntrospectionException e) {
      System.out.println("PropertySheet: Couldn't introspect."); e.printStackTrace();
      return;
    }
    setBean(bean, bi);
  }
  synchronized void setBean(Object bean, BeanInfo bi) {
    processEvents = true;
    if((this.bean == null) || (this.bean.getClass() != bean.getClass())) {
      if(properties != null) {
        removeAll();
      }
      setVisible(false);
      
      this.bean = bean;
      PropertyDescriptor[] descriptors;
      descriptors = bi.getPropertyDescriptors();
      java.util.Collection props = new java.util.Vector(descriptors.length);
      
      for (int i = 0; i < descriptors.length; i++) {
        EditedProperty ep;
        try { ep = new EditedProperty(bean, descriptors[i]); }
        catch(Exception e) {
          System.err.println("Warning: " + e.getMessage() + ". Skipping.");
          continue;
        }
        props.add(ep);
        if(ep.editor != null) ep.editor.addPropertyChangeListener(this);
        if(ep.isDisplayable()) {
          add(ep.label);
          add(ep.view);
        }
      }
      properties = (EditedProperty[]) props.toArray(new EditedProperty[0]);
      
      // Sort.
      Arrays.sort(properties, new Comparator() {
        public int compare(Object o1, Object o2) {
          EditedProperty ep1 = (EditedProperty) o1;
          EditedProperty ep2 = (EditedProperty) o2;
          return ep1.descriptor.getDisplayName().compareTo(ep2.descriptor.getDisplayName());
        }
        public boolean equals(Object o) { return o.getClass() == getClass(); }
      });
      
      processEvents = true;
      setVisible(true);
      doLayout();
    }
    else { // Reuse the same.
      this.bean = bean;
      refresh();
      processEvents = true;
    }
  }

  public void doLayout() {
    if(!processEvents) return;
    if(properties == null) return;
    
    // First figure out the size of the columns.
    int labelWidth = 0, viewWidth = 120, w;
    for(int i = 0; i < properties.length; i++) {
      EditedProperty ep = properties[i];
      if(ep.isDisplayable()) {
        w = ep.label.getPreferredSize().width;
        if (w > labelWidth) labelWidth = w;
        if(ep.view instanceof Container) ((Container) ep.view).doLayout();
        w = ep.view.getPreferredSize().width;
        if (w > viewWidth) viewWidth = w;
      }
    }
    int width = 3 * hPad + labelWidth + viewWidth;
    
    // Now position all the components.
    int y = vPad;
    for(int i = 0; i < properties.length; i++) {
      EditedProperty ep = properties[i];
      if(ep.isDisplayable()) {
        ep.label.setBounds(hPad, y + 5, labelWidth, 25);
        Dimension viewSize = ep.view.getPreferredSize();
        int h = viewSize.height;
        if(h < 30) h = 30;
        ep.view.setBounds(labelWidth + 2 * hPad, y, viewWidth, h);
        y = y + h + vPad;
      }
    }
    setSize(width, y + vPad);
    
    Dimension apreferredSize = preferredSize;
    preferredSize = new Dimension(width, y + vPad);
  }
  private Dimension preferredSize;
  public Dimension getPreferredSize() { 
    if(preferredSize == null) return super.getPreferredSize();
    else return preferredSize; 
  }

  public synchronized void propertyChange(PropertyChangeEvent e) {
    if(!processEvents) return;
    processEvents = false; // Avoid that the refresh at the end of this method causes another property change that causes another refresh...
    
    if(e.getSource() instanceof PropertyEditor) {
      PropertyEditor editor = (PropertyEditor) e.getSource();
      for(int i = 0 ; i < properties.length; i++) {
        if(properties[i].editor == editor) {
          EditedProperty ep = properties[i];
          PropertyDescriptor descriptor = ep.descriptor;
          Object value = editor.getValue();
          ep.value = value;
          Method setter = descriptor.getWriteMethod();
          if(setter != null) {
            try {
              Object args[] = { value };
              setter.invoke(bean, args);
            }
            catch (InvocationTargetException e2) {
              if(e2.getTargetException() instanceof PropertyVetoException) { System.err.println("Warning: Vetoed; reason is: " + e2.getTargetException().getMessage()); }
              else { System.out.println("InvocationbeanException while updating " + descriptor.getName() + "."); e2.printStackTrace(); }
            }
            catch(Exception e2) { System.out.println("Unexpected exception while updating " + descriptor.getName() + "."); e2.printStackTrace(); }
            if(ep.view != null && ep.view instanceof PropertyCanvas) ep.view.repaint();
          }
          break;
        }
      }
    }
    refresh(); // Required, because an other property may has changed.
    processEvents = true;
  }


  // Refreshing is multithreaded for speed boost!!!
  private class Refresher extends Thread {
    public boolean canCoalesce = true;
    public boolean hasFinish;
    public void run() {
      try { Thread.currentThread().sleep(100); } // The secret of the speed! Just wait a little before refreshing, just in case there would be another refreshing called...
      catch(Exception e) {  }
      canCoalesce = false;
      immediateRefresh();
      hasFinish = true;
    }
  }
  private Refresher refresher;
  private synchronized void postRefresh() {
    if(refresher == null) {
      refresher = new Refresher();
      refresher.start();
    }
    else {
      if(refresher.hasFinish) {
        refresher = new Refresher();
        refresher.start();
        return;
      }
      if(!refresher.canCoalesce) {
        restartResfresh = true;
        /*
        refresher.stop();
        refresher = new Refresher();
        refresher.start();
        */
      } // Else, a former refreshing has been invoked, but it is always sleeping. Let it refresh for us!
    }
  }
  private boolean restartResfresh;
  private synchronized void immediateRefresh() {
    processEvents = false; // Avoid that the refresh at the end of this method causes another property change that causes another refresh...
    
    for(int i = 0; i < properties.length; i++) {
      Object o;
      EditedProperty ep = properties[i];
      
      Method getter = ep.descriptor.getReadMethod();
      if(getter != null) {
        try { o = getter.invoke(bean, NO_ARG); }
        catch(ThreadDeath e2) { throw(e2); }
        catch(Exception e2) { System.out.println("Can't read property " + ep.descriptor.getDisplayName() + "."); e2.printStackTrace(); continue; }
        if(!((o == ep.value) || (o != null && o.equals(ep.value)))) { // The property has changed.
          ep.value = o;
          ep.editor.setValue(o);
          if(ep.view != null) ep.view.repaint();
        }
      }
      if(restartResfresh) {
        restartResfresh = false;
        i = -1;
      }
    }
    if(Beans.isInstanceOf(bean, Component.class)) ((Component) (Beans.getInstanceOf(bean, Component.class))).repaint();
    processEvents = true;
  }
  private void refresh() { postRefresh(); }
}
