/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.awt;

import opale.soya.*;
import opale.soya.soya2d.*;
import gl4java.*;

/**
 * Quality is the static class that inits all rendering contexts according to the desired
 * level of quality.
 * 
 * You can choose the quality of soya with the Quality.quality field.
 * 
 * @author Artiste on the Web
 */

public class Quality extends Object {
  private Quality() {  }
  
  /** Low graphic quality for rendering. Use it if you don't have a 3D card. :-( */
  public static final int QUALITY_LOW = -1;
  /** Normal graphic quality for rendering. Use it if you have a normal 3D card. :-| */
  public static final int QUALITY_NORMAL = 0;
  /** Hight graphic quality for rendering. Use it if you have a super 3D card. :-) */
  public static final int QUALITY_HIGH = 1;

  /**
   * Current quality. Should take the value of a QUALITY_* constant. Default is
   * QUALITY_NORMAL.
   */
  public static int quality = QUALITY_NORMAL;

  /**
   * True for using mipmaps. Default is true.
   */
  public static boolean useMipmaps = true;
  
  /**
   * Inits the GL with soya default values.
   * @param gl  the gl for calling gl method (should be the gl of a rendering surface). 
   * @param glu the glu for calling gl method (should be the gl of a rendering surface). 
   */
  public static void initContext(GLFunc gl, GLUFunc glu) {
    /*
    float bias = -.2f, scale = 1.5f;
    gl.glPixelTransferf(GLEnum.GL_RED_BIAS  , bias);
    gl.glPixelTransferf(GLEnum.GL_GREEN_BIAS, bias);
    gl.glPixelTransferf(GLEnum.GL_BLUE_BIAS , bias);
    gl.glPixelTransferf(GLEnum.GL_ALPHA_BIAS, bias);
    
    gl.glPixelTransferf(GLEnum.GL_RED_SCALE  , scale);
    gl.glPixelTransferf(GLEnum.GL_GREEN_SCALE, scale);
    gl.glPixelTransferf(GLEnum.GL_BLUE_SCALE , scale);
    gl.glPixelTransferf(GLEnum.GL_ALPHA_SCALE, scale);
    */
    
    
    gl.glEnable (GLEnum.GL_NORMALIZE);
    
    gl.glEnableClientState(GLEnum.GL_VERTEX_ARRAY);
    
    gl.glClearDepth(1f);
    gl.glEnable(GLEnum.GL_DEPTH_TEST);
    
    gl.glBlendFunc(GLEnum.GL_SRC_ALPHA, GLEnum.GL_ONE_MINUS_SRC_ALPHA);
    gl.glAlphaFunc(GLEnum.GL_GREATER, 0.5f);
    gl.glEnable(GLEnum.GL_CULL_FACE);
    
    gl.glEnable(GLEnum.GL_LIGHTING);
    gl.glMaterialfv(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_AMBIENT, Material.BLACK_COLOR);
    gl.glColorMaterial(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_AMBIENT_AND_DIFFUSE);
    gl.glLightModelf(GLEnum.GL_LIGHT_MODEL_TWO_SIDE, 0);
    //gl.glLightModeli(gl.GL_LIGHT_MODEL_COLOR_CONTROL, gl.GL_SEPARATE_SPECULAR_COLOR);
    
    gl.glEnable (gl.GL_POINT_SMOOTH);
    gl.glDisable(gl.GL_POLYGON_SMOOTH);
    gl.glShadeModel(GLEnum.GL_SMOOTH);
    /*
    if(quality != QUALITY_LOW) gl.glShadeModel(GLEnum.GL_SMOOTH);
    else                       gl.glShadeModel(GLEnum.GL_FLAT  );
    */
    gl.glDisable(GLEnum.GL_DITHER);
    //gl.glEnable(GLEnum.GL_DITHER);
    gl.glPixelStorei(GLEnum.GL_UNPACK_ALIGNMENT, 1);
    gl.glTexParameterf(GLEnum.GL_TEXTURE_2D, GLEnum.GL_TEXTURE_WRAP_S, GLEnum.GL_REPEAT);
    gl.glTexParameterf(GLEnum.GL_TEXTURE_2D, GLEnum.GL_TEXTURE_WRAP_T, GLEnum.GL_REPEAT);
    
    int error = gl.glGetError();
    if(error != 0) System.out.println("Error while initializing context :" + error + " : " + glu.gluErrorString(error));
    
    makeQualityCurrent(gl, glu);
  }
  
  private static void makeQualityCurrent(GLFunc gl, GLUFunc glu) {
    int flag = gl.GL_DONT_CARE;
    switch(quality) {
    case QUALITY_LOW:
      flag = gl.GL_FASTEST;
      break;
    case QUALITY_NORMAL:
      flag = gl.GL_DONT_CARE;
      break;
    case QUALITY_HIGH:
      flag = gl.GL_NICEST;
      break;
    }
    gl.glHint(gl.GL_FOG_HINT, flag);
    gl.glHint(gl.GL_PERSPECTIVE_CORRECTION_HINT, flag);
    gl.glHint(gl.GL_POINT_SMOOTH_HINT, flag);
    gl.glHint(gl.GL_LINE_SMOOTH_HINT, flag);
    gl.glHint(gl.GL_POLYGON_SMOOTH_HINT, flag);
    
    makeQualityCurrentForTexture(gl, glu);
    
    int error;
    error = gl.glGetError();
    if(error != 0) System.out.println("Error while making quality current :" + error + " : " + glu.gluErrorString(error));
  }
  public static void makeQualityCurrentForTexture(GLFunc gl, GLUFunc glu) {
    if(useMipmaps) {
      switch(quality) {
      case QUALITY_LOW:
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MIN_FILTER, gl.GL_NEAREST_MIPMAP_NEAREST);
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MAG_FILTER, gl.GL_NEAREST);
        break;
      case QUALITY_NORMAL:
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MIN_FILTER, gl.GL_NEAREST_MIPMAP_NEAREST);
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MAG_FILTER, gl.GL_LINEAR );
        break;
      case QUALITY_HIGH:
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MIN_FILTER, gl.GL_LINEAR_MIPMAP_LINEAR );
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MAG_FILTER, gl.GL_LINEAR );
        break;
      }
    }
    else {
      switch(quality) {
      case QUALITY_LOW:
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MIN_FILTER, gl.GL_NEAREST);
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MAG_FILTER, gl.GL_NEAREST);
        break;
      case QUALITY_NORMAL:
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MIN_FILTER, gl.GL_NEAREST);
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MAG_FILTER, gl.GL_LINEAR );
        break;
      case QUALITY_HIGH:
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MIN_FILTER, gl.GL_LINEAR );
        gl.glTexParameterf(gl.GL_TEXTURE_2D, gl.GL_TEXTURE_MAG_FILTER, gl.GL_LINEAR );
        break;
      }
    }
  }
  


  private static final int conservedStates = GLEnum.GL_ENABLE_BIT | GLEnum.GL_LIGHTING_BIT;

  /**
   * Sets the OpenGL states, as required for drawing overlays (modify the projection and
   * modelview matrix, and disable states that are useless for overlays, like fog, texture,
   * lighting,...).
   * Must be call before drawing overlays.
   * @param gl  the gl for calling gl method (should be the gl of the rendering surface). 
   * @param glu the glu for calling gl method (should be the gl of the rendering surface). 
   * @param rs the rendering surface in whitch you want to render overlay.
   */
  public static void makeOverlaysCurrent(GLFunc gl, GLUFunc glu, RenderingSurface rs) { 
    gl.glPushAttrib(conservedStates);
    gl.glDisable(GLEnum.GL_LIGHTING);
    gl.glDisable(GLEnum.GL_FOG);
    gl.glDisable(GLEnum.GL_BLEND);
    gl.glDisable(GLEnum.GL_DITHER);
    gl.glDisable(GLEnum.GL_COLOR_MATERIAL);
    gl.glDisable(GLEnum.GL_DEPTH_TEST);
    gl.glDisable(GLEnum.GL_CULL_FACE);
    gl.glDisable(GLEnum.GL_TEXTURE_2D);
    gl.glShadeModel(GLEnum.GL_FLAT);
    gl.glColor3f(1f, 1f, 1f);
    
    gl.glMatrixMode(GLEnum.GL_PROJECTION);
    gl.glPushMatrix();
    gl.glLoadIdentity();
    glu.gluOrtho2D(0d, (double) rs.getSurfaceWidth(), 0d, (double) rs.getSurfaceHeight());
    
    gl.glMatrixMode(GLEnum.GL_MODELVIEW);
    gl.glPushMatrix();
    gl.glLoadIdentity();
    
    int error;
    error = gl.glGetError();
    if(error != 0) System.out.println("Error while making overlay current :" + error + " : " + glu.gluErrorString(error));
  }
  /**
   * Unsets the OpenGL states required for drawing overlays (restore the projection and
   * modelview matrix, and re-enable states that were useless for overlays, like fog,
   * texture, lighting,...).
   * Must be call after drawing overlays.
   * @param gl  the gl for calling gl method (should be the gl of the rendering surface). 
   * @param glu the glu for calling gl method (should be the gl of the rendering surface). 
   * @param rs the rendering surface in whitch you have finish to render overlay.
   */
  public static void unmakeOverlaysCurrent(GLFunc gl, GLUFunc glu, RenderingSurface rs) {
    gl.glPopAttrib();
    
    gl.glMatrixMode(GLEnum.GL_PROJECTION);
    gl.glPopMatrix();
    
    gl.glMatrixMode(GLEnum.GL_MODELVIEW);
    gl.glPopMatrix();

    int error;
    error = gl.glGetError();
    if(error != 0) System.out.println("Error while unmaking overlay current :" + error + " : " + glu.gluErrorString(error));
  }
}
