/*
Copyright (C) 2000  Olivier Cioni

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/


package opale.m2d.ker;
import opale.tools.*;

/**
* Cette classe est la classe de base de l'objet gomtrique reprsentant un vecteur dans le plan.
* @author O.C.
* @version 0.1
*/

public class Vector2D extends Object2D implements Transformable, Cloneable
{

// membres protected ou public ??!!
/**
* Membres publiques.
* Reprsentent les composantes du vecteur.
*/
protected double x,y;	// les composantes du vecteur

/**
* Constructeur par dfaut.
* @since Opale 0.1
*/
public Vector2D()
	{
	super();
	System.out.println("ici dan Vector2D");
	}


/**
* Le constructeur par dfaut
* Construit et initialise le vecteur avec les coordonnes (0; 0) et l'origine  l'origine du repre.
* @param OIJ rep, le repre de dfinition du vecteur
* @since Opale 0.1
*/
public Vector2D(OIJ rep)
	{
	super(rep);
	}

/**
* Construit et initialise le vecteur avec les coordonnes (x; y).
* @param OIJ rep, le repre de dfinition du vecteur
* @param double x,y les coordonnes du vecteur.
* @since Opale 0.1
*/
public Vector2D(OIJ rep,double x, double y)
	{
	super(rep);
	this.x = x;
	this.y = y;
	}

/**
* Le constructeur de copie
* @param Vector2D v, le vecteur  copier
* @since Opale 0.1
*/
public Vector2D(Vector2D v)
	{
	super(v.getOIJ());
	this.x = v.x;
	this.y = v.y;
	}

/**
* Construit  partir de deux instances de Point2D.
* @param Point2D p l'origine du vecteur;
* @param Point2D q l'extrmit du vecteur.
* @since Opale 0.1
*/
public Vector2D(Point2D p, Point2D q)
	{
	super(p.getOIJ());
	Point2D qq = (Point2D) q.clone();
	qq.changeOIJ(p.getOIJ());
	
	this.x = qq.x -p.x;
	this.y = qq.y - p.y;
	}

//Les methodes publiques

/**
* Renvoie la premire composante du vecteur.
* @return double la premire composante.
* @since Opale 0.1
*/
public final double getX()
	{
	return x;
	}

/**
* Renvoie la deuxime composante du vecteur.
* @return double la deuxime composante.
* @since Opale 0.1
*/
public final double getY()
	{
	return y;
	}

/**
* Fixe la premire composante du vecteur.
* @param double x, la nouvelle valeur composante.
* @since Opale 0.1
*/
public final void setX(double x)
	{
	this.x = x;
	}

/**
* Fixe la deuxime composante du vecteur.
* @param double y, la nouvelle valeur composante.
* @since Opale 0.1
*/
public final void setY(double y)
	{
	this.y = y;
	}



/**
* Fixe les composantes du vecteur.
* @param x, y les nouvelles composantes.
* @since Opale 0.1
*/
public final void set(double x, double y)
	{
	this.x = x;
	this.y = y;
	}

/**
* Fixe les composantes du vecteur  la m^eme valeur qu'un autre vecteur.
* @param Vector2D v, le vecteur  copier.
* @since Opale 0.1
*/
public final void set(Vector2D v)
	{
	Vector2D vv = (Vector2D) v.clone();
	vv.changeOIJ(getOIJ());
	
	set(vv.x,vv.y);
	}




/**
* Renvoie le carr de la norme du vecteur dans un repere orthonorm.
* @return double le carr de la norme.
* @since Opale 0.1
*/
public final double normSq()
	{
	return getOIJ().normSq(x,y);
	}

/**
* Renvoie la norme du vecteur dans un repre orthonorm.
* @return double la norme.
* @since Opale 0.1
*/
public final double norm()
	{
	return Math.sqrt(normSq());
	}

/**
* Renvoie le carr de la norme d'un vecteur.
* @param double x,y les composantes du vecteur;
* @return double le carr de la norme.
* @since Opale 0.1
*/
static final public double normSq(double x, double y)
	{
	return x*x + y*y;
	}

/**
* Renvoie la norme d'un vecteur.
* @param double x,y les composantes du vecteur;
* @return double la norme.
* @since Opale 0.1
*/
static public double norm(double x, double y)
	{
	return Math.sqrt(normSq(x,y));
	}

/**
* Renvoie le carr de la norme d'un vecteur.
* @param Vecteur2D v le vecteur;
* @return double le carr de la norme.
* @since Opale 0.1
*/
static public double normSq(Vector2D v)
	{
	return v.getOIJ().normSq(v.x,v.y);
	}

/**
* Renvoie la norme d'un vecteur.
* @param Vecteur2D v le vecteur;
* @return double la norme.
* @since Opale 0.1
*/
static public double norm(Vector2D v)
	{
	return Math.sqrt(normSq(v));
	}





/**
* Cette mthode teste si le vecteur est gal  un objet pass en argument.
* @param Object obj un objet  comparer avec le vecteur.
* @return <code>true</code> si l'objet  comparer est une instance de Vector2D et est gal au vecteur courant; <code>false</code> sinon.
* @since Opale 0.1
*/
public boolean equals(Object obj)
	{
	if (obj instanceof Vector2D)
		{
        	Vector2D v = (Vector2D) obj;
		return (x == v.x) && (y == v.y) && (getOIJ().equals(v.getOIJ()));
        	}
    	return false;
    }

/**
* Cre un nouvel objet de mme classe et de mme contenu.
* @return Object un clone de l'objet.
* @exception  OutOfMemoryError s'il n'y a pas assez de mmoire.
* @see        java.lang.Cloneable
* @since Opale 0.1
*/
public Object clone()
	{
	Vector2D v = (Vector2D) super.clone();
	v.x = x;
	v.y = y;
	return v;
	}



//les mthodes de l'interface Transformable
public void translate(double dx,double dy)
	{
	}


	
public void rotate(double theta)
	{
	Point2D p = new Point2D(getOIJ(),x,y);
	p.rotate(theta);
	set(p.x,p.y);
	}
	
public void rotate(double x, double y,double theta)
	{
	this.translate(-x,-y);
	this.rotate(theta);
	this.translate(x,y);

	}

/**
* Effectue un changement de repere du vecteur.
* @param OIJ rep, le nouveau repere.
* @since Opale 0.1
*/
public void changeOIJ(OIJ rep)
	{
	Matrix2D mv = new Matrix2D();
	
	getOIJ().matPassage(rep,mv);
	
	double[] q = mv.compute(x,y);
	double[] p = mv.compute(0,0);

	set(q[0]-p[0],q[1]-p[1]);	
	setOIJ(rep);
	}
	
public String toString()
	{
	String s="("+this.x+"; "+this.y+")";
	return s;
	}

public void writeFileMore(WFile f, Pack2D p)
	{
	f.writeln("X " +x);
	f.writeln("Y " +y);
	}	
public int readKeyWord(String word, RFile f, Pack2D p) throws java.io.IOException, MalformedFileException
	{
	int type;
	if ( word.equals("X"))
		{
		type = f.next();
		if (type != RFile.RF_DOUBLE) return -1;
		x = f.dval;
		return 0;
		}
	else if ( word.equals("Y"))
		{
		type = f.next();
		if (type != RFile.RF_DOUBLE) return -1;
		y = f.dval;
		return 0;
		}
	return -1;
	}
}
