/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "Om.h"
#include "OmApp.h"	
#include "config.h"
#include "tuning.h"
#include <sys/mman.h>
#include <iostream>
#include <unistd.h>
#include "JackDriver.h"
#include "NodeFactory.h"
#include "OSCReceiver.h"
#include "OSCSender.h"
#include "Patch.h"
#include "ObjectStore.h"
#include "Maid.h"
#ifdef HAVE_ALSA
#include "AlsaDriver.h"
#endif
#ifdef HAVE_LASH
#include "LashDriver.h"
#endif
using std::cout; using std::cerr; using std::endl;

namespace Om {


OmApp::OmApp(const char* const port)
: m_maid(new Maid(maid_queue_size, maid_rate)),
  m_jack_driver(new JackDriver()),
#ifdef HAVE_ALSA
  m_alsa_driver(new AlsaDriver()),
#else
  m_alsa_driver(NULL),
#endif
  m_osc_receiver(new OSCReceiver(port)),
  m_osc_sender(new OSCSender(m_osc_receiver)),
  m_object_store(new ObjectStore()),
  m_node_factory(new NodeFactory()),
  m_zero_buffer(NULL),
  m_quit_flag(false),
  m_activated(false)
{
	if (mlockall(MCL_CURRENT | MCL_FUTURE))
		cerr << "[Main] Warning:  Failed to lock memory.  Timing could suffer." << endl;
	else
		cout << "[Main] Successfully locked all memory." << endl;

	m_maid->start();
}


OmApp::~OmApp()
{
	disable();
	deactivate();

	m_maid->stop();

	for (List<Patch*>::iterator i = m_patches.begin(); i != m_patches.end(); ++i)
		if ((*i)->parent() == NULL)
			delete (*i);
	
	delete m_object_store;
	delete m_osc_sender;
	delete m_osc_receiver;
	delete m_node_factory;
	delete m_alsa_driver;
	delete m_jack_driver;
	
	delete m_maid;

	//munlockall();
}


int
OmApp::main()
{
	// Loop until quit flag is set (by OSCReceiver)
	while ( ! m_quit_flag) {
		usleep(250000); // quarter second
#ifdef HAVE_LASH
		if (lash_driver->enabled())
			lash_driver->process_events();
#endif
	}
	cout << "[Main] Done main loop." << endl;
	
	if (m_activated) {
		disable();
		deactivate();
	}

	sleep(1);
	cout << "[Main] Om exiting..." << endl;
	
	return 0;
}


void
OmApp::activate()
{
	if (m_activated)
		return;
	
	if (m_zero_buffer != NULL)
		free(m_zero_buffer);
	
	m_zero_buffer = (sample*)calloc(m_jack_driver->buffer_size(), sizeof(sample));
	m_jack_driver->activate();
#ifdef HAVE_ALSA
	m_alsa_driver->activate();
#endif
	m_activated = true;
}


void
OmApp::deactivate()
{
	if (!m_activated)
		return;
	
	for (List<Patch*>::iterator i = m_patches.begin(); i != m_patches.end(); ++i)
		(*i)->deactivate();
	
	free(m_zero_buffer);
	m_zero_buffer = NULL;

	m_osc_receiver->deactivate();
#ifdef HAVE_ALSA
	m_alsa_driver->deactivate();
#endif 
	m_jack_driver->deactivate();

	m_activated = false;
}


void
OmApp::enable()
{
	m_jack_driver->enable();
#ifdef HAVE_ALSA
	m_alsa_driver->enable();
#endif
}


void
OmApp::disable()
{
#ifdef HAVE_ALSA
	m_alsa_driver->disable();
#endif
	m_jack_driver->disable();
}


void
OmApp::remove_patch(Patch* patch)
{
	List<Patch*>::iterator i = m_patches.begin();
	for ( ; i != m_patches.end(); ++i)
		if (*i == patch) break;
	
	if (i != m_patches.end())
		m_patches.remove(i);
}


} // namespace Om
