/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef PATCH_H
#define PATCH_H

#include <ladspa.h>
#include "Array.h"
#include "NodeTree.h"
#include "NodeBase.h"
#include "List.h"
#include "CrossThreadMutex.h"
#include "AlsaDriver.h"
#include "PluginInfo.h"

using std::string;

namespace Om {

class AlsaPort;
class Connection;
class InputNode;
class OutputNode;
class MidiInNode;


/** A list of nodes in a graph, possibly polyphonic.
 *
 * Note that this is also a Node, just one which contains Nodes.
 * Therefore infinite subpatching is possible, of polyphonic
 * patches of polyphonic nodes etc. etc.
 *
 * \ingroup engine
 */
class Patch : public NodeBase
{
public:
	Patch(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size, uint local_poly);
	virtual ~Patch();

	void activate();
	void deactivate();

	void run(size_t nframes);
	
	virtual Port* const port(const string& port_name) const;
	
	uint num_ports() const;

	void send_creation_messages(lo_address addr) const;
	void send_deletion_messages(lo_address addr) const;
	
	// Patch specific stuff not inherited from Node
	
	void add_node(TreeNode* tn);
	TreeNode* remove_node(TreeNode* tn);

	const List<Connection*>& connections() const { return m_connections; }
	
	void                         add_connection(ListNode<Connection*>* const c) { m_connections.push_back(c); }
	ListNode<Connection*>* const remove_connection(const string& src_port_path, const string& dst_port_path);
	
	List<MidiInNode*>& midi_in_nodes()                            { return m_midi_in_nodes; }
	void                     add_midi_in_node(ListNode<MidiInNode*>* const n) { m_midi_in_nodes.push_back(n); }
	
	List<InputNode*>&        input_nodes()        { return m_input_nodes; }
	const List<InputNode*>&  input_nodes()  const { return m_input_nodes; }
	List<OutputNode*>&       output_nodes()       { return m_output_nodes; }
	const List<OutputNode*>& output_nodes() const { return m_output_nodes; }
	
	List<Node*>* const process_order()                { return m_process_order; }
	void               process_order(List<Node*>* po) { m_process_order = po; }
	
	/** Whether to run this patch's DSP in the audio thread */
	bool process() const { return m_process; }
	void process(bool b) { m_process = b; }

	/** Used to protect access to the node tree while removing (not thread safe) */
	CrossThreadMutex&  node_remove_mutex()            { return m_node_remove_mutex; }

	uint num_in_ports() const; 
	uint num_out_ports() const;

	uint internal_poly() const { return m_internal_poly; }
	
	const NodeTree&    nodes() const { return m_nodes; }

	const PluginInfo& plugin_info() const { return m_plugin_info; }

	void alsa_port(AlsaPort* ap) { m_alsa_port = ap; }
	const AlsaPort* const alsa_port() const { return m_alsa_port; }
	
	snd_seq_event_t* dssi_events_array() { return m_alsa_port->m_dssi_events_array; }
	size_t           dssi_events_size()  { return m_alsa_port->m_dssi_events_size; }
	
private:
	Patch(const Patch& copy) : NodeBase(copy) { throw; }  // disallow copies

	uint              m_internal_poly;
	List<Node*>*      m_process_order;
	List<Connection*> m_connections;
	CrossThreadMutex  m_node_remove_mutex;
	List<MidiInNode*> m_midi_in_nodes;
	List<InputNode*>  m_input_nodes;
	List<OutputNode*> m_output_nodes;
	AlsaPort*         m_alsa_port;
	NodeTree          m_nodes;
	bool              m_process;

	PluginInfo m_plugin_info;
};


} // namespace Om

#endif // PATCH_H
