/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef OSCRECEIVER_H
#define OSCRECEIVER_H

#include <string>
#include <list>
#include <lo/lo.h>

using std::string; using std::list;

namespace Om {

class JackDriver;
class AlsaDriver;
class PluginFactory;
class Patch;


/** Used to store client incoming URLs and the corresponding
 * return address.
 */
class ClientRecord
{
public:
	ClientRecord(const string& hostname, const string& port)
	: m_hostname(hostname),
	  m_port(port),
	  m_addr(lo_address_new(hostname.c_str(), port.c_str()))
	{}

	~ClientRecord()
		{ lo_address_free(m_addr); }

	const string& hostname() const  { return m_hostname; }
	const string& port() const      { return m_port; }
	lo_address    addr() const      { return m_addr; }

private:
	ClientRecord(const ClientRecord& copy) { throw; } // Disallow copies

	string     m_hostname;
	string     m_port;
	lo_address m_addr;
};



// Forgive the extremely long lines below, I like to be able to easily see
// when two lines are identical (ie parameter lists).  This is data more
// than code really...




/** Receives OSC messages and dispatches the appropriate Events.
 *
 * \ingroup engine
 */
class OSCReceiver
{
public:
	OSCReceiver(const char* const port);
	~OSCReceiver();

	void activate();

	bool register_client(const string& host, const string& port);
	bool unregister_client(const string& host, const string& port);
	
	const list<ClientRecord*>& clients() const { return m_clients; }

private:
	const char* const m_port;
	lo_server_thread  m_st;
	
	list<ClientRecord*> m_clients;
	
	lo_address get_reply_addr(lo_message msg);

	static void error_cb(int num, const char* msg, const char* path);	
	static int  generic_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	static int  unknown_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);

	// Static callbacks, these call the non-static member functions below
	inline static int quit_cb                  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int ping_cb                  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int register_client_cb       (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int register_client_host_cb  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int unregister_client_cb     (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int unregister_client_host_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int load_plugins_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int send_plugins_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int send_all_objects_cb      (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int engine_activate_cb       (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int engine_deactivate_cb     (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int engine_enable_cb         (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int engine_disable_cb        (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int create_patch_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int destroy_patch_cb         (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int patch_add_node_cb        (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int enable_patch_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int disable_patch_cb         (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int patch_remove_node_cb     (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int patch_connect_cb         (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int patch_disconnect_cb      (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int patch_disconnect_all_cb  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int patch_request_control_cb (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int set_port_value_cb        (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int set_port_value_voice_cb  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int set_port_value_slow_cb   (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int dssi_cb                  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int note_on_cb               (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int note_off_cb              (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int midi_learn_cb            (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int metadata_get_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);
	inline static int metadata_set_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver);

	int m_quit_cb                  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_ping_cb                  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_register_client_cb       (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_register_client_host_cb  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_unregister_client_cb     (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_unregister_client_host_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_load_plugins_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_send_plugins_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_send_all_objects_cb      (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_engine_activate_cb       (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_engine_deactivate_cb     (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_engine_enable_cb         (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_engine_disable_cb        (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_create_patch_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_destroy_patch_cb         (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_patch_add_node_cb        (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_enable_patch_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_disable_patch_cb         (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_patch_remove_node_cb     (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_patch_connect_cb         (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_patch_disconnect_cb      (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_patch_disconnect_all_cb  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_patch_request_control_cb (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_set_port_value_cb        (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_set_port_value_voice_cb  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_set_port_value_slow_cb   (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_dssi_cb                  (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_note_on_cb               (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_note_off_cb              (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_midi_learn_cb            (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_metadata_get_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
	int m_metadata_set_cb          (const char* path, const char* types, lo_arg** argv, int argc, lo_message msg);
};


inline int OSCReceiver::quit_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_quit_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::ping_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_ping_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::create_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_create_patch_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::destroy_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_destroy_patch_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::register_client_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_register_client_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::register_client_host_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_register_client_host_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::unregister_client_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_unregister_client_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::unregister_client_host_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_unregister_client_host_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::load_plugins_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_load_plugins_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::send_plugins_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_send_plugins_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::send_all_objects_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_send_all_objects_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::engine_activate_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_engine_activate_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::engine_deactivate_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_engine_deactivate_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::engine_enable_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_engine_enable_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::engine_disable_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_engine_disable_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::patch_add_node_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_patch_add_node_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::enable_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_enable_patch_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::disable_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_disable_patch_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::patch_remove_node_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_patch_remove_node_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::patch_connect_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_patch_connect_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::patch_disconnect_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_patch_disconnect_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::patch_disconnect_all_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_patch_disconnect_all_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::patch_request_control_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_patch_request_control_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::set_port_value_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_set_port_value_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::set_port_value_voice_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_set_port_value_voice_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::set_port_value_slow_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_set_port_value_slow_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::dssi_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_dssi_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::note_on_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_note_on_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::note_off_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_note_off_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::midi_learn_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_midi_learn_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::metadata_get_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_metadata_get_cb(path, types, argv, argc, msg);
}

inline int OSCReceiver::metadata_set_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* osc_receiver) {
	return ((OSCReceiver*)osc_receiver)->m_metadata_set_cb(path, types, argv, argc, msg);
}


} // namespace Om

#endif // OSCRECEIVER_H
