/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "NodeBase.h"
#include <cassert>
#include <iostream>
#include <stdint.h>
#include "util.h"
#include "Array.h"
#include "PluginInfo.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "OSCSender.h"
#include "Patch.h"
#include "Om.h"
using std::cout; using std::cerr; using std::endl;

namespace Om {


NodeBase::NodeBase(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size)
: Node(path),
  m_poly(poly),
  m_parent(parent),
  m_srate(srate),
  m_buffer_size(buffer_size),
  m_activated(false),
  m_num_ports(0),
  m_traversed(false),
  m_providers(new List<Node*>()),
  m_dependants(new List<Node*>())
{
	assert(m_parent == NULL || (m_poly == parent->internal_poly() || m_poly == 1));

	//std::cerr << "Creating node " << name << ", poly = " << poly << std::endl;
}


NodeBase::~NodeBase()
{
	//std::cerr << "Node destructor\n";

	//std::cerr << "Destroying " << m_path << "(" << this << ")" << std::endl;

	deactivate();

	delete m_providers;
	delete m_dependants;
	//delete m_plugin_info;

	//for (PortVect::iterator i = m_ports.begin(); i != m_ports.end(); ++i)
	//	delete (*i);
	for (uint i=0; i < m_ports.size(); ++i)
		delete m_ports.at(i);
}


void
NodeBase::activate()
{
	//std::cerr << "Activating node " << m_path << std::endl;

	m_activated = true;
}


void
NodeBase::deactivate()
{
	m_activated = false;
}


void
NodeBase::send_creation_messages(lo_address addr) const
{
	om->osc_sender()->send_node_to(addr, this);
}


void
NodeBase::send_deletion_messages(lo_address addr) const
{
	om->osc_sender()->send_node_removal_to(addr, m_path);
}


/** Runs the Node for the specified number of frames (block size)
 */
void
NodeBase::run(size_t nframes)
{
	assert(m_activated);
	// Mix down any ports with multiple inputs
	Port* p;
	for (uint i=0; i < m_ports.size(); ++i) {
		p = m_ports.at(i);
		p->prepare_buffers();
	}
}


void
NodeBase::set_port_buffer(uint voice, uint port_num, sample* buf)
{
}


Port* const
NodeBase::port(const string& port_name) const
{
	//assert(voice_num < m_poly);
	Port* p = NULL;
	
	for (uint i=0; i < num_ports(); ++i) {
		p = m_ports.at(i);
		if (p->name() == port_name)
			return p;
	}

	cerr << "[NodeBase::port] Failed to find port " << port_name << "!" << endl;

	return NULL;
}


} // namespace Om

