/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "MidiControlNode.h"
#include <math.h>
#include "InputPort.h"
#include "OutputPort.h"
#include "PluginInfo.h"
#include "util.h"

namespace Om {

	
MidiControlNode::MidiControlNode(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size)
: MidiInNode(path, 1, parent, srate, buffer_size)
{
	m_num_ports = 6;
	m_ports.alloc(m_num_ports);

	m_param_port = new InputPort(this, path+"/Controller Number", 0, 1, 
		new PortInfo("Controller Number", CONTROL, INPUT, INTEGER, 60, 0, 127), 1);
	m_param_port->set_value(0, 60.0f); // Set default value
	m_ports.at(0) = m_param_port;
	
	m_log_port = new InputPort(this, path+"/Logarithmic", 0, 1,
		new PortInfo("Logarithmic", CONTROL, INPUT, TOGGLE, 0, 0, 1), 1);
	m_log_port->set_value(0, 0.0f);
	m_ports.at(1) = m_log_port;
	
	m_min_port = new InputPort(this, path+"/Min", 0, 1, 
		new PortInfo("Min", CONTROL, INPUT, NONE, 0, 0, 65535), 1);
	m_min_port->set_value(0, 0.0f); // Set default value
	m_ports.at(2) = m_min_port;
	
	m_max_port = new InputPort(this, path+"/Max", 0, 1, 
		new PortInfo("Max", CONTROL, INPUT, NONE, 1, 0, 65535), 1);
	m_max_port->set_value(0, 1.0f); // Set default value
	m_ports.at(3) = m_max_port;
	
	// Add audio rate out port
	m_audio_port = new OutputPort(this, path+"/Out (AR)", 1, 1, 
		new PortInfo("Out (AR)", AUDIO, OUTPUT, 0, 0, 1), m_buffer_size);
	m_ports.at(4) = m_audio_port;

	// Add control rate out port
	m_control_port = new OutputPort(this, path+"/Out (CR)", 1, 1, 
		new PortInfo("Out (CR)", CONTROL, OUTPUT, 0, 0, 1), m_buffer_size);
	m_ports.at(5) = m_control_port;
	
	m_plugin_info.type(PluginInfo::Internal);
	m_plugin_info.lib_path("");
	m_plugin_info.lib_name("");
	m_plugin_info.plug_label("midi_control_in");
	m_plugin_info.name("Om patch MIDI control input node");
}


void
MidiControlNode::control(int control_num, int val, int start_sample)
{
	assert(start_sample >= 0 && start_sample < static_cast<int>(m_buffer_size));

	sample scaled_value;
	
	sample nval = (val / 127.0f); // normalized: [0, 1]
	
	if (m_log_port->get_value(0, 0) > 0.0f) {
		// haaaaack, stupid negatives and logarithms
		sample offset = 0;
		if (m_min_port->get_value(0, 0) < 0)
			offset = abs(static_cast<int>(m_min_port->get_value(0, 0)));
		const sample min = log(m_min_port->get_value(0, 0)+1+offset);
		const sample max = log(m_max_port->get_value(0, 0)+1+offset);
		scaled_value = exp(nval * (max - min) + min)-1-offset;
	} else {
		const sample min = m_min_port->get_value(0, 0);
		const sample max = m_max_port->get_value(0, 0);
		scaled_value = ((nval) * (max - min)) + min;
	}
	
	if (control_num == (int)(m_param_port->get_value(0, 0))) {
		m_control_port->set_value(start_sample, scaled_value);
		m_audio_port->set_value(start_sample, scaled_value);
	}
}


void
MidiControlNode::run(size_t nframes)
{
	NodeBase::run(nframes);
}


} // namespace Om

