/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef JACKDRIVER_H
#define JACKDRIVER_H

#include <jack/jack.h>
#include <jack/transport.h>
#include "Queue.h"
#include "List.h"
#include "SlowEventQueue.h"
#include "Maid.h"
#include "PostProcessor.h"

namespace Om {

class Event;
class SlowEvent;
class Patch;
class Port;
class InputNode;
class OutputNode;

typedef jack_default_audio_sample_t jack_sample_t;


/** Used internally by JackDriver to represent a Jack port.
 *
 * A Jack port always has a one-to-one association with a patch port.
 */
class JackPort
{
public:
	JackPort(jack_port_t* jack_port, jack_sample_t* jack_buffer, Patch* patch, Port* patch_port)
	: m_jack_port(jack_port),
	  m_jack_buffer(jack_buffer),
	  m_patch(patch),
	  m_patch_port(patch_port),
	  m_is_registered(true)
	{
	}
	
	jack_port_t* const   jack_port()                   { return m_jack_port; }
	jack_sample_t* const jack_buffer()                 { return m_jack_buffer; }
	void                 jack_buffer(jack_sample_t* s) { m_jack_buffer = s; }
	Patch* const         patch()                       { return m_patch; }
	Port* const          patch_port()                  { return m_patch_port; }
	bool                 is_registered()               { return m_is_registered; }
	void                 is_registered(bool b)         { m_is_registered = b; }

private:
	jack_port_t*   m_jack_port;
	jack_sample_t* m_jack_buffer;
	Patch*         m_patch;
	Port*          m_patch_port;
	bool           m_is_registered;
};



/** The Jack audio driver.
 *
 * The process callback here drives the entire audio thread by "pulling"
 * events from queues, processing them, then running the patches.
 *
 * \ingroup engine
 */
class JackDriver
{
public:	
	JackDriver();
	~JackDriver();

	void activate();
	void deactivate();
	void enable();
	void disable();

	void push_event(Event* const ev);
	void push_slow_event(SlowEvent* const ev);

	void process_events();
	
	ListNode<JackPort>* add_input(Patch* const p, InputNode* node);
	ListNode<JackPort>* add_output(Patch* const p, OutputNode* node);
	void remove_input(Patch* const p, InputNode* node);
	void remove_output(Patch* const p, OutputNode* node);
	void remove_port(JackPort& port);


	/** Transport state for this frame.  Intended to only be called from the
	 * audio thread. */
	const jack_position_t* position() { return &m_position; }
	
	bool is_realtime() { return jack_is_realtime(m_client); }
	
	jack_nframes_t buffer_size() const  { return m_buffer_size; }
	jack_nframes_t sample_rate() const  { return m_sample_rate; }
	bool           is_activated() const { return m_is_activated; }
	bool           is_enabled() const   { return m_is_enabled; }
	
	jack_nframes_t time_stamp() { return jack_frame_time(m_client); }

	SlowEventQueue* slow_event_queue() { return &m_slow_event_queue; }

private:

	// These are the static versions of the callbacks, they call
	// the non-static ones below
	inline static int  process_cb(jack_nframes_t nframes, void* const jack_driver);
	inline static void shutdown_cb(void* const jack_driver);
	inline static int  buffer_size_cb(jack_nframes_t nframes, void* const jack_driver);
	inline static int  sample_rate_cb(jack_nframes_t nframes, void* const jack_driver);

	// Non static callbacks
	int  m_process_cb(jack_nframes_t nframes);
	void m_shutdown_cb();
	int  m_buffer_size_cb(jack_nframes_t nframes);
	int  m_sample_rate_cb(jack_nframes_t nframes);

	jack_client_t* m_client;
	jack_nframes_t m_buffer_size;
	jack_nframes_t m_sample_rate;
	bool           m_is_activated;
	bool           m_is_enabled;
	
	jack_position_t m_position;
	
	List<JackPort> m_in_ports;
	List<JackPort> m_out_ports;

	// FIXME: these probably shouldn't be in this class
	Queue<Event*>  m_event_queue;
	SlowEventQueue m_slow_event_queue;
	PostProcessor  m_post_processor;

	// These are used to allow multiple threads to push events, ie the MIDI
	// and OSC threads.  They're not touched by the audio thread, promise. :)
	pthread_mutex_t m_event_queue_mutex;
	pthread_mutex_t m_slow_event_queue_mutex;

	jack_nframes_t m_start_of_current_cycle;
	jack_nframes_t m_start_of_last_cycle;
};


inline int JackDriver::process_cb(jack_nframes_t nframes, void* jack_driver)
{
	return ((JackDriver*)jack_driver)->m_process_cb(nframes);
}

inline void JackDriver::shutdown_cb(void* jack_driver)
{
	return ((JackDriver*)jack_driver)->m_shutdown_cb();
}


inline int JackDriver::buffer_size_cb(jack_nframes_t nframes, void* jack_driver)
{
	return ((JackDriver*)jack_driver)->m_buffer_size_cb(nframes);
}


inline int JackDriver::sample_rate_cb(jack_nframes_t nframes, void* jack_driver)
{
	return ((JackDriver*)jack_driver)->m_sample_rate_cb(nframes);
}
	

} // namespace Om

#endif // JACKDRIVER_H
