/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "DisconnectNodeEvent.h"
#include <iostream>
#include "Om.h"
#include "OmApp.h"
#include "Maid.h"
#include "List.h"
#include "Node.h"
#include "Connection.h"
#include "DisconnectionEvent.h"
#include "Port.h"
#include "Array.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "Patch.h"
#include "OSCSender.h"
#include "util.h"


using std::cerr; using std::endl;

namespace Om {


DisconnectNodeEvent::DisconnectNodeEvent(Request request, const string& node_path)
: SlowEvent(request),
  m_node_path(node_path),
  m_patch(NULL),
  m_node(NULL),
  m_succeeded(true),
  m_lookup(true)
{
}


/** Internal version, disconnects parent port as well (in the case of InputNode, etc).
 */
DisconnectNodeEvent::DisconnectNodeEvent(Node* node)
: SlowEvent(NIL_REQUEST),
  m_node_path(""),
  m_patch(node->parent()),
  m_node(node),
  m_succeeded(true),
  m_lookup(false)
{
}


DisconnectNodeEvent::~DisconnectNodeEvent()
{
	for (List<DisconnectionEvent*>::iterator i = m_disconnection_events.begin(); i != m_disconnection_events.end(); ++i)
		delete (*i);
}


void
DisconnectNodeEvent::prepare()
{
	// cerr << "Preparing disconnection event...\n";
	
	if (m_lookup) {
		m_patch = om->path_parser()->find_patch(
			om->path_parser()->parent(m_node_path));
	
		if (m_patch == NULL) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
		
		m_patch->node_remove_mutex().soft_lock();
		
		m_node = om->path_parser()->find_node(m_node_path);
		
		if (m_node == NULL) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
	}

	Connection* c = NULL;
	for (List<Connection*>::const_iterator i = m_patch->connections().begin(); i != m_patch->connections().end(); ++i) {
		c = (*i);
		if ((c->src_port()->node() == m_node || c->dst_port()->node() == m_node) && !c->pending_disconnection()) {
			DisconnectionEvent* ev = new DisconnectionEvent(c->src_port(), c->dst_port());
			ev->prepare();
			m_disconnection_events.push_back(new ListNode<DisconnectionEvent*>(ev));
			c->pending_disconnection(true);
		}
	}
	
	m_succeeded = true;
	SlowEvent::prepare();	
}


void
DisconnectNodeEvent::execute(uint sample_offset)
{
	if (m_succeeded) {
		for (List<DisconnectionEvent*>::iterator i = m_disconnection_events.begin(); i != m_disconnection_events.end(); ++i)
			(*i)->execute(sample_offset);
	}
	
	SlowEvent::execute(sample_offset);
}


void
DisconnectNodeEvent::post_process()
{
	if (m_lookup)
		m_patch->node_remove_mutex().soft_unlock();

	if (m_succeeded) {
		m_request.respond_ok();
		for (List<DisconnectionEvent*>::iterator i = m_disconnection_events.begin(); i != m_disconnection_events.end(); ++i)
			(*i)->post_process();
	} else {
		m_request.respond_error("Unable to disconnect all ports.");
	}
}


} // namespace Om

