/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef CROSSTHREADMUTEX_H
#define CROSSTHREADMUTEX_H

#include <pthread.h>
#include <cassert>
#include <iostream>

using std::cout; using std::cerr; using std::endl;


/** A mutex/semaphor that can be locked/unlocked from any thread.
 *
 * This is used by events to lock processing, ie RemoveNodeEvent which
 * modified the node tree in a patch in a non-threadsafe way.  So, this
 * can be locked by multiple objects (not necessarily threads!).  There is
 * a hard lock, which will block until it is the _only_ locker and hold that
 * lock, and a soft lock, which can coincide with other soft locks (like a kind
 * of backwards semaphore).
 *
 * The idea is that many things can work on a piece of data simultaneously
 * (they soft lock) but one operation can not be some simultaneously with
 * any of these other processes (this hard locks).  Hard lock is king.  All
 * hail hard lock.
 *
 * It's completely up to the users to not abuse this thing, there's no
 * checking done at all.
 *
 * This object is unfortunately named.  It made sense at one point.  Swear!
 */
class CrossThreadMutex
{
public:
	CrossThreadMutex()
	: m_is_hard_locked(false),
	  m_num_soft_locks(0)
	{
		pthread_mutex_init(&m_mutex, NULL);
		pthread_cond_init(&m_cond, NULL);
	}

	~CrossThreadMutex()
	{
		pthread_mutex_destroy(&m_mutex);
		pthread_cond_destroy(&m_cond);
	}

	/** If there is a hard lock, wait; then increment the soft lock count.
	 */
	void soft_lock()
	{
		pthread_mutex_lock(&m_mutex);

		if (m_is_hard_locked) {
			pthread_cond_wait(&m_cond, &m_mutex);
			assert( ! m_is_hard_locked);
		}
		++m_num_soft_locks;

		pthread_mutex_unlock(&m_mutex);
	}
	
	/** Decrement the soft lock count.
	 */
	void soft_unlock()
	{
		pthread_mutex_lock(&m_mutex);
		
		assert(m_num_soft_locks > 0);
		--m_num_soft_locks;

		pthread_mutex_unlock(&m_mutex);
	}

	/** Wait until all soft locks are released, then hard lock.
	 */
	void hard_lock()
	{
		pthread_mutex_lock(&m_mutex);
		
		while (m_num_soft_locks > 0 || m_is_hard_locked)
			pthread_cond_wait(&m_cond, &m_mutex);
		assert(m_num_soft_locks == 0);
		assert(m_is_hard_locked == false);
		m_is_hard_locked = true;
		
		pthread_mutex_unlock(&m_mutex);
	}

	/** Release a hard lock.
	 */
	void hard_unlock()
	{
		pthread_mutex_lock(&m_mutex);
		
		assert(m_is_hard_locked);
		m_is_hard_locked = false;
		pthread_cond_signal(&m_cond);
		
		pthread_mutex_unlock(&m_mutex);
	}

private:
	pthread_mutex_t m_mutex;
	pthread_cond_t  m_cond;
	bool            m_is_hard_locked;
	uint            m_num_soft_locks;
};


#endif // CROSSTHREADMUTEX_H
