/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "AlsaDriver.h"
#include <iostream>
#include <pthread.h>
#include "Om.h"
#include "OmApp.h"
#include "Maid.h"
#include "JackDriver.h"
#include "NoteOnEvent.h"
#include "NoteOffEvent.h"
#include "MidiControlEvent.h"
#include "MidiLearnEvent.h"
#include "Maid.h"
#include "Patch.h"

using std::cout; using std::cerr; using std::endl;

namespace Om {


//// AlsaPort ////

AlsaPort::AlsaPort(snd_seq_t* seq, Patch* patch)
: m_seq(seq),
  m_patch(patch),
  m_dssi_events(1024),
  m_dssi_events_array(new snd_seq_event_t[1024]),
  m_dssi_events_size(0),
  m_midi_learn_event(NULL)
{
	if ((m_port_id = snd_seq_create_simple_port(seq, patch->path().c_str(),
    	SND_SEQ_PORT_CAP_WRITE |
        SND_SEQ_PORT_CAP_SUBS_WRITE,
        SND_SEQ_PORT_TYPE_APPLICATION)) < 0)
	{
		std::cerr << "Error creating sequencer port.\n";
		throw;
	}

	//cerr << "[AlsaPort] Created port " << m_port_id << endl;
}


AlsaPort::~AlsaPort()
{
	snd_seq_delete_simple_port(m_seq, m_port_id);
}


void
AlsaPort::event(snd_seq_event_t* const ev)
{
	bool dssi_event = false;
	
	// Abuse the tick field to hold the timestamp, for DSSI
	ev->time.tick = om->jack_driver()->time_stamp();
	
	switch (ev->type) {
	case SND_SEQ_EVENT_NOTEON:
		if (ev->data.note.velocity == 0) {
			note_off(ev->data.note);
			ev->type = SND_SEQ_EVENT_NOTEOFF;  // fix for DSSI
		} else {
			note_on(ev->data.note);
		}
		dssi_event = true;
		break;
	case SND_SEQ_EVENT_NOTEOFF:
		note_off(ev->data.note);
		dssi_event = true;
		break;
	case SND_SEQ_EVENT_CONTROLLER:
		control(ev->data.control);
		dssi_event = true;
		break;
	// bank select, program change not yet supported
	}
	if (dssi_event) {
		m_dssi_events.push(*ev);
		dssi_event = false;
	}
}


/** Called from MIDI thread on a note-on event.
 */
void
AlsaPort::note_on(const snd_seq_ev_note_t& note) 
{
	int note_num = note.note;
	int velocity = note.velocity;

	NoteOnEvent* ev = new NoteOnEvent(NIL_REQUEST, m_patch, note_num, velocity);
	om->jack_driver()->push_event(ev);
}


/** Called from MIDI thread on a note-off event.
 */
void
AlsaPort::note_off(const snd_seq_ev_note_t& note) 
{
	int note_num = note.note;

	NoteOffEvent* ev = new NoteOffEvent(NIL_REQUEST, m_patch, note_num);
	om->jack_driver()->push_event(ev);
}


/** Called from MIDI thread on a control event.
 */
void
AlsaPort::control(const snd_seq_ev_ctrl_t& ctrl)
{
	//std::cerr << "[MIDI] Got controller, param = " << ctrl.param <<
	//	", value = " << ctrl.value << "... ";
	if (m_midi_learn_event != NULL) {
		m_midi_learn_event->set_control(ctrl);
		m_midi_learn_event->post_process(); // FIXME holding up the MIDI thread, ugh
		delete m_midi_learn_event;
		m_midi_learn_event = NULL;
	}

	MidiControlEvent* ev = new MidiControlEvent(NIL_REQUEST, m_patch, ctrl.param, ctrl.value);
	om->jack_driver()->push_event(ev);
}



/** Generates a flat array of MIDI events for DSSI plugins.
 */
void
AlsaPort::build_dssi_events_array(const samplecount block_start, const samplecount block_end)
{
	snd_seq_event_t ev;
	m_dssi_events_size = 0;
	
	while (!m_dssi_events.empty() && m_dssi_events.front().time.tick < block_end) {
		ev = m_dssi_events.pop();
		ev.time.tick -= block_start;
		m_dssi_events_array[m_dssi_events_size++] = ev;
		++m_dssi_events_size;
	}
}



//// AlsaDriver ////


bool AlsaDriver::m_midi_thread_exit_flag = true;


AlsaDriver::AlsaDriver()
: m_seq_handle(NULL),
  m_is_enabled(false)
{
	if (snd_seq_open(&m_seq_handle, "hw", SND_SEQ_OPEN_DUPLEX, 0) < 0) {
		std::cerr << "Error opening ALSA sequencer.";
		throw;
	} else {
		std::cout << "Successfully opened ALSA sequencer." << std::endl;
	}

	snd_seq_set_client_name(m_seq_handle, "Om");
}


/** Launch and start the MIDI thread.
 */
void
AlsaDriver::activate() 
{
	// Just exit if already running
	if (m_midi_thread_exit_flag == false)
		return;
	
	bool success = false;
	m_midi_thread_exit_flag = false;

	if (om->jack_driver()->is_realtime()) {
		pthread_attr_t attr;
		pthread_attr_init(&attr);

		if (pthread_attr_setschedpolicy(&attr, SCHED_FIFO)) {
			cerr << "[AlsaDriver] Unable to set realtime scheduling for MIDI thread." << endl;
		} else {
			cout << "[AlsaDriver] Started realtime MIDI thread." << endl;
		}

		// FIXME: priority?
		//sched_param param;
		//param.schedpriority =
		//pthread_attr_setschedparam(&attr, &param);
	
		if (!pthread_create(&m_process_thread, &attr, process_midi_in, this))
			success = true;
		else
			cerr << "[AlsaDriver] Unable to start realtime MIDI thread." << endl;
		pthread_attr_destroy(&attr);
	}
	
	if (!success) {
		pthread_create(&m_process_thread, NULL, process_midi_in, this);
		cerr << "[AlsaDriver] Started non-realtime MIDI thread." << endl;
	}

}


/** Terminate the MIDI thread.
 */
void
AlsaDriver::deactivate() 
{
	m_midi_thread_exit_flag = true;
}


/** Queue a MIDI learn event (which is port specific).
 */
void
AlsaDriver::set_midi_learn_event(MidiLearnEvent* ev)
{
	for (list<AlsaPort*>::iterator i = m_ports.begin(); i != m_ports.end(); ++i) {
		if ((*i)->patch() == ev->patch()) {
			(*i)->set_midi_learn_event(ev);
			break;
		}
	}
}


/** Build flat arrays of events for DSSI plugins for each AlsaPort.
 */
void
AlsaDriver::build_dssi_events_arrays(const samplecount block_start, const samplecount block_end)
{
	for (list<AlsaPort*>::iterator i = m_ports.begin(); i != m_ports.end(); ++i)
		(*i)->build_dssi_events_array(block_start, block_end);
}


/** Add an Alsa Midi port for the given patch.
 */
void
AlsaDriver::add_port(Patch* p)
{
	assert(p != NULL);
	AlsaPort* ap = new AlsaPort(m_seq_handle, p);
	p->alsa_port(ap);
	m_ports.push_back(ap);
}


/** Remove the Alsa Midi port for the given Patch.
 */
void
AlsaDriver::remove_port(Patch* p)
{
	assert(p != NULL);
	for (list<AlsaPort*>::iterator i = m_ports.begin(); i != m_ports.end(); ++i) {
		if ((*i)->patch() == p) {
			m_ports.erase(i);
			delete (*i);
			return;
		}
	}
	//cerr << "[AlsaDriver] WARNING: Unable to find patch, not removing an Alsa Midi port." << endl;
}		


/** MIDI thread.
 */
void*
AlsaDriver::process_midi_in(void* alsa_driver) 
{
	AlsaDriver* ad = (AlsaDriver*)alsa_driver;

	snd_seq_event_t* ev;

	int npfd = snd_seq_poll_descriptors_count(ad->m_seq_handle, POLLIN);
	struct pollfd pfd;
	snd_seq_poll_descriptors(ad->m_seq_handle, &pfd, npfd, POLLIN);
	
	while ( ! m_midi_thread_exit_flag)
		if (poll(&pfd, npfd, 100000) > 0)
			while (snd_seq_event_input(ad->m_seq_handle, &ev) > 0 && ad->m_is_enabled)
				for (list<AlsaPort*>::iterator i = ad->m_ports.begin(); i != ad->m_ports.end(); ++i)
					if ((*i)->port_id() == (int)ev->dest.port)
						(*i)->event(ev);
	
	pthread_exit(NULL);
	return NULL;
}


} // namespace Om

