/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "Comm.h"
#include "PatchLibrarian.h"
#include "PatchModel.h"
#include <iostream>
#include <unistd.h>
#include "cmdline.h"  // generated by gengetopt

using std::cout;
using std::endl;

using namespace LibOmClient;

int main(int argc, char** argv)
{
	bool using_default = false;
	
	const char* engine_host = NULL;
	int         engine_port = 0;
	const char* engine_url  = NULL;
	const char* client_host = NULL;
	int         client_port = 0;
	
	Comm           comm;
	PatchLibrarian librarian(&comm);
	
	/* Parse command line options */
	gengetopt_args_info args_info;
	if (cmdline_parser (argc, argv, &args_info) != 0)
		return 1;

	if ( /*! args_info.launch_engine_given &&*/ ! (args_info.port_given || args_info.url_given)) {
		//cout << "You must either use -l to launch an engine, or specify which engine to connect to." << endl;
		cout << "No engine port or URL specified.  Attempting to use localhost:16180." << endl;
		using_default = true;
	}	

	// Attempt to connect if the user specified a server
	if (args_info.client_host_given) {
		client_host = args_info.client_host_arg;
	}
	if (args_info.client_port_given) {
		client_port = args_info.client_port_arg;
	}
	
	if (using_default) {
		engine_host = "localhost";
		engine_port = 16180;
		comm.attach(engine_host, engine_port, client_port);
	} else if (args_info.port_given) {
		if (args_info.host_given)
			engine_host = args_info.host_arg;
		else
			engine_host = "localhost";
		engine_port = args_info.port_arg;
		cout << "Attempting to contact Om engine at " << engine_host
			<< ":" << engine_port << "..." << endl;
		comm.attach(engine_host, engine_port, client_port);
	} else if (args_info.url_given) {
		engine_url = args_info.url_arg;
		cout << "Attempting to contact Om engine at " << engine_url << "..." << endl;
		comm.attach_url(engine_url, client_port);
	} else {
		cerr << "Invalid command line arguments, aborting." << endl;
		exit(1);
	}

	comm.activate();
	comm.enable();
	
	if (args_info.client_host_given) {
		if (client_port != 0)
			cout << "(Registering as " << client_host << ":" << client_port << ")" << endl;
		else
			cout << "(Registering as " << client_host << ")" << endl;
		comm.register_client(client_host, client_port);
	} else {
		comm.register_client();
	}
	
	int id = comm.get_next_request_id();
	comm.set_wait_response_id(id);
	comm.load_plugins(id);
	comm.wait_for_response();

	// Load patches
	for (uint i=0; i < args_info.inputs_num; ++i) {
		PatchModel* pm = new PatchModel();
		pm->path("");
		pm->parent(NULL);
		pm->poly(1);
		pm->filename(args_info.inputs[i]);
		librarian.load_patch(pm);
		delete pm;
	}
	
	return 0;
}
