/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "util.h"
#include "Node.h"
#include "OutputNode.h"
#include "Patch.h"


namespace Om {


/** Find the process order for a patch.
 *
 * The process order is a flat list that the patch will execute in order
 * when it's run() method is called.
 *
 * This function is not realtime safe, due to the use of the NodeTree iterator
 */
List<Node*>* const
find_process_order(const Patch* const patch)
{
	List<Node*>* process_order = new List<Node*>();
	const NodeTree& nodes = patch->nodes();
	
	for (NodeTree::iterator i = nodes.begin(); i != nodes.end(); ++i)
		(*i)->traversed(false);
			
	for (List<OutputNode*>::const_iterator i = patch->output_nodes().begin(); i != patch->output_nodes().end(); ++i)
		// If the output node has been disconnected and has no connections left, don't traverse
		// into it so it's not in the process order (and can be removed w/o flaming segfault death)
		if ( (*i)->providers()->size() > 0)
			find_process_order_recursive((*i), process_order);

	/*
	cerr << endl << "***** PROCESS ORDER: " << patch->path() << " *****" << endl;
	for (List<Node*>::iterator i = process_order->begin(); i != process_order->end(); ++i)
		if ((*i) != NULL)
			cerr << (*i)->name() << endl;
	cerr << "*************************" << endl;
	*/

	return process_order;
}


inline void
find_process_order_recursive(Node* const n, List<Node*>* const order)
{
	if (n == NULL || n->traversed()) return;
	n->traversed(true);
	assert(order != NULL);
	
	for (List<Node*>::iterator i = n->providers()->begin(); i != n->providers()->end(); ++i)
		if ( ! (*i)->traversed() )
			find_process_order_recursive((*i), order);
	
	order->push_back(new ListNode<Node*>(n));
	//n->traversed(true);
}


} // namespace Om
