/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SLOWEVENT_H
#define SLOWEVENT_H

#include "Event.h"

namespace Om {


/** A non-realtime event.
 *
 * These events are events that aren't able to be executed immediately by the
 * Jack thread (because they allocate memory or whatever).  They are pushed
 * on to the "slow event queue" where they are preprocessed then pushed on
 * to the realtime Jack event queue when they are ready.
 *
 * Lookups for these events should go in the prepare() method, since they are
 * not time critical and shouldn't waste time in the audio thread doing
 * lookups they can do beforehand.
 *
 * \ingroup engine
 */
class SlowEvent : public Event
{
public:
	/** Process this event into a realtime-suitable event.
	 */
	virtual void prepare() {
		//cout << "Preparing event..." << endl;
		assert(m_prepared == false);
		m_prepared = true;
	}
	
	virtual void execute(uint sample_offset) {
		//cout << "Executing event..." << endl;
		Event::execute(sample_offset);
		assert(m_prepared);
	}

	virtual void post_process() {}

	/** If this event blocks the prepare phase of other slow events */
	bool is_blocking() { return m_blocking; }

	bool is_prepared() { return m_prepared; }
	
protected:
	bool m_prepared;
	bool m_blocking;
	
	SlowEvent(Request request, bool blocking = false) : Event(request), m_prepared(false), m_blocking(blocking) {}
};


} // namespace Om

#endif // SLOWEVENT_H
