/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef PLUGINFACTORY_H
#define PLUGINFACTORY_H

#include <list>
#include <string>
#include <ladspa.h>
#include <pthread.h>
#include "PluginInfo.h"

using std::string; using std::list;

namespace Om {

class Node;
class PortInfo;
class Patch;


/** Loads plugins and creates Nodes from them.
 *
 * PluginFactory's responsibility is to get enough information to allow the 
 * loading of a plugin possible (ie finding/opening shared libraries etc)
 *
 * The constructor of various Node types (ie LADSPAPlugin) are responsible
 * for actually creating a Node instance of the plugin.
 *
 * \ingroup engine
 */
class PluginFactory
{
public:
	PluginFactory();
	~PluginFactory();

	void  load_plugins();
	Node* load_plugin(const PluginInfo* info, const string& path, uint poly, Patch* parent);
	
	const list<PluginInfo>& plugins() { return m_plugins; }
	
	void lock_plugin_list()   { pthread_mutex_lock(&m_plugin_list_mutex); }
	void unlock_plugin_list() { pthread_mutex_unlock(&m_plugin_list_mutex); }
	
private:
	void load_ladspa_plugins();
	void load_dssi_plugins();

	Node* load_ladspa_plugin(const string& lib_name, const string& plug_name, const string& path, uint poly, Patch* parent);
	Node* load_dssi_plugin(const string& lib_name, const string& plug_name, const string& path, uint poly, Patch* parent);
	Node* load_internal_plugin(const string& plug_label, const string& path, uint poly, Patch* parent);
	
	list<PluginInfo> m_internal_plugins;
	list<PluginInfo> m_plugins;

	/** Used to protect the list while load_plugins is building it. */
	pthread_mutex_t m_plugin_list_mutex;

	bool m_has_loaded;
};


} // namespace Om

#endif // PLUGINFACTORY_H
