/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OmApp.h"	
#include <sys/mman.h>
#include <iostream>
#include "JackDriver.h"
#include "AlsaDriver.h"
#include "PluginFactory.h"
#include "OSCReceiver.h"
#include "OSCSender.h"
#include "Patch.h"
#include "PathParser.h"
#include "Maid.h"
using std::cout; using std::cerr; using std::endl;

namespace Om {


OmApp::OmApp(const char* const port)
: m_jack_driver(new JackDriver()),
  m_alsa_driver(new AlsaDriver()),
  m_osc_receiver(new OSCReceiver(port)),
  m_osc_sender(new OSCSender(m_osc_receiver)),
  m_path_parser(new PathParser()),
  m_plugin_factory(new PluginFactory()),
  m_maid(new Maid(2048)), // FIXME: figure out appropriate size
  m_quit_flag(false)
{
	if (mlockall(MCL_CURRENT | MCL_FUTURE))
		cerr << "Warning:  Failed to lock memory.  Timing could suffer." << endl;
	else
		cout << "Successfully locked all memory." << endl;

	m_maid->start();
}


OmApp::~OmApp()
{
	munlockall();

	for (List<Patch*>::iterator i = m_patches.begin(); i != m_patches.end(); ++i)
		delete (*i);
}


int
OmApp::main()
{
	// Loop until quit flag is set (by OSCReceiver)
	while ( ! m_quit_flag) {
		sleep(1);
	}
	cout << "Done main loop." << endl;
	disable();
	deactivate();
	cout << "Om exiting." << endl;
	
	return 0;
}


void
OmApp::activate()
{
	m_zero_buffer = (sample*)calloc(m_jack_driver->buffer_size(), sizeof(sample));
	m_jack_driver->activate();
	m_alsa_driver->activate();
}


void
OmApp::deactivate()
{
	free(m_zero_buffer);
	m_zero_buffer = NULL;
	m_alsa_driver->deactivate();
	m_jack_driver->deactivate();
}


void
OmApp::enable()
{
	m_jack_driver->enable();
	m_alsa_driver->enable();
}


void
OmApp::disable()
{
	m_alsa_driver->disable();
	m_jack_driver->disable();
}


void
OmApp::remove_patch(Patch* patch)
{
	List<Patch*>::iterator i = m_patches.begin();
	for ( ; i != m_patches.end(); ++i)
		if (*i == patch) break;
	
	if (i != m_patches.end())
		m_patches.remove(i);
}


} // namespace Om
