/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef NODEBASE_H
#define NODEBASE_H

#include <string>
#include "Node.h"
using std::string;

namespace Om {

class PluginInfo;
class OutputPort;


/** Common implementation stuff for Node.
 *
 * Pretty much just attributes and getters/setters are here.
 *
 * \ingroup engine
 */
class NodeBase : public Node
{
public:
	NodeBase(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size);

	virtual ~NodeBase();

	virtual void activate();
	virtual void deactivate();
	
	virtual void run(size_t nframes);
		
	virtual void set_port_buffer(uint voice, uint port_num, sample* buf);
	
	virtual Port* const port(const string& port_name) const;

	virtual void add_to_patch(Patch* p) {}
	virtual void remove_from_patch(Patch* p) {}

	virtual void send_creation_messages(lo_address addr) const;
	virtual void send_deletion_messages(lo_address addr) const;
	
	uint          num_ports() const { return m_num_ports; }
	uint          poly() const      { return m_poly; }
	bool          traversed() const { return m_traversed; }
	void          traversed(bool b) { m_traversed = b; }
	
	const Array<Port*>& ports() const { return m_ports; }

	virtual List<Node*>* const providers()               { return m_providers; }
	virtual void               providers(List<Node*>* l) { m_providers = l; }
	
	virtual List<Node*>* const dependants()               { return m_dependants; }
	virtual void               dependants(List<Node*>* l) { m_dependants = l; }
	
	Patch* const parent() const   { return m_parent; }
	void         parent(Patch* p) { m_parent = p; }

	const PluginInfo& plugin_info() const               { throw; }
	void              plugin_info(const PluginInfo& pi) { throw; }
	
protected:	
	NodeBase(const NodeBase& copy) : Node("") { throw; }  // disallow copies
	
	uint        m_poly;
	Patch*      m_parent;

	samplerate  m_srate;
	size_t      m_buffer_size;
	bool        m_activated;

	uint         m_num_ports; // number of ports PER VOICE
	Array<Port*> m_ports;

	bool         m_traversed;
	List<Node*>* m_providers;     // Nodes connected to this one's input ports
	List<Node*>* m_dependants;    // Nodes this one's output ports are connected to
};


} // namespace Om

#endif // NODEBASE_H
