/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "ConnectionEvent.h"
#include <string>
#include "Om.h"
#include "OmApp.h"
#include "Connection.h"
#include "InputPort.h"
#include "Patch.h"
#include "JackDriver.h"
#include "OSCSender.h"
#include "Port.h"
#include "util.h"

using std::string;
namespace Om {


ConnectionEvent::ConnectionEvent(Request request, const string& src_port_path, const string& dst_port_path)
: SlowEvent(request),
  m_src_port_path(src_port_path),
  m_dst_port_path(dst_port_path),
  m_patch(NULL),
  m_src_node(NULL),
  m_dst_node(NULL),
  m_src_port(NULL),
  m_dst_port(NULL),
  m_connection(NULL),
  m_process_order(NULL),
  m_succeeded(true)
{
}


void
ConnectionEvent::prepare()
{
	//std::cerr << "Preparing connection event...\n";
	
	m_patch = om->path_parser()->find_patch(
		om->path_parser()->parent(om->path_parser()->parent(m_src_port_path)));

	if (m_patch == NULL) {
		m_succeeded = false;
		SlowEvent::prepare();
		return;
	}
	
	m_patch->node_remove_mutex().soft_lock();
	
	Port* port1 = om->path_parser()->find_port(m_src_port_path);
	Port* port2 = om->path_parser()->find_port(m_dst_port_path);
	
	if (port1 == NULL || port2 == NULL) {
		m_succeeded = false;
		SlowEvent::prepare();
		return;
	}

	if (port1->is_audio() != port2->is_audio()) {
		m_succeeded = false;
		SlowEvent::prepare();
		return;
	}
	
	if (port1->is_output() && port2->is_input()) {
		m_src_port = (OutputPort*)port1;
		m_dst_port = (InputPort*)port2;
	} else if (port2->is_output() && port1->is_input()) {
		m_src_port = (OutputPort*)port2;
		m_dst_port = (InputPort*)port1;
	} else {
		m_succeeded = false;
		SlowEvent::prepare();
		return;
	}

	m_src_node = m_src_port->node();
	m_dst_node = m_dst_port->node();
	
	if (m_src_node == NULL || m_dst_node == NULL) {
		m_succeeded = false;
		SlowEvent::prepare();
		return;
	}
	
	if (m_src_node->parent() != m_patch || m_dst_node->parent() != m_patch) {
		m_succeeded = false;
		SlowEvent::prepare();
		return;
	}

	m_connection = new Connection(m_src_port, m_dst_port);
	m_port_listnode = new ListNode<Connection*>(m_connection);
	m_patch_listnode = new ListNode<Connection*>(m_connection);
	
	m_dst_node->providers()->push_back(new ListNode<Node*>(m_src_node));
	m_src_node->dependants()->push_back(new ListNode<Node*>(m_dst_node));

	if (m_patch->process())
		m_process_order = Om::find_process_order(m_patch);
	
	m_succeeded = true;
	SlowEvent::prepare();
}


void
ConnectionEvent::execute(uint sample_offset)
{
	SlowEvent::execute(sample_offset);

	if (m_succeeded) {
		// These must be inserted here, since they're actually used by the audio thread
		m_dst_port->add_connection(m_port_listnode);
		m_patch->add_connection(m_patch_listnode);
		if (m_patch->process_order() != NULL)
			om->maid()->push(m_patch->process_order());
		m_patch->process_order(m_process_order);
	}
}


void
ConnectionEvent::post_process()
{
	if (m_patch != NULL) {
		m_patch->node_remove_mutex().soft_unlock();
	}
	
	/*if (patch == NULL) {
		m_request.respond_error("Could not find patch in /patch/connect");
		return 0;
	}
	Node* src_node = patch->nodes().find(&argv[1]->s);
	if (src_node == NULL) {
		m_request.respond_error("Could not find source node in /patch/connect");
		return 0;
	}
	Node* dst_node = patch->nodes().find(&argv[3]->s);
	if (dst_node == NULL) {
		m_request.respond_error("Could not find destination node in /patch/connect");
		return 0;
	}*/
	if (m_succeeded) {
		m_request.respond_ok();
		/*om->osc_sender()->send_new_connection(m_patch_name,
			m_src_node_name, m_src_port_name, m_dst_node_name, m_dst_port_name);*/
		om->osc_sender()->send_connection(m_connection);
	} else {
		string msg = "Unable to make connection ";
		msg.append(m_src_port_path + " -> " + m_dst_port_path);
		m_request.respond_error(msg);
	}
}


} // namespace Om

