/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "AudioOutputNode.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "PluginInfo.h"
#include "Patch.h"
#include "Om.h"
#include "OSCSender.h"
#include "Connection.h"
#include "JackDriver.h"
#include <cassert>

namespace Om {


AudioOutputNode::AudioOutputNode(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size)
: OutputNode(path, poly, parent, srate, buffer_size),
  m_jack_port(NULL)
{
	m_external_port = new OutputPort(parent, path.substr(0, path.find_last_of("/")) + "/" + m_name,
		0, m_poly, new PortInfo(m_name, AUDIO, OUTPUT), m_buffer_size);
	InputPort* iport = new InputPort(this, path + "/out", 0, m_poly, new PortInfo("out", AUDIO, INPUT), m_buffer_size);
	iport->join(m_external_port);
	
	m_num_ports = 1;
	m_ports.alloc(m_num_ports);
	m_ports.at(0) = iport;

	m_plugin_info.type(PluginInfo::Internal);
	m_plugin_info.lib_path("");
	m_plugin_info.lib_name("");
	m_plugin_info.plug_label("audio_output");
	m_plugin_info.name("Om patch audio output node");

	m_listnode = new ListNode<OutputNode*>(this);
}


AudioOutputNode::~AudioOutputNode()
{
	deactivate();
	delete m_external_port;
}


void
AudioOutputNode::activate()
{
	assert(m_parent != NULL);
	
	InternalNode::activate();
	if (m_parent->parent() == NULL && om != NULL)
		m_jack_port = om->jack_driver()->add_output(m_parent, this);
}


void
AudioOutputNode::deactivate()
{
	InternalNode::deactivate();

	if (m_parent->parent() == NULL && om != NULL) {
		om->jack_driver()->remove_output(m_parent, this);
		//om->osc_sender()->send_port_removal(m_parent, m_external_port->port_info());
	}
}


void
AudioOutputNode::add_to_patch(Patch* patch)
{
	assert(m_parent != NULL);
	assert(m_parent == patch);
	
	patch->output_nodes().push_back(m_listnode);
	InternalNode::add_to_patch(patch);
	
	// Activate jack port now in the audio thread (not before when created, to avoid race issues)
	if (m_parent->parent() == NULL) {
		assert(m_jack_port != NULL);
		m_jack_port->elem().is_registered(true);
	}
}


void
AudioOutputNode::remove_from_patch(Patch* patch)
{
	patch->output_nodes().remove(this);
}


void
AudioOutputNode::send_creation_messages(lo_address addr) const
{
	om->osc_sender()->send_new_port_to(addr, m_external_port);
	InternalNode::send_creation_messages(addr);
}


void
AudioOutputNode::send_deletion_messages(lo_address addr) const
{
	InternalNode::send_deletion_messages(addr);
	om->osc_sender()->send_port_removal_to(addr, m_external_port);
}


} // namespace Om

