/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef QUEUE_H
#define QUEUE_H

#include <cstdlib>
#include <iostream>
#include <sys/mman.h>
#include "types.h"

class QueueUnderflowException {};
class QueueOverflowException {};


/** Threadsafe queue (actually a deque)
 *
 * Implemented as a deque (lock free ringbuffer) in an array.  Pushing and
 * popping are threadsafe, as long as a single thread pushes and a single
 * thread pops (ie there can be at most one read thread and at most one
 * write thread)
 */
template <class T>
class Queue
{
public:
	Queue(uint size);
	~Queue();

	void push(T obj);
	T    pop();
	T    front() { return m_objects[m_front]; }
	bool empty() { return (m_back == m_front); }

private:
	uint       m_back;
	uint       m_front;
	uint       m_end_of_queue;
	const uint m_size; // size of actual array, NOT number of elements in queue
	T*         m_objects;
};


template<class T>
Queue<T>::Queue(uint size)
: m_back(0),
  m_front(0),
  m_end_of_queue(size),
  m_size(size+1)
{
	m_objects = (T*)calloc(m_size, sizeof(T));
}


template <class T>
Queue<T>::~Queue()
{
	free(m_objects);
}


/** Push an item on to the Queue - realtime-safe, not thread-safe.
 *
 * Note this function may not be used concurrently by multiple threads.
 */
template <class T>
void
Queue<T>::push(T elem)
{
	if (m_back == m_end_of_queue)
		throw QueueOverflowException();
	
	m_objects[m_back] = elem;
	
	if (m_back < (m_size-1))
		++m_back;
	else
		m_back = 0;
}


/** Pop an item off the front of the queue - realtime-safe.
 *
 * It is an error to call pop() when the queue is empty.
 */
template <class T>
T
Queue<T>::pop()
{
	if (m_back == m_front)
		throw QueueUnderflowException();

	T r = m_objects[m_front];
	m_end_of_queue = m_front;
	//m_objects[m_front] = NULL;
	
	if (m_front < (m_size-1))
		++m_front;
	else
		m_front = 0;
	
	return r;
}


#endif // QUEUE_H
