/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License alongCont
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "NodeControlWindow.h"
#include "OmGtk.h"
#include "PatchController.h"
#include "Controller.h"
#include "NodeModel.h"
#include "PortModel.h"
#include "ControlGroups.h"
#include <iostream>
#include <cmath>
using std::cerr; using std::cout; using std::endl;

namespace OmGtk {


NodeControlWindow::NodeControlWindow(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& xml)
: Gtk::Window(cobject),
  m_patch_controller(NULL),
  m_node_model(NULL),
  m_callback_enabled(false)
{
	xml->get_widget("node_control_controls_box", m_control_box);
	xml->get_widget("node_control_controls_scrolled_win", m_scrolled_window);
	xml->get_widget("node_control_all_voices_radio", m_all_voices_radio);
	xml->get_widget("node_control_specific_voice_radio", m_specific_voice_radio);
	xml->get_widget("node_control_voice_spinbutton", m_voice_spinbutton);
	
	m_all_voices_radio->signal_toggled().connect(sigc::mem_fun(this, &NodeControlWindow::all_voices_selected));
	m_specific_voice_radio->signal_toggled().connect(sigc::mem_fun(this, &NodeControlWindow::specific_voice_selected));
	m_voice_spinbutton->signal_value_changed().connect(sigc::mem_fun(this, &NodeControlWindow::voice_selected));
}


NodeControlWindow::~NodeControlWindow()
{
	for (vector<ControlGroup*>::iterator i = m_controls.begin(); i != m_controls.end(); ++i)
		delete (*i);
}


/** Initialize the window.  MUST be called before using in any way!
 */
void
NodeControlWindow::init(PatchController* patch_controller, NodeModel* node_model)
{
	m_patch_controller = patch_controller;
	m_node_model = node_model;
	
	if (node_model->polyphonic())
		m_voice_spinbutton->set_range(0, patch_controller->model()->poly()-1);
	else
		m_voice_spinbutton->set_range(0, 0);

	m_controls.reserve(node_model->num_ports());

	for (PortModelList::const_iterator i = node_model->port_models().begin(); i != node_model->port_models().end(); ++i)
		add_port(*i, false);
	
	resize();
	
	property_resizable() = true;

	Glib::ustring title = "Om - ";
	title += m_node_model->name();
	property_title() = title;

	m_callback_enabled = true;

	show_all();
}


void
NodeControlWindow::add_port(PortModel* pm, bool resize_to_fit)
{
	if (pm->is_control() && pm->is_input()) {
		ControlGroup* cg = NULL;
		if (pm->is_integer())
			cg = new IntegerControlGroup(pm);
		else if (pm->is_toggle())
			cg = new ToggleControlGroup(pm);
		else
			cg = new SliderControlGroup(pm);
		cg->signal_value_changed().connect(
			sigc::bind<const string&>(
				sigc::mem_fun(*this, &NodeControlWindow::value_changed),
				pm->path()));
		m_controls.push_back(cg);
		m_control_box->pack_start(*cg, false, false, 5);

		if (resize_to_fit) resize();
	}
}


void
NodeControlWindow::resize()
{
	Gtk::Requisition box_size;
	m_control_box->size_request(box_size);
	int height = box_size.height + 40;
	int width = 440;

	if (height > property_screen().get_value()->get_height() - 64)
		height = property_screen().get_value()->get_height() - 64;
	if (width > property_screen().get_value()->get_width() - 64)
		width = property_screen().get_value()->get_width() - 64;

	Gtk::Window::resize(width, height);
}


void
NodeControlWindow::value_changed(float val, const string& port_path)
{
	if (m_callback_enabled) {
		if (m_all_voices_radio->get_active()) {
			controller->set_control(port_path, val);
		} else {
			int voice = m_voice_spinbutton->get_value_as_int();
			controller->set_control(port_path, voice, val);
		}
	}
}


void
NodeControlWindow::set_control(const string& port_path, float val)
{
	m_callback_enabled = false;
	bool found = false;
	for (vector<ControlGroup*>::iterator i = m_controls.begin(); i != m_controls.end(); ++i) {
		if ((*i)->port_model()->path() == port_path) {
			found = true;
			(*i)->set_value(val);
		}
	}
	if (found == false)
		cerr << "[NodeControlWindow::set_control] Unable to find control " << port_path << endl;
	m_callback_enabled = true;
}


void
NodeControlWindow::set_range_min(const string& port_path, float val)
{
	bool found = false;
	for (vector<ControlGroup*>::iterator i = m_controls.begin(); i != m_controls.end(); ++i) {
		if ((*i)->port_model()->path() == port_path) {
			found = true;
			(*i)->set_min(val);
		}
	}
	if (found == false)
		cerr << "[NodeControlWindow::set_range_min] Unable to find control " << port_path << endl;
}


void
NodeControlWindow::set_range_max(const string& port_path, float val)
{
	bool found = false;
	for (vector<ControlGroup*>::iterator i = m_controls.begin(); i != m_controls.end(); ++i) {
		if ((*i)->port_model()->path() == port_path) {
			found = true;
			(*i)->set_max(val);
		}
	}
	if (found == false)
		cerr << "[NodeControlWindow::set_range_max] Unable to find control " << port_path << endl;
}


void
NodeControlWindow::all_voices_selected()
{
	m_voice_spinbutton->property_sensitive() = false;
}


void
NodeControlWindow::specific_voice_selected()
{
	m_voice_spinbutton->property_sensitive() = true;
}


void
NodeControlWindow::voice_selected()
{
}

	
} // namespace OmGtk
