/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* These are the events that are pushed into a queue by the OSC thread and
 * read by the Gtk thread to do things in the gtk client in response to
 * engine events.
 */

#ifndef GTKCLIENTHOOKSEVENTS_H
#define GTKCLIENTHOOKSEVENTS_H

#include "GtkClientHooks.h"
#include <iostream>
#include <string>
#include <cassert>
#include "OmGtk.h"
#include "Controller.h"
#include "OmGtkApp.h"
#include "PatchWindow.h"
#include "PatchController.h"
#include "PatchModel.h"
#include "NodeModel.h"
#include "ConnectionModel.h"
#include "MetadataModel.h"
#include "ControlModel.h"
#include "PluginInfo.h"
#include "ClientPathParser.h"
#include "NodeControlWindow.h"

using std::cout; using std::cerr; using std::endl;
using std::string;

namespace OmGtk {


/**	Pure virtual base class for all gtk client events.
 */
class Event
{
public:
	virtual void execute() = 0;
};



class ErrorEvent : public Event
{
public:
	ErrorEvent(const string& msg) : m_msg(msg) {}
	virtual ~ErrorEvent() {}

	void execute()
	{
		app->error_message(m_msg);
		app->show_engine_error_dialog();
	}
	
private:
	string m_msg;
};


class EngineEnabledEvent : public Event
{
public:
	EngineEnabledEvent() {}
	virtual ~EngineEnabledEvent() {}
	
	void execute()
	{
		app->engine_enabled(true);
	}
};


class EngineDisabledEvent : public Event
{
public:
	EngineDisabledEvent() {}
	virtual ~EngineDisabledEvent() {}
	
	void execute()
	{
		app->engine_enabled(false);
	}
};


class NewPatchEvent : public Event
{
public:
	NewPatchEvent(PatchModel* const pm) : m_patch_model(pm) {}
	virtual ~NewPatchEvent() {}
	
	void execute()
	{
		//cout << "[GtkClientHooks] New patch." << endl;
		if (app->patch_window(m_patch_model->path()) != NULL) {
			delete m_patch_model;
		} else {
			bool subpatch = false;
			
			PatchWindow* parent_pw = NULL;
			string parent_path = ClientPathParser::parent(m_patch_model->path());
			if (parent_path != "") {
				subpatch = true;
				parent_pw = app->patch_window(parent_path);
				if (parent_pw != NULL)
					m_patch_model->parent(parent_pw->patch_controller()->model());
			}
			
			bool show = false;
			
			// See if we cached this patch model to store it's location (to avoid the
			// module "jumping") and filename (which isn't sent to engine)
			PatchModel* pm = controller->yank_added_patch(m_patch_model->path());
			if (pm != NULL) {
				if (subpatch) {
					m_patch_model->x(pm->x());
					m_patch_model->y(pm->y());
				}
				m_patch_model->filename(pm->filename());
				// Show patch windows that are explicitly added
				show = true;
			} else { // pick a default
				if (subpatch) {
					if (parent_pw == NULL) {
						cerr << "[NewPatchEvent] WARNING: Unable to find parent's patch window ("
							<< parent_path << ")" << endl;
					} else {
						int x, y;
						parent_pw->patch_controller()->get_new_module_location(x, y);
						m_patch_model->x(x);
						m_patch_model->y(y);
					}
				}
			}
			
			//cerr << "[NewPatchEvent] Creating patch " << m_patch_model->path() << endl;
			
			if (m_patch_model->parent() == NULL)
				show = true;
			PatchWindow* pw = app->add_patch_window(m_patch_model, show);
		
			if (subpatch) {
				if (parent_pw != NULL) {
				m_patch_model->parent(parent_pw->patch_controller()->model());
				parent_pw->patch_controller()->new_subpatch(pw);
				} else {
					cerr << "[NewPatchEvent] Did not find window for parent of " << m_patch_model->path()
						<< ". Subpatch will not appear." << endl;
				}
			}
		}
	}
	
private:
	PatchModel* m_patch_model;
};



class NewPortEvent : public Event
{
public:
	NewPortEvent(PortModel* const pm)
		: m_port_model(pm) {}
	virtual ~NewPortEvent() {}
	
	void execute()
	{
		const string node_path = ClientPathParser::parent(m_port_model->path());

		string patch_path = node_path.substr(0, node_path.find_last_of("/"));
	
		PatchWindow* pw = NULL;
		
		if (patch_path == "") { // port on a top level patch
			patch_path = node_path;
			pw = app->patch_window(patch_path);
			if (pw != NULL) {
				pw->new_port(m_port_model);
				return;
			}
		}

		/*cerr << "[GtkClientHooks] New port, name = " << m_node_path <<
			", (parsed) patch path = " << patch_path <<
			", node name = " << node_name << endl;
		*/
	
		pw = app->patch_window(patch_path);
		if (pw != NULL) {
			pw->patch_controller()->new_port(m_port_model);
		} else {
			cerr << "[NewPortEvent] Could not find patch window \'" << patch_path
				<< "\' (port = \'" << m_port_model->path() << "\')" << endl;
		}
	}

private:
	PortModel* m_port_model;
};


class PortRemovalEvent : public Event
{
public:
	PortRemovalEvent(const string& path)
	: m_path(path) {}
	virtual ~PortRemovalEvent() {}

	void execute()
	{
		const string node_path = m_path.substr(0, m_path.find_last_of("/"));
		const string patch_path = node_path.substr(0, node_path.find_last_of("/"));
	
		PatchWindow* pw = app->patch_window(patch_path);
		if (pw != NULL) {
			pw->patch_controller()->remove_port(m_path);
		} else {
			cerr << "[PortRemovalEvent] Could not find patch window \'" << patch_path
				<< "\' (port = \'" << m_path << "\')" << endl;
		}
	}
private:
	string m_path;
};
		

class PatchDestructionEvent : public Event
{
public:
	PatchDestructionEvent(const string& path) : m_path(path) {}
	virtual ~PatchDestructionEvent() {}
	
	void execute()
	{
		//cerr << "[GtkClientHooks] Patch destruction" << endl;
	
		app->destroy_patch_window(m_path);
	}

private:
	string m_path;
};
	

class PatchEnabledEvent : public Event
{
public:
	PatchEnabledEvent(const string& patch_path) : m_patch_path(patch_path) {}
	virtual ~PatchEnabledEvent() {}
	
	void execute()
	{
		PatchWindow* pw = app->patch_window(m_patch_path);
		
		if (pw != NULL) {
			pw->enabled(true);
		} else {
			cerr << "[PatchEnabledEvent] Can not find window for patch " << m_patch_path
				<< "." << endl;
		}
	}
	
private:
	string m_patch_path;
};


class PatchDisabledEvent : public Event
{
public:
	PatchDisabledEvent(const string& patch_path) : m_patch_path(patch_path) {}
	virtual ~PatchDisabledEvent() {}
	
	void execute()
	{
		PatchWindow* pw = app->patch_window(m_patch_path);
		
		if (pw != NULL) {
			pw->enabled(false);
		} else {
			cerr << "[PatchDisabledEvent] Can not find window for patch " << m_patch_path
				<< "." << endl;
		}
	}
	
private:
	string m_patch_path;
};


class NewNodeEvent : public Event
{
public:
	NewNodeEvent(NodeModel* const nm) : m_node_model(nm) {}
	virtual ~NewNodeEvent() {}
	
	void execute()
	{
		//cerr << "[GtkClientHooks] New node: " << m_node_model->name() << endl;
		
		string parent_path = ClientPathParser::parent(m_node_model->path());
		
		PatchWindow* pw = app->patch_window(parent_path);
		
		if (pw != NULL) {

			// See if we cached this node model to store it's location (to avoid the
			// module "jumping")
			/*NodeModel* nm = controller->yank_added_node(m_node_model->path());
			if (nm != NULL) {
				m_node_model->x(nm->x());
				m_node_model->y(nm->y());
			} else { // pick a default
			*/
				int x, y;
				pw->patch_controller()->get_new_module_location(x, y);
				m_node_model->x(x);
				m_node_model->y(y);
			//}*/
			//cerr << "[NewNodeEvent] Creating node " << m_node_model->path() << endl;
			m_node_model->parent(pw->patch_controller()->model());
			pw->patch_controller()->new_node(m_node_model);
		} else {
			cerr << "[NewNodeEvent] Can not find window for patch " << parent_path
				<< ".  Module will not appear." << endl;
		}
	}
	
private:
	NodeModel* m_node_model;
};



class NodeRemovalEvent : public Event
{
public:
	NodeRemovalEvent(const string& path) : m_path(path) {}
	virtual ~NodeRemovalEvent() {}

	void execute()
	{
		//cerr << "[GtkClientHooks] Node removal" << endl;
	
		PatchWindow* pw = app->patch_window(ClientPathParser::parent(m_path));
		
		if (pw != NULL) {
			pw->patch_controller()->node_removal(ClientPathParser::name(m_path));
		} else {
			cerr << "[NodeRemoveEvent] Can not find window for patch " << m_path
				<< ".  Module will not be destroyed." << endl;
		}
	}

private:
	string m_path;
};


class ConnectionEvent : public Event
{
public:
	ConnectionEvent(ConnectionModel* const cm) : m_connection_model(cm) {}
	virtual ~ConnectionEvent() {}
	
	void execute()
	{
		//cerr << "[GtkClientHooks] Connection" << endl;
		
		ConnectionModel* cm = m_connection_model;
		PatchWindow* pw = app->patch_window(cm->patch_path());
		
		if (pw != NULL) {
			pw->patch_controller()->connection(cm);
		} else {
			cerr << "[ConnectionEvent] Can not find window for patch " << cm->patch_path()
				<< ".  Connection will not be made." << endl;
		}
	}

private:
	ConnectionModel* m_connection_model;
};



class DisconnectionEvent : public Event
{
public:
	DisconnectionEvent(const string& src_port_path, const string& dst_port_path)
	: m_src_port_path(src_port_path), m_dst_port_path(dst_port_path) {}
	virtual ~DisconnectionEvent() {}
	
	void execute()
	{
		//cerr << "[GtkClientHooks] Disconnection" << endl;
		string patch_path = m_src_port_path;
		patch_path = patch_path.substr(0, patch_path.find_last_of("/"));
		patch_path = patch_path.substr(0, patch_path.find_last_of("/"));
		
		PatchWindow* pw = app->patch_window(patch_path);
		
		if (pw != NULL) {
			pw->patch_controller()->disconnection(
			m_src_port_path, m_dst_port_path);
		} else {
			cerr << "[DisconnectionEvent] Can not find window for patch " << patch_path
				<< ".  Connection will not be removed." << endl;
		}
	}

private:
	string m_src_port_path;
	string m_dst_port_path;
};
	

class MetadataUpdateEvent : public Event
{
public:
	MetadataUpdateEvent(MetadataModel* const mm) : m_metadata_model(mm) {}
	virtual ~MetadataUpdateEvent() {}
	
	void execute()
	{
		string parent_path = ClientPathParser::parent(m_metadata_model->path());
		
		// Gtk client doesn't do anything with top level patch metadata (yet?)
		if (parent_path == "")
			return;

		// Maybe it's a port, try the parent's parent first
		// hack hack hack - a node has the same path as a port in the case of
		// inputs/outputs, which is probably bad - need to do this for port
		// ranges on a subpatch's node control window
		
		PatchWindow* pw = app->patch_window(parent_path);
		PatchWindow* parent_pw = app->patch_window(ClientPathParser::parent(parent_path));

		
		if (parent_pw != NULL) { // it's a node
			parent_pw->patch_controller()->metadata_update(m_metadata_model);
		}
		if (pw != NULL) { // it's a patch
			pw->patch_controller()->metadata_update(m_metadata_model);
			pw->port_metadata_update(m_metadata_model);
		}

		if (pw == NULL && parent_pw == NULL) {
			cerr << "[MetadataUpdateEvent] Can not find window for patch \'"
				<< ClientPathParser::parent(parent_path)
				<< "\'.  Metadata \'" << m_metadata_model->key() << 
				"\' will be discarded." << endl;
		}

		delete m_metadata_model;
	}

private:
	MetadataModel* m_metadata_model;
};



class ControlChangeEvent : public Event
{
public:
	ControlChangeEvent(ControlModel* const cm) : m_control_model(cm) {}
	virtual ~ControlChangeEvent() {}
	
	void execute()
	{
		//cerr << "Control change: " << m_control_model->port_path() << endl;

		string patch_path = ClientPathParser::parent(ClientPathParser::parent(
			m_control_model->port_path()));
		PatchWindow* pw;
		
		if (patch_path == "") { // port on a top level patch
			patch_path = ClientPathParser::parent(m_control_model->port_path());
			pw = app->patch_window(patch_path);
			if (pw != NULL) {
				pw->set_control(m_control_model);
				return;
			}
		}
		
		pw = app->patch_window(patch_path);
		
		// Ports on patches
		PatchWindow* parent_pw = app->patch_window(ClientPathParser::parent(patch_path));
			
		if (pw != NULL)
			pw->patch_controller()->control_change(m_control_model);
		if (parent_pw != NULL) {
			// hackity hack hack
			m_control_model->port_path(ClientPathParser::parent(m_control_model->port_path()));
			parent_pw->patch_controller()->control_change(m_control_model);
		}
		if (pw == NULL && parent_pw == NULL) {
			cerr << "[ControlChangeEvent] Can not find window for patch "
				<< patch_path << ".  Control change will be discarded." << endl;
		}

		delete m_control_model;
	}

private:
	ControlModel* m_control_model;
};
	


class NewPluginEvent : public Event
{
public:
	NewPluginEvent(PluginInfo* pi) : m_plugin_info(pi) {}
	virtual ~NewPluginEvent() {}

	void execute()
	{
		app->new_plugin(m_plugin_info);
	}

private:
	PluginInfo* m_plugin_info;
};

	
} // namespace OmGtk

#endif // GTKCLIENTHOOKSEVENTS_H
