/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef PORTMODEL_H
#define PORTMODEL_H

#include "ClientPathParser.h"
#include <string>
#include <list>
using std::string; using std::list; // using std::map;


enum PortType      { CONTROL, AUDIO };
enum PortDirection { INPUT, OUTPUT };
enum PortHint      { NONE, INTEGER, TOGGLE, LOGARITHMIC };


namespace LibOmClient {


class PortModel
{
public:
	PortModel(const string& port_path, PortType type, PortDirection dir, PortHint hint,
		float default_val, float min, float max)
	: m_path(port_path),
	  m_type(type),
	  m_direction(dir),
	  m_hint(hint),
	  m_default_val(default_val),
	  m_min_val(min),
	  m_user_min(min),
	  m_max_val(max),
	  m_user_max(max),
	  m_current_val(default_val)
	{
	}
	
	PortModel() {}  // Should only be used by patch loader


	const string& path() const          { return m_path; }
	void          path(const string& p) { m_path = p; }
	const string  name() const          { return ClientPathParser::name(m_path); }
	float         min_val() const       { return m_min_val; }
	float         user_min() const      { return m_user_min; }
	void          user_min(float f)     { m_user_min = f; }
	float         default_val() const   { return m_default_val; }
	void          default_val(float f)  { m_default_val = f; }
	float         max_val() const       { return m_max_val; }
	float         user_max() const      { return m_user_max; }
	void          user_max(float f)     { m_user_max = f; }
	float         value() const         { return m_current_val; }
	void          value(float f)        { m_current_val = f; }
	
	bool is_input() const       { return (m_direction == INPUT); }
	bool is_output() const      { return (m_direction == OUTPUT); }
	bool is_audio() const       { return (m_type == AUDIO); }
	bool is_control() const     { return (m_type == CONTROL); }
	bool is_logarithmic() const { return (m_hint == LOGARITHMIC); }
	bool is_integer() const     { return (m_hint == INTEGER); }
	bool is_toggle() const      { return (m_hint == TOGGLE); }
	
	inline bool operator==(const PortModel& pm)
		{ return (m_path == pm.m_path); }

private:
	PortModel(const PortModel& copy) { throw; }
	
	string        m_path;
	PortType      m_type;
	PortDirection m_direction;
	PortHint      m_hint;
	float         m_default_val;
	float         m_min_val;
	float         m_user_min;
	float         m_max_val;
	float         m_user_max;
	float         m_current_val;
};

typedef list<PortModel*>          PortModelList;


} // namespace LibOmClient

#endif // PORTMODEL_H
