/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef NODEMODEL_H
#define NODEMODEL_H

#include <map>
#include <iostream>
#include <string>
#include <algorithm>
#include "PortModel.h"
#include "ControlModel.h"
#include "PluginInfo.h"
#include "ClientPathParser.h"

using std::string; using std::map; using std::find;
using std::cout; using std::cerr; using std::endl;
using Om::PluginInfo;

namespace LibOmClient {

class PatchModel;

	
/** Node model class, used by the client to store engine's state.
 *
 * \ingroup libomclient
 */
class NodeModel
{
public:
	NodeModel(const string& node_path)
	: m_path(node_path), m_parent(NULL), m_plugin_info(NULL), m_poly(1), m_x(0.0f), m_y(0.0f)
	{
	}
	
	NodeModel()
	: m_parent(NULL), m_plugin_info(NULL), m_poly(1), m_x(0.0f), m_y(0.0f)
	{}

	NodeModel(const string& node_path, uint poly)
	: m_path(node_path), m_parent(NULL), m_plugin_info(NULL), m_poly(poly), m_x(0.0), m_y(0.0)
	{}

	virtual ~NodeModel() {
		for (PortModelList::iterator i = m_port_models.begin(); i != m_port_models.end(); ++i)
			delete (*i);
	}
	
	PortModel*
	port_model(const string& port_name)
	{
		//bool found = false;
		for (PortModelList::iterator i = m_port_models.begin(); i != m_port_models.end(); ++i)
			if ((*i)->name() == port_name)
				return (*i);
		
		return NULL;
	}
	
	bool operator==(const NodeModel& nm)
		{ return (m_path == nm.m_path); }
	
	bool operator==(const string& path) { return (m_path == path); }

	const PluginInfo* const  plugin_info() const               { return m_plugin_info; }
	void                     plugin_info(const PluginInfo* pi) { m_plugin_info = pi; }

	void control_change(const ControlModel* const cm) {
		//bool found = false;
		for (PortModelList::iterator i = m_port_models.begin(); i != m_port_models.end(); ++i) {
			if ((*i)->path() == cm->port_path()) {
				//found = true;
				(*i)->value(cm->value());
			}
		}
		//if (!found)
		//	cerr << "[NodeModel::control_change] WARNING:  Did not find port "
		//		<< cm->port_path() << endl;
	}

	uint poly() const   { return m_poly; }
	void poly (uint p)  { m_poly = p; }
	int                  num_ports() const             { return m_port_models.size(); }
	const PortModelList& port_models() const           { return m_port_models; }
	void                 add_port_model(PortModel* pm) { m_port_models.push_back(pm); }
	virtual bool         polyphonic() const            { return m_polyphonic; }
	void                 polyphonic(bool b)            { m_polyphonic = b; }
	const string&        path() const                  { return m_path; }
	void                 path(const string& p)         { m_path = p; }
	const string         name() const                  { return ClientPathParser::name(m_path); }
	PatchModel*          parent()                      { return m_parent; }
	void                 parent(PatchModel* pm)        { m_parent = pm; }
	float                x() const                     { return m_x; }
	void                 x(float a)                    { m_x = a; }
	float                y() const                     { return m_y; }
	void                 y(float a)                    { m_y = a; }
	
	const map<string, string>& metadata() const { return m_metadata; }
	void set_metadata(const string& key, const string& value) { m_metadata[key] = value; }

protected:
	NodeModel(const NodeModel& copy) { throw; }

	string            m_path;
	PatchModel*       m_parent;
	bool              m_polyphonic;
	PortModelList     m_port_models; // list instead of map to preserve order
	const PluginInfo* m_plugin_info; // info about the plugin this node is an instance of
	uint              m_poly;
	float             m_x;
	float             m_y;

	map<string, string> m_metadata;
};


typedef map<string, NodeModel*>   NodeModelMap;


} // namespace LibOmClient

#endif // NODEMODEL_H
