/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef COMM_H
#define COMM_H

#include <list>
#include <lo/lo.h>
#include <iostream>
#include "ClientHooks.h"

using std::string; using std::list;

/** Om C++ client library namespace.
 *
 * \ingroup libomclient
 */
namespace LibOmClient {

class NodeModel;
class PresetModel;


/** \defgroup libomclient Om client library.
 *
 * This library wraps all the low-level OSC communication in nice blocking
 * functions, and provides some other things that should be useful to any
 * Om client.
 */



/** Provides OSC communication to clients.
 *
 * Basically, this wraps all the OSC communication stuff in normal, blocking
 * function calls with returns and whatnot.
 *
 * \ingroup libomclient
 */
class Comm
{
public:
	Comm(ClientHooks* ch);
	Comm();

	~Comm();
	
	void attach(const string& engine_host, int engine_port, int client_port = 0);
	void attach_url(const string& engine_url, int client_port = 0);
	
	void detach();

	void register_client(const string& client_host = "", int client_port = 0);
	
	string engine_url() { return lo_address_get_url(m_engine_addr); }

	void set_wait_response_id(int id);
	bool wait_for_response();
	
	int get_next_request_id() { return m_request_id++; }

	// Commands //
	
	
	// Engine commands
	void load_plugins() { load_plugins(m_request_id++); }
	void load_plugins(int id);
	void activate();
	void deactivate();
	void enable();
	void disable();
	int quit();
			
	// Patch commands
	int create_patch(const PatchModel* pm) { return create_patch(pm, m_request_id++); }
	int create_patch(const PatchModel* pm, int id);
	int destroy_patch(const string& patch_path);
	void enable_patch(const string& patch_path);
	void disable_patch(const string& patch_path);
	int add_node(const NodeModel* nm) { return add_node(nm, m_request_id++); }
	int add_node(const NodeModel* nm, int id);
	int remove_node(const string& node_path);
	int connect(const string& port1_path, const string& port2_path);
	int disconnect(const string& port1_path, const string& port2_path);
	int disconnect_all(const string& node_path);
	int set_control(const string& port_path, float val);
	int set_control(const string& port_path, int voice, float val);
	int set_control_slow(const string& port_path, float val);
	int midi_learn(const string& node_path);

	// Metadata commands
	int set_metadata(const string& obj_path, const string& key, const string& value);

	// Requests //
	void ping(int id);
	int request_control(const string& port_path);
	int request_plugins();
	int request_all_objects();

	// Utility commands, etc (ie things that don't map directly to OSC) //
	int set_all_metadata(const NodeModel* nm);
	void set_preset(const string& patch_path, const PresetModel* const pm);

private:
	void start_listen_thread(int client_port = 0);
	
	void m_attach(lo_address engine_addr, int client_port = 0);

	ClientHooks* m_client_hooks;

	lo_server_thread m_st;
	lo_address       m_engine_addr;
	
	int m_request_id;

	bool m_waiting_for_response;
	int  m_wait_response_id;
	bool m_response_received;
	bool m_wait_response_was_affirmative;
	
	pthread_mutex_t m_response_mutex;
	pthread_cond_t  m_response_cond;

	// Used for receiving nodes - multiple messages are received before
	// sending an event to the client (via ClientHooks)
	bool m_receiving_node;
	NodeModel* m_receiving_node_model;
	
	// liblo callbacks
	
	static void error_cb(int num, const char* msg, const char* path);	
	static int  generic_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* user_data);
	static int  unknown_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* osc_receiver);
	
	inline static int om_response_ok_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_om_response_ok_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int om_response_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_om_response_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);

	inline static int om_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_om_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int plugin_list_begin_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_plugin_list_begin_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int plugin_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_plugin_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int plugin_list_end_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_plugin_list_end_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int engine_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_engine_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int engine_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_engine_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int new_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_new_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int patch_destruction_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_patch_destruction_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int patch_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_patch_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int patch_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_patch_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int connection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_connection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int disconnection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_disconnection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int node_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_node_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int new_node_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_new_node_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int new_node_end_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_new_node_end_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int new_port_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_new_port_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int port_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_port_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int metadata_update_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_metadata_update_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
	inline static int control_change_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm);
	int             m_control_change_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data);
};


inline int
Comm::om_response_ok_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_om_response_ok_cb(path, types, argv, argc, data);
}

inline int
Comm::om_response_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_om_response_error_cb(path, types, argv, argc, data);
}

inline int
Comm::om_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_om_error_cb(path, types, argv, argc, data);
}

inline int
Comm::plugin_list_begin_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_plugin_list_begin_cb(path, types, argv, argc, data);
}

inline int
Comm::plugin_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_plugin_cb(path, types, argv, argc, data);
}

inline int
Comm::plugin_list_end_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_plugin_list_end_cb(path, types, argv, argc, data);
}

inline int
Comm::engine_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_engine_enabled_cb(path, types, argv, argc, data);
}

inline int
Comm::engine_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_engine_disabled_cb(path, types, argv, argc, data);
}

inline int
Comm::new_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_new_patch_cb(path, types, argv, argc, data);
}

inline int
Comm::patch_destruction_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_patch_destruction_cb(path, types, argv, argc, data);
}

inline int
Comm::patch_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_patch_enabled_cb(path, types, argv, argc, data);
}

inline int
Comm::patch_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_patch_disabled_cb(path, types, argv, argc, data);
}

inline int
Comm::connection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_connection_cb(path, types, argv, argc, data);
}

inline int
Comm::disconnection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_disconnection_cb(path, types, argv, argc, data);
}

inline int
Comm::node_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_node_removal_cb(path, types, argv, argc, data);
}

inline int
Comm::new_node_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_new_node_cb(path, types, argv, argc, data);
}

inline int
Comm::new_node_end_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_new_node_end_cb(path, types, argv, argc, data);
}

inline int
Comm::new_port_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_new_port_cb(path, types, argv, argc, data);
}

inline int
Comm::port_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_port_removal_cb(path, types, argv, argc, data);
}

inline int
Comm::metadata_update_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_metadata_update_cb(path, types, argv, argc, data);
}

inline int
Comm::control_change_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* comm) {
	return ((Comm*)comm)->m_control_change_cb(path, types, argv, argc, data);
}


} // namespace LibOmClient

#endif // COMM_H
