/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "Comm.h"
#include "DummyClientHooks.h"
#include "PatchModel.h"
#include "ConnectionModel.h"
#include "MetadataModel.h"
#include "PresetModel.h"
#include "ControlModel.h"
#include "NodeModel.h"
#include "PluginInfo.h"
#include <list>
#include <cassert>
#include <cstring>
#include <iostream>
using std::cerr; using std::cout; using std::endl;
using Om::PluginInfo;

namespace LibOmClient {

/** Construct a Comm with a user-provided ClientHooks object for notification
 *  of engine events.
 */
Comm::Comm(ClientHooks* ch)
: m_client_hooks(ch),
  m_st(NULL),
  m_engine_addr(NULL),
  m_request_id(0),
  m_waiting_for_response(false),
  m_wait_response_id(0),
  m_response_received(false),
  m_wait_response_was_affirmative(false)
{
	pthread_mutex_init(&m_response_mutex, NULL);
	pthread_cond_init(&m_response_cond, NULL);
}


/** Construct a Comm with no ClientHooks object.
 *
 * The user will receive no notification of anything that happens in the
 * engine.  (If this seems silly, it's because I don't want to have
 * "if (m_client_hooks != NULL)" before every single call to a ClientHooks
 * method in this code).
 */
Comm::Comm()
: m_client_hooks(new DummyClientHooks()), // FIXME: leak
  m_st(NULL),
  m_engine_addr(NULL),
  m_request_id(0),
  m_waiting_for_response(false),
  m_wait_response_id(0),
  m_response_received(false)
{	
	pthread_mutex_init(&m_response_mutex, NULL);
	pthread_cond_init(&m_response_cond, NULL);
}


Comm::~Comm()
{
	detach();
	pthread_mutex_destroy(&m_response_mutex);
	pthread_cond_destroy(&m_response_cond);
}


/** Connect to the Om engine and notify it of our existance.
 *
 * Pass the empty string for client_host to have the engine use the incoming
 * address - parameter is just for working around bugs.
 */
void
Comm::attach(const string& engine_host, int engine_port, int client_port)
{
	char port_str[8];
	snprintf(port_str, 8, "%d", engine_port);
	m_attach(lo_address_new(engine_host.c_str(), port_str), client_port);
}


/** Connect to the Om engine and notify it of our existance.
 *
 * Pass the empty string for client_host to have the engine use the incoming
 * address - parameter is just for working around bugs.
 */
void
Comm::attach_url(const string& engine_url, int client_port)
{
	m_attach(lo_address_new_from_url(engine_url.c_str()), client_port);
}


/** Register this client with the engine.
 *
 * A registered client will receive notifications of everything happening with
 * the engine (ie new nodes, etc).
 */
void
Comm::register_client(const string& client_host, int client_port)
{
	assert(m_engine_addr != NULL);

	int id = m_request_id++;
	set_wait_response_id(id);
	
	if (client_host == "" && client_port == 0)
		lo_send(m_engine_addr, "/om/engine/register_client", "i", id);
	else if (client_host != "" && client_port == 0)
		lo_send(m_engine_addr, "/om/engine/register_client", "isi",
			id, client_host.c_str(), lo_server_thread_get_port(m_st));
	else
		lo_send(m_engine_addr, "/om/engine/register_client", "isi",
			id, client_host.c_str(), client_port);
	
	wait_for_response();
}


/** Connect to the Om engine and notify it of our existance.
 *
 * Optionally takes the client's address (to advertise to the engine) as
 * a parameter, useful for working around networking problems sometimes.
 */
void
Comm::m_attach(lo_address engine_addr, int client_port)
{
	start_listen_thread(client_port);

	m_engine_addr = engine_addr;
	if (m_engine_addr == NULL) {
		cerr << "Unable to connect, aborting." << endl;
		throw;
	}

	char* lo_url = lo_address_get_url(m_engine_addr);
	cout << "[Comm] Sending to engine at address " << lo_url << endl;
	free(lo_url);
}


void
Comm::detach()
{
	lo_send(m_engine_addr, "/om/engine/unregister_client", "i", m_request_id++);
	lo_server_thread_free(m_st);
}


void
Comm::start_listen_thread(int client_port)
{
	if (client_port == 0) { 
		m_st = lo_server_thread_new(NULL, error_cb);
	} else {
		char port_str[8];
		snprintf(port_str, 8, "%d", client_port);
		m_st = lo_server_thread_new(port_str, error_cb);
	}

	if (m_st == NULL) {
		cerr << "[Comm] Could not start OSC listener.  Aborting." << endl;
		throw;
	} else {
		cout << "[Comm] Started OSC listener on port " << lo_server_thread_get_port(m_st) << " ***" << endl;
	}

	//lo_server_thread_add_method(m_st, NULL, NULL, generic_cb, NULL);

	lo_server_thread_add_method(m_st, "/om/response/ok", "i", om_response_ok_cb, this);
	lo_server_thread_add_method(m_st, "/om/response/error", "is", om_response_error_cb, this);
	lo_server_thread_add_method(m_st, "/om/error", "s", om_error_cb, this);
	//lo_server_thread_add_method(m_st, "/om/plugin_list_begin", "", plugin_list_begin_cb, this);
	lo_server_thread_add_method(m_st, "/om/plugin", "ssss", plugin_cb, this);
	//lo_server_thread_add_method(m_st, "/om/plugin_list_end", "", plugin_list_end_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine_enabled", "", engine_enabled_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine_disabled", "", engine_disabled_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_patch", "si", new_patch_cb, this);
	lo_server_thread_add_method(m_st, "/om/patch_destruction", "s", patch_destruction_cb, this);
	lo_server_thread_add_method(m_st, "/om/patch_enabled", "s", patch_enabled_cb, this);
	lo_server_thread_add_method(m_st, "/om/patch_disabled", "s", patch_disabled_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_connection", "ss", connection_cb, this);
	lo_server_thread_add_method(m_st, "/om/disconnection", "ss", disconnection_cb, this);
	lo_server_thread_add_method(m_st, "/om/node_removal", "s", node_removal_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_node", "sisss", new_node_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_node_end", "", new_node_end_cb, this);
	lo_server_thread_add_method(m_st, "/om/new_port", "ssssfff", new_port_cb, this);
	lo_server_thread_add_method(m_st, "/om/port_removal", "s", port_removal_cb, this);
	lo_server_thread_add_method(m_st, "/om/metadata/update", "sss", metadata_update_cb, this);
	lo_server_thread_add_method(m_st, "/om/control_change", "sf", control_change_cb, this);
	
	lo_server_thread_add_method(m_st, NULL, NULL, generic_cb, NULL);
	
	lo_server_thread_start(m_st);
}



///// Commands /////



/** Pings the engine.
 */
void
Comm::ping(int id)
{
	lo_send(m_engine_addr, "/om/ping", "i", id);
}


/** Activate the engine.
 */
void
Comm::activate()
{
	int id = m_request_id++;
	
	/*set_wait_response_id(id);
	lo_send(m_engine_addr, "/om/engine/activate", "i", id);
	wait_for_response();*/

	lo_send(m_engine_addr, "/om/engine/activate", "i", id);
}


/** Deactivate the engine.
 */
void
Comm::deactivate()
{
	int id = m_request_id++;
	
	/*set_wait_response_id(id);
	lo_send(m_engine_addr, "/om/engine/deactivate", "i", id);
	wait_for_response();*/

	lo_send(m_engine_addr, "/om/engine/deactivate", "i", id);
}


/** Enable DSP processing.
 */
void
Comm::enable()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/engine/enable", "i", id);
}


/** Disable DSP processing.
 */
void
Comm::disable()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/engine/disable", "i", id);
}


/** Kill the engine.
 */
int
Comm::quit()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/engine/quit", "i", id);
	return id;
}


/** Load all (LADSPA, etc) plugins into the engine's database.
 */
void
Comm::load_plugins(int id)
{
	lo_send(m_engine_addr, "/om/engine/load_plugins", "i", id);
}


/** Load a node.
 */
int
Comm::add_node(const NodeModel* nm, int id)
{
	assert(m_engine_addr != NULL);
	
	lo_send(m_engine_addr, "/om/synth/create_node",  "issssi", id, nm->path().c_str(),
	        nm->plugin_info()->type_string(),
	        nm->plugin_info()->lib_name().c_str(), nm->plugin_info()->plug_label().c_str(),
			(nm->polyphonic() ? 1 : 0));
	
	return id;
}


/** Remove a node.
 */
int
Comm::remove_node(const string& node_path)
{
	int id = m_request_id++;
	assert(m_engine_addr != NULL);
	lo_send(m_engine_addr, "/om/synth/destroy_node", "is", id, node_path.c_str());
	return id;	
}


/** Create a patch.
 */
int
Comm::create_patch(const PatchModel* pm, int id)
{
	assert(m_engine_addr != NULL);
	lo_send(m_engine_addr, "/om/synth/create_patch", "isi", id, pm->path().c_str(), pm->poly());
	return id;
}


/** Destroy a patch.
 */
int
Comm::destroy_patch(const string& patch_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/destroy_patch", "is", id, patch_path.c_str());
	return id;
}


/** Enable a patch.
 */
void
Comm::enable_patch(const string& patch_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/enable_patch", "is", id, patch_path.c_str());
}


/** Disable a patch.
 */
void
Comm::disable_patch(const string& patch_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/disable_patch", "is", id, patch_path.c_str());
}


/** Connect two ports.
 */
int
Comm::connect(const string& port1_path, const string& port2_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/connect", "iss", id, port1_path.c_str(), port2_path.c_str());
	return id;
}


/** Disconnect two ports.
 */
int
Comm::disconnect(const string& port1_path, const string& port2_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/disconnect", "iss", id, port1_path.c_str(), port2_path.c_str());
	return id;
}


/** Disconnect all connections from a node.
 */
int
Comm::disconnect_all(const string& node_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/disconnect_all", "is", id, node_path.c_str());
	return id;
}


/** Set a port's value.
 */
int
Comm::set_control(const string& port_path, const float val)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/set_port_value", "isf", id, port_path.c_str(), val);
	return id;
}


/** Set a port's value for a specific voice.
 */
int
Comm::set_control(const string& port_path, int voice, const float val)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/set_port_value", "isif", id, port_path.c_str(), voice, val);
	return id;
}


/** Set a port's value.
 */
int
Comm::set_control_slow(const string& port_path, const float val)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/synth/set_port_value_slow", "isf", id, port_path.c_str(), val);
	return id;
}


/** Set a preset by setting all relevant controls for a patch.
 */
void
Comm::set_preset(const string& patch_path, const PresetModel* const pm)
{
	for (list<ControlModel>::const_iterator i = pm->controls().begin(); i != pm->controls().end(); ++i) {
		set_control_slow((*i).port_path(), (*i).value());
		usleep(1000);
	}
}


/** Learn a binding for a Midi control node.
 */
int
Comm::midi_learn(const string& node_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/midi/learn", "is", id, node_path.c_str());
	return id;
}


/** Set/add a piece of metadata.
 */
int
Comm::set_metadata(const string& obj_path,
                   const string& key, const string& value)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;

	// Deal with the "special" DSSI metadata strings
	if (key == "dssi-configure") {
		string path = "/dssi" + obj_path + "/configure";
		string dssi_key = value.substr(0, value.find("="));
		string dssi_value = value.substr(value.find("=")+1);
		lo_send(m_engine_addr, path.c_str(), "ss", dssi_key.c_str(), dssi_value.c_str());
	} else if (key == "dssi-program") {
		string path = "/dssi" + obj_path + "/program";
		string dssi_bank_str = value.substr(0, value.find("/"));
		int dssi_bank = atoi(dssi_bank_str.c_str());
		string dssi_program_str = value.substr(value.find("/")+1);
		int dssi_program = atoi(dssi_program_str.c_str());
		lo_send(m_engine_addr, path.c_str(), "ii", dssi_bank, dssi_program);
	}
	
	// normal metadata
	lo_send(m_engine_addr, "/om/metadata/set", "isss", id,
		obj_path.c_str(), key.c_str(), value.c_str());

	return id;
}


/** Set all pieces of metadata in a NodeModel.
 */
int
Comm::set_all_metadata(const NodeModel* nm)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;

	for (map<string, string>::const_iterator i = nm->metadata().begin(); i != nm->metadata().end(); ++i) {
		set_metadata(nm->path(), (*i).first, (*i).second.c_str());
	}

	return id;
}


///// Requests /////



int
Comm::request_control(const string& port_path)
{
	assert(m_engine_addr != NULL);
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/request/port_value", "is", id, port_path.c_str());
	return id;
}


/** Requests all known plugins from the server.
 */
int
Comm::request_plugins()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/request/plugins", "i", id);
	return id;
}


/** Requests all object from the server (Nodes, Patches, Ports).
 */
int
Comm::request_all_objects()
{
	int id = m_request_id++;
	lo_send(m_engine_addr, "/om/request/all_objects", "i", id);
	return id;
}


/** Sets the response ID to be waited for on the next call to wait_for_response()
 */
void
Comm::set_wait_response_id(int id)
{	
	pthread_mutex_lock(&m_response_mutex);

	assert(!m_waiting_for_response);
	m_wait_response_id = id;
	m_response_received = false;
	
	pthread_mutex_unlock(&m_response_mutex);
}


/** Waits for the response set by set_wait_response() from the server.
 *
 * Returns whether or not the response was positive (ie a success message)
 * or negative (ie an error)
 */
bool
Comm::wait_for_response()
{
	cerr << "Waiting for response " << m_wait_response_id << ": ";
	bool ret = true;
	
	pthread_mutex_lock(&m_response_mutex);

	m_waiting_for_response = true;
	
	if ( ! m_response_received) {
		while (!m_response_received) {
			pthread_cond_wait(&m_response_cond, &m_response_mutex);
			cerr << " . ";
		}
	}
	cerr << " received." << endl;

	m_waiting_for_response = false;
	ret = m_wait_response_was_affirmative;
	
	pthread_mutex_unlock(&m_response_mutex);

	return ret;
}


///// Static OSC callbacks //////


void
Comm::error_cb(int num, const char* msg, const char* path)
{
	cerr << "Got error from server: " << msg << endl;
}


int
Comm::generic_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* user_data)
{
	if ( ! strcmp(path, "/om/plugin") ) return 1;

    printf("Path:  %s\n", path);
    
	for (int i=0; i < argc; ++i) {
		printf("Arg %d: '%c'  ", i, types[i]);
		lo_arg_pp(lo_type(types[i]), argv[i]);
		printf("\n");
    }
    printf("\n");
	
	return 1;  // not handled
}


int
Comm::unknown_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data, void* user_data)
{
	string msg = "Received unknown OSC message: ";
	msg += path;

	cerr << msg << endl;

	return 0;
}



//// End static callbacks, member callbacks below ////


int
Comm::m_om_response_ok_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{	
	assert(argc == 1 && !strcmp(types, "i"));
	
	const int request_id = argv[0]->i;

	pthread_mutex_lock(&m_response_mutex);
	
	if (request_id == m_wait_response_id) {
		m_response_received = true;
		m_wait_response_was_affirmative = true;
	}
	
	pthread_cond_signal(&m_response_cond);
	pthread_mutex_unlock(&m_response_mutex);

	return 0;
}


int
Comm::m_om_response_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	assert(argc == 2 && !strcmp(types, "is"));

	const int   request_id =  argv[0]->i;
	const char* msg        = &argv[1]->s;
	
	pthread_mutex_lock(&m_response_mutex);
	
	if (request_id == m_wait_response_id) {
		m_response_received = true;
		m_wait_response_was_affirmative = false;
	}

	m_client_hooks->error(msg);
	
	pthread_cond_signal(&m_response_cond);
	pthread_mutex_unlock(&m_response_mutex);

	return 0;
}


/** Catches errors that aren't a direct result of a client request.
 */
int
Comm::m_om_error_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->error((char*)argv[0]);
	return 0;
}
 

int
Comm::m_engine_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->engine_enabled();
	return 0;
}


int
Comm::m_engine_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->engine_disabled();
	return 0;
}


int
Comm::m_new_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	PatchModel* pm = new PatchModel(&argv[0]->s, argv[1]->i);
	PluginInfo* pi = new PluginInfo();
	pi->type(PluginInfo::Patch);
	pm->plugin_info(pi);

	m_client_hooks->new_patch(pm);
	return 0;
}


int
Comm::m_patch_destruction_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->patch_destruction((const char*)&argv[0]->s);
	return 0;
}


int
Comm::m_patch_enabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->patch_enabled((const char*)&argv[0]->s);
	return 0;
}


int
Comm::m_patch_disabled_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->patch_disabled((const char*)&argv[0]->s);
	return 0;
}


int
Comm::m_connection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* src_port_path = &argv[0]->s;
	const char* dst_port_path = &argv[1]->s;
	
	m_client_hooks->connection(new ConnectionModel(src_port_path, dst_port_path));

	return 0;
}


int
Comm::m_disconnection_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* src_port_path = &argv[0]->s;
	const char* dst_port_path = &argv[1]->s;

	m_client_hooks->disconnection(src_port_path, dst_port_path);

	return 0;
}


int
Comm::m_node_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->node_removal((const char*)&argv[0]->s);
	
	return 0;
}


/** Notification of a new node creation.
 */
int
Comm::m_new_node_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* node_path  = &argv[0]->s;
	const int   poly       =  argv[1]->i;
	const char* type       = &argv[2]->s;
	const char* lib_name   = &argv[3]->s;
	const char* plug_label = &argv[4]->s;

	m_receiving_node_model = new NodeModel(node_path);
	m_receiving_node_model->polyphonic((poly == 1));

	PluginInfo* pi = new PluginInfo();
	pi->set_type(type);
	pi->lib_name(lib_name);
	pi->plug_label(plug_label);
	m_receiving_node_model->plugin_info(pi);
	
	m_receiving_node = true;
	
	return 0;
}


/** Notification of the end of a node transmission (which is a bundle).
 */
int
Comm::m_new_node_end_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	m_client_hooks->new_node(m_receiving_node_model);
	m_receiving_node = false;
	m_receiving_node_model = NULL;
	
	return 0;
}


/** Notification of a new port creation.
 */
int
Comm::m_new_port_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* port_path   = &argv[0]->s;
	const char* type        = &argv[1]->s;
	const char* direction   = &argv[2]->s;
	const char* hint        = &argv[3]->s;
	float       default_val =  argv[4]->f;
	float       min_val     =  argv[5]->f;
	float       max_val     =  argv[6]->f;

	PortType ptype;
	if (!strcmp(type, "AUDIO")) ptype = AUDIO;
	else if (!strcmp(type, "CONTROL")) ptype = CONTROL;
	else throw;

	PortDirection pdir;
	if (!strcmp(direction, "INPUT")) pdir = INPUT;
	else if (!strcmp(direction, "OUTPUT")) pdir = OUTPUT;
	else throw;

	PortHint phint;
	if (!strcmp(hint, "LOGARITHMIC")) phint = LOGARITHMIC;
	else if (!strcmp(hint, "INTEGER")) phint = INTEGER;
	else if (!strcmp(hint, "TOGGLE")) phint = TOGGLE;
	else phint = NONE;
	
	PortModel* port_model = new PortModel(port_path, ptype, pdir, phint, default_val, min_val, max_val);
	
	if (m_receiving_node) {
		m_receiving_node_model->add_port_model(port_model);
	} else {
		m_client_hooks->new_port(port_model);
	}

	return 0;	
}


/** Notification of a port's removal (destruction).
 */
int
Comm::m_port_removal_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* port_path = &argv[0]->s;

	m_client_hooks->port_removal(port_path);

	return 0;
}


/** Notification of a new or updated piece of metadata.
 */
int
Comm::m_metadata_update_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* node_path  = &argv[0]->s;
	const char* key        = &argv[1]->s;
	const char* value      = &argv[2]->s;

	m_client_hooks->metadata_update(new MetadataModel(node_path, key, value));

	return 0;	
}


int
Comm::m_control_change_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	const char* port_path  = &argv[0]->s;
	float       value      =  argv[1]->f;

	m_client_hooks->control_change(new ControlModel(port_path, value));

	return 0;	
}


/** A plugin info response from the server, in response to a /send_plugins
 */
int
Comm::m_plugin_cb(const char* path, const char* types, lo_arg** argv, int argc, void* data)
{
	//static int count = 0;
	//++count;
	
	Om::PluginInfo* info = new Om::PluginInfo();
	info->lib_name(&argv[0]->s);
	info->plug_label(&argv[1]->s);
	info->name(&argv[2]->s);
	info->set_type(&argv[3]->s);
	
	//cout << "[Comm] Received plugin " << info->plug_label() << " (" << count << " plugins)" << endl;

	m_client_hooks->new_plugin(info);

	return 0;	
}

} // namespace LibOmClient
