#!/usr/bin/perl -w
#
# Copyright (C) 2003,2004,2006,2008 Dmitry Fedorov <dm.fedorov@gmail.com>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA


=head1 NAME

offmirror-list - get list of files

=head1 SYNOPSIS

offmirror-list basedir [options]

=head1 DESCRIPTION

This program is part of files tree offline mirroring utility package,
which is used to synchronize file trees without any direct connection
between them.

This program walks files tree and produce file list with file attributes.

=cut

require 5.004;
use strict;
local $^W=1; # use warnings only since 5.006

use integer;
use Getopt::Long;

use FindBin;
use lib "$FindBin::Bin/../lib";
use OffMirror::FileList;


my $progname = 'offmirror-list';

sub usage {
	warn "\n".join(" ", @_)."\n" if @_;
	warn <<EOF;

Usage: $progname basedir [-help] [-verbose] \
		[-exclude=regex] [-no-md5sums] [-pretty]
		[-host=HOST] [-dir=DIR]

For details, see man page.
EOF
	exit(1);
}

=head1 OPTIONS

=over 4

=item basedir

This required parameter specifies where the root of the files tree is.

=item -exclude=REGEX

Perl regular expression.
File entries matched after basedir are excluded from the generated list.

The option may be repeated many times.

Example:

-exclude='^lost\+found' -exclude='CVS'

-exclude='.*' # To generate empty list


=item -no-md5sums

Don't compute MD5 digest for each of files.

=item -pretty

Print file attribute fields aligned in columns.

=item -host=HOSTNAME

Override host name field in generated list header

=item -dir=DIR

Override dir name field in generated list header

=item -verbose

List to STDERR files processed

=item -help

Display a usage summary.

=back

=head1 SEE ALSO

L<offmirror(1)>

=cut


my ( $help, $nomd5sums, $pretty, $host, $dir, $verbose, @exclude ) =
	(undef,undef,undef,undef,undef,());

GetOptions(
	'exclude=s'	=> \@exclude,
	'no-md5sums'	=> \$nomd5sums,
	'pretty'	=> \$pretty,
	'host=s'	=> \$host,
	'dir=s'		=> \$dir,
	'verbose'	=> \$verbose,
	'help'		=> \$help
) or usage;

usage if $help;

my $basedir = shift @ARGV or  usage("basedir not specified");
              shift @ARGV and usage("extra parameter specified");

OffMirror::FileList::generate_list($basedir, \@exclude, !$nomd5sums, $pretty,
				   $host, $dir, $verbose);


=head1 AUTHOR

Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 COPYRIGHT

Copyright (C) 2003,2004,2006,2008 Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 LICENSE

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License,
or (at your option) any later version.

=head1 DISCLAIMER

The author disclaims any responsibility for any mangling of your system
etc, that this script may cause.

=cut
