#!/usr/bin/perl -w
#
# Copyright (C) 2003,2004,2006,2008 Dmitry Fedorov <dm.fedorov@gmail.com>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA


=head1 NAME

offmirror-diff - compare offmirror file lists

=head1 SYNOPSIS

offmirror-diff src-list dst-list [options]

=head1 DESCRIPTION

This program is part of files tree offline mirroring utility package,
which is used to synchronize file trees without any direct connection
between them.

This program compares two file tree lists and produces list of files
which are needed to synchronize.

=cut

require 5.004;
use strict;
local $^W=1; # use warnings only since 5.006

use integer;
use Getopt::Long;

use FindBin;
use lib "$FindBin::Bin/../lib";
use OffMirror::FileList;


my $progname = 'offmirror-diff';

sub usage
{
	warn "\n".join(" ", @_)."\n" if @_;
	warn <<EOF;

Usage: $progname src-list dst-list [-pretty] [-help]

For details, see man page.
EOF
	exit(1);
}


=head1 OPTIONS

=over 4

=item src-list[.gz|.bz2|.Z|.zip]

Specifies file with source files tree list.

The file can be compressed with gzip or bzip2 or compress or
zip (single file only).

=item dst-list[.gz|.bz2|.Z|.zip]

Specifies file with destination files tree list.

The file can be compressed with gzip or bzip2 or compress or
zip (single file only).

=item -pretty

Print file attribute fields aligned in columns.

=item -help

Display a usage summary.

=back

=head1 SEE ALSO

L<offmirror(1)>

=cut


my ( $help, $pretty ) = (0,0);

GetOptions(
	'pretty'	=> \$pretty,
	'help'		=> \$help
) or usage;

usage if $help;

my $src_list_file = shift @ARGV or  usage("no src-list file specified");
my $dst_list_file = shift @ARGV or  usage("no dst-list file specified");
                    shift @ARGV and usage("extra parameter specified");

my $src_list = OffMirror::FileList::list_from_file($src_list_file);
my $dst_list = OffMirror::FileList::list_from_file($dst_list_file);

OffMirror::FileList::diff_list($src_list, $dst_list, $pretty);

exit 0;


=head1 AUTHOR

Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 COPYRIGHT

Copyright (C) 2003,2004,2006,2008 Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 LICENSE

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License,
or (at your option) any later version.

=head1 DISCLAIMER

The author disclaims any responsibility for any mangling of your system
etc, that this script may cause.

=cut
