/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <algorithm>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "vrhomboid.h"
#include "track.h"


// rectangle_vector must be ordered by increasing hcenter().
void Track::initialize( const std::vector< Rectangle > rectangle_vector ) throw()
  {
  if( data.size() ) data.clear();
  if( !rectangle_vector.size() ) return;

  int rectangles = rectangle_vector.size();
  int mean_vcenter = 0, mean_height = 0, mean_width = 0;

  for( int i = 0; i < rectangles; ++i )
    {
    mean_vcenter += rectangle_vector[i].vcenter();
    mean_height += rectangle_vector[i].height();
    mean_width += rectangle_vector[i].width();
    }
  if( rectangles )
    { mean_vcenter /= rectangles; mean_height /= rectangles; mean_width /= rectangles; }

  int l, r, lmax = rectangles / 4, rmin = rectangles - 1 - lmax;
  for( l = 0; l < lmax; ++l )
    {
    const Rectangle & r1 = rectangle_vector[l];
    const Rectangle & r2 = rectangle_vector[l+1];
    if( 5 * r1.height() > 4 * mean_height &&
        5 * r2.height() > 4 * mean_height &&
        10 * std::abs( r1.vcenter() - r2.vcenter() ) <= mean_height ) break;
    }
  for( r = rectangles - 1; r > rmin; --r )
    {
    const Rectangle & r1 = rectangle_vector[r];
    const Rectangle & r2 = rectangle_vector[r-1];
    if( 5 * r1.height() > 4 * mean_height &&
        5 * r2.height() > 4 * mean_height &&
        10 * std::abs( r1.vcenter() - r2.vcenter() ) <= mean_height ) break;
    }

  if( rectangles < 10 || l >= lmax || r <= rmin )
    {
    const Rectangle & r1 = rectangle_vector[0];
    const Rectangle & r2 = rectangle_vector[rectangles-1];
    data.push_back( Vrhomboid( r1.hcenter(), mean_vcenter,
                               r2.hcenter(), mean_vcenter, mean_height ) );
    return;
    }

  const Rectangle & r1 = rectangle_vector[l];
  const Rectangle & r2 = rectangle_vector[r];
  data.push_back( Vrhomboid( r1.hcenter(), r1.vcenter(),
                             r2.hcenter(), r2.vcenter(), mean_height ) );
  if( l != 0 ) data[0].extend_left( rectangle_vector[0].hcenter() );
  if( r != rectangles - 1 )
    data[0].extend_right( rectangle_vector[rectangles - 1].hcenter() );
  }


int Track::height() const throw()
  {
  if( data.size() ) return data[0].height(); else return 0;
  }


int Track::vcenter( int col ) const throw()
  {
  for( unsigned int i = 0; i < data.size(); ++i )
    {
    const Vrhomboid & vr = data[i];
    if( col <= vr.right() || i >= data.size() - 1 ) return vr.vcenter( col );
    }
  return 0;
  }


int Track::bottom( int col ) const throw()
  {
  for( unsigned int i = 0; i < data.size(); ++i )
    {
    const Vrhomboid & vr = data[i];
    if( col <= vr.right() || i >= data.size() - 1 ) return vr.bottom( col );
    }
  return 0;
  }


int Track::top( int col ) const throw()
  {
  for( unsigned int i = 0; i < data.size(); ++i )
    {
    const Vrhomboid & vr = data[i];
    if( col <= vr.right() || i >= data.size() - 1 ) return vr.top( col );
    }
  return 0;
  }
