/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <cctype>
#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "ucs.h"
#include "block.h"
#include "character.h"
#include "textline.h"


int Textline::find_space( int begin ) const throw()
  {
  int end = begin;
  for( ; end < characters(); ++end )
    {
    const Character & c = character( end );
    if( c.guesses() && UCS::isspace( c.guess(0).code ) ) break;
    }
  return end;
  }


const Character & Textline::add_character( const Character & c ) throw()
  {
  data.push_back( c ); return data.back();
  }


void Textline::insert_character( int i, const Character & c ) throw()
  {
  if( i < 0 || i > characters() )
    Ocrad::internal_error( "insert_character, index out of bounds" );
  data.insert( data.begin() + i, c );
  }


void Textline::delete_character( int i ) throw()
  {
  if( i < 0 || i >= characters() )
    Ocrad::internal_error( "delete_character, index out of bounds" );
  data.erase( data.begin() + i );
  }


const Character & Textline::shift_character( const Character & c ) throw()
  {
  int i;

  for( i = characters(); i > 0; --i )
    if( c.left() >= data[i-1].left() ) break;
  data.insert( data.begin() + i, c );
  return data[i];
  }


void Textline::swap_characters( int i, int j ) throw()
  {
  if( i < 0 || i >= characters() || j < 0 || j >= characters() )
    Ocrad::internal_error( "swap_characters, index out of bounds" );
  Character tmp = data[i]; data[i] = data[j]; data[j] = tmp;
  }


Character & Textline::character( int i ) const throw()
  {
  if( i < 0 || i >= characters() )
    Ocrad::internal_error( "character, index out of bounds" );
  return data[i];
  }


int Textline::mean_height() const throw()
  {
  int sum = 0;

  if( characters() == 0 ) return sum;
  for( int i = 0; i < characters(); ++i ) sum += data[i].height();
  return sum / characters();
  }


int Textline::mean_height1() const throw()
  {
  int sum = 0;

  if( characters() <= 1 ) return sum;
  for( int i = 1; i < characters(); ++i ) sum += data[i].height();
  return sum / ( characters() - 1 );
  }


int Textline::mean_width() const throw()
  {
  int sum = 0;

  if( characters() == 0 ) return sum;
  for( int i = 0; i < characters(); ++i ) sum += data[i].width();
  return sum / characters();
  }


int Textline::mean_gap_width() const throw()
  {
  int sum = 0;

  if( characters() < 2 ) return sum;
  for( int i = 1; i < characters(); ++i )
    if( data[i].left() > data[i-1].right() )
      sum += data[i].left() - data[i-1].right() - 1;
  return sum / ( characters() - 1 );
  }


int Textline::mean_hcenter() const throw()
  {
  int sum = 0;

  if( characters() == 0 ) return sum;
  for( int i = 0; i < characters(); ++i ) sum += data[i].hcenter();
  return sum / characters();
  }


int Textline::mean_vcenter() const throw()
  {
  int sum = 0;

  if( characters() == 0 ) return sum;
  for( int i = 0; i < characters(); ++i ) sum += data[i].vcenter();
  return sum / characters();
  }


void Textline::join( Textline & l ) throw()
  {
  for( int i = 0; i < l.characters(); ++i )
    shift_character( l.data[i] );
  l.data.clear();
  }


void Textline::print( const Control & control ) const throw()
  {
  for( int i = 0; i < characters(); ++i ) character( i ).print( control );
  std::fputs( "\n", control.outfile );
  }


void Textline::dprint( const Control & control, bool graph, bool recursive )
								const throw()
  {
  if( graph || recursive )
    std::fprintf( control.outfile, "mean_height = %d\n", mean_height() );

  for( int i = 0; i < characters(); ++i )
    character( i ).dprint( control, graph, recursive );
  std::fputs( "\n", control.outfile );
  }


void Textline::xprint( const Control & control ) const throw()
  {
  for( int i = 0; i < characters(); ++i )
    character( i ).xprint( control );
  }


void Textline::recognize1( const Charset & charset ) const throw()
  {
  if( characters() == 0 ) return;
  int mh = mean_height(), l, r;

  for( l = 0; l < characters() / 2; ++l )
    if( Ocrad::similar( character( l ).height(), mh, 20 ) ) break;
  for( r = characters() - 1; r > characters() / 2; --r )
    if( Ocrad::similar( character( r ).height(), mh, 20 ) ) break;
  if( r - l < characters() / 2 )
    {
    for( l = 0; l < characters() / 2; ++l )
      if( character( l ).height() > mh ) break;
    for( r = characters() - 1; r > characters() / 2; --r )
      if( character( r ).height() > mh ) break;
    if( r - l < characters() / 2 ) { l = 0; r = characters() - 1; }
    }

  int xl = character( l ).hcenter(), yl = character( l ).vcenter();
  int xr = character( r ).hcenter(), yr = character( r ).vcenter();
  int dx = xr - xl, dy = yr - yl;
  for( int i = 0; i < characters(); ++i )
    {
    Character & c = character( i );
    int charbox_vcenter = yl;
    if( dx )
      charbox_vcenter += ( dy * ( c.hcenter() - xl ) / dx );
    c.recognize1( charset, charbox_vcenter );
    }
  }
