/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstddef>
#include <cstdio>
#include "common.h"
#include "rectangle.h"


Rectangle::Rectangle( int l, int t, int r, int b ) throw()
  {
  if( r < l || b < t )
    {
    std::fprintf( stderr, "l = %d, t = %d, r = %d, b = %d\n", l, t, r, b );
    Ocrad::internal_error( "bad parameter building a Rectangle" );
    }
  _left = l; _top = t; _right = r; _bottom = b;
  }


void Rectangle::left( int l ) throw()
  {
  if( l > _right )
    Ocrad::internal_error( "left, bad parameter resizing a Rectangle" );
  _left = l;
  }


void Rectangle::top( int t ) throw()
  {
  if( t > _bottom )
    Ocrad::internal_error( "top, bad parameter resizing a Rectangle" );
  _top = t;
  }


void Rectangle::right( int r ) throw()
  {
  if( r < _left )
    Ocrad::internal_error( "right, bad parameter resizing a Rectangle" );
  _right = r;
  }


void Rectangle::bottom( int b ) throw()
  {
  if( b < _top )
    Ocrad::internal_error( "bottom, bad parameter resizing a Rectangle" );
  _bottom = b;
  }


int Rectangle::hpos( int p ) const throw()
  { return ( ( ( _right - _left ) * p ) / 100 ) + _left; }


int Rectangle::vpos( int p ) const throw()
  { return ( ( ( _bottom - _top ) * p ) / 100 ) + _top; }


void Rectangle::add_point( int row, int col ) throw()
  {
  if( row > _bottom ) _bottom = row; else if( row < _top ) _top = row;
  if( col > _right ) _right = col;   else if( col < _left ) _left = col;
  }


void Rectangle::add_rectangle( const Rectangle & r ) throw()
  {
  if( r._left < _left ) _left = r._left;
  if( r._top < _top ) _top = r._top;
  if( r._right > _right ) _right = r._right;
  if( r._bottom > _bottom ) _bottom = r._bottom;
  }


bool Rectangle::includes( const Rectangle & r ) const throw()
  {
  return ( _left  <= r._left  && _top    <= r._top &&
           _right >= r._right && _bottom >= r._bottom );
  }


bool Rectangle::includes( int row, int col ) const throw()
  {
  return ( _left <= col && _right >= col && _top <= row && _bottom >= row );
  }


bool Rectangle::strictly_includes( const Rectangle & r ) const throw()
  {
  return ( _left  < r._left  && _top    < r._top &&
           _right > r._right && _bottom > r._bottom );
  }


bool Rectangle::includes_hcenter( const Rectangle & r ) const throw()
  { return ( _left <= r.hcenter() && _right >= r.hcenter() ); }


bool Rectangle::includes_vcenter( const Rectangle & r ) const throw()
  { return ( _top <= r.vcenter() && _bottom >= r.vcenter() ); }


bool Rectangle::h_includes( const Rectangle & r ) const throw()
  { return ( _left <= r._left && _right >= r._right ); }


bool Rectangle::v_includes( const Rectangle & r ) const throw()
  { return ( _top <= r._top && _bottom >= r._bottom ); }


bool Rectangle::h_includes( int col ) const throw()
  { return ( _left <= col && _right >= col ); }


bool Rectangle::v_includes( int row ) const throw()
  { return ( _top <= row && _bottom >= row ); }


bool Rectangle::h_overlaps( const Rectangle & r ) const throw()
  { return ( _left <= r.right() && _right >= r.left() ); }


bool Rectangle::v_overlaps( const Rectangle & r ) const throw()
  { return ( _top <= r.bottom() && _bottom >= r.top() ); }

/*
bool Rectangle::precedes( const Rectangle & r ) const throw()
  {
  if( v_overlaps( r ) ) return ( _left < r._left );
  if( h_overlaps( r ) ) return ( _top < r._top );
  return ( _left < r._left && _top < r._top );
  }
*/
