/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "blockmap.h"
#include "block.h"


// Return the area of this block only (no recursive)
int Block::area() const throw()
  {
  int area = 0;

  for( int row = top(); row <= bottom(); ++row )
    for( int col = left(); col <= right(); ++col )
      if( _blockmap->id( row, col ) == _id ) ++area;

  return area;
  }


int Block::id( int row, int col ) const throw()
  { return _blockmap->id( row, col ); }


bool Block::compare_id( int id, bool recursive ) const throw()
  {
  if( id == _id ) return true;
  if( recursive )
    {
    std::list< Block >::const_iterator p = _block_list.begin();
    for( ; p != _block_list.end(); ++p )
      if( p->compare_id( id, recursive ) ) return true;
    }
  return false;
  }


bool Block::escape_left( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == _id ) return false;
  bool array[height()];

  for( int r = row; r > top(); --r )
    {
    if( bm.id( r, col ) != _id ) array[r-top()] = true;
    else { for( int i = r - top(); i >= 0; --i ) array[i] = false; break; }
    }
  for( int r = row + 1; r < bottom(); ++r )
    {
    if( bm.id( r, col ) != _id ) array[r-top()] = true;
    else { for( int i = r - top(); i < height(); ++i ) array[i] = false; break; }
    }

  while( --col >= left() )
    {
    bool alive = false;
    for( int i = 0; i < height(); ++i ) if( array[i] )
      { if( bm.id( top() + i, col ) == _id ) array[i] = false;
      else alive = true; }
    if( !alive ) return false;

    for( int i = 2; i < height() - 1; ++i )
      if( array[i-1] && !array[i] &&
          bm.id( top() + i, col ) != _id ) array[i] = true;
    for( int i = height() - 3; i > 0; --i )
      if( array[i+1] && !array[i] &&
          bm.id( top() + i, col ) != _id ) array[i] = true;
    }
  return true;
  }


bool Block::escape_top( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == _id ) return false;
  int l, r;

  for( l = col; l > left() + 1; --l )
    if( bm.id( row, l - 1 ) == _id ) break;
  for( r = col; r < right() - 1; ++r )
    if( bm.id( row, r + 1 ) == _id ) break;
  while( l <= r && --row >= top() )
    {
    while( l <= r && bm.id( row, l ) == _id ) ++l;
    while( l <= r && bm.id( row, r ) == _id ) --r;
    }
  return ( row < top() );
  }


bool Block::escape_right( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == _id ) return false;
  bool array[height()];

  for( int r = row; r > top(); --r )
    {
    if( bm.id( r, col ) != _id ) array[r-top()] = true;
    else { for( int i = r - top(); i >= 0; --i ) array[i] = false; break; }
    }
  for( int r = row + 1; r < bottom(); ++r )
    {
    if( bm.id( r, col ) != _id ) array[r-top()] = true;
    else { for( int i = r - top(); i < height(); ++i ) array[i] = false; break; }
    }

  while( ++col <= right() )
    {
    bool alive = false;
    for( int i = 0; i < height(); ++i ) if( array[i] )
      { if( bm.id( top() + i, col ) == _id ) array[i] = false;
      else alive = true; }
    if( !alive ) return false;

    for( int i = 2; i < height() - 1; ++i )
      if( array[i-1] && !array[i] &&
          bm.id( top() + i, col ) != _id ) array[i] = true;
    for( int i = height() - 3; i > 0; --i )
      if( array[i+1] && !array[i] &&
          bm.id( top() + i, col ) != _id ) array[i] = true;
    }
  return true;
  }


bool Block::escape_bottom( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == _id ) return false;
  int l, r;

  for( l = col; l > left(); --l )
    if( bm.id( row, l - 1 ) == _id ) break;
  for( r = col; r < right(); ++r )
    if( bm.id( row, r + 1 ) == _id ) break;
  while( l <= r && ++row <= bottom() )
    {
    while( l <= r && bm.id( row, l ) == _id ) ++l;
    while( l <= r && bm.id( row, r ) == _id ) --r;
    }
  return ( row > bottom() );
  }


int Block::follow_top( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) != _id ) return row;
  bool array[width()];

  for( int c = col; c >= left(); --c )
    {
    if( bm.id( row, c ) == _id ) array[c-left()] = true;
    else { for( c -= left(); c >= 0; --c ) array[c] = false; break; }
    }
  for( int c = col + 1; c <= right(); ++c )
    {
    if( bm.id( row, c ) == _id ) array[c-left()] = true;
    else { for( c -= left(); c < width(); ++c ) array[c] = false; break; }
    }

  while( --row >= top() )
    {
    bool alive = false;
    for( int i = 0; i < width(); ++i ) if( array[i] )
      { if( bm.id( row, left() + i ) != _id ) array[i] = false;
      else alive = true; }
    if( !alive ) break;

    for( int i = 1; i < width(); ++i )
      if( array[i-1] && !array[i] &&
          bm.id( row, left() + i ) == _id ) array[i] = true;
    for( int i = width() - 2; i >= 0; --i )
      if( array[i+1] && !array[i] &&
          bm.id( row, left() + i ) == _id ) array[i] = true;
    }
  return row + 1;
  }


int Block::follow_bottom( int row, int col ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) != _id ) return row;
  bool array[width()];

  for( int c = col; c >= left(); --c )
    {
    if( bm.id( row, c ) == _id ) array[c-left()] = true;
    else { for( c -= left(); c >= 0; --c ) array[c] = false; break; }
    }
  for( int c = col + 1; c <= right(); ++c )
    {
    if( bm.id( row, c ) == _id ) array[c-left()] = true;
    else { for( c -= left(); c < width(); ++c ) array[c] = false; break; }
    }

  while( ++row <= bottom() )
    {
    bool alive = false;
    for( int i = 0; i < width(); ++i ) if( array[i] )
      { if( bm.id( row, left() + i ) != _id ) array[i] = false;
      else alive = true; }
    if( !alive ) break;

    for( int i = 1; i < width(); ++i )
      if( array[i-1] && !array[i] &&
          bm.id( row, left() + i ) == _id ) array[i] = true;
    for( int i = width() - 2; i >= 0; --i )
      if( array[i+1] && !array[i] &&
          bm.id( row, left() + i ) == _id ) array[i] = true;
    }
  return row - 1;
  }

/*
bool Block::seek_around( int *row, int *col, bool black ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  if( ( bm.id( *row, *col ) == _id ) == black ) return true;
  bool d[3][3] = { { true,  true, true },
                   { true, false, true },
                   { true,  true, true } };
  for( int i = 0 ; i < 3; ++i )
    for( int j = 0 ; j < 3; ++j )
      if( d[i][j] )
        if( ( bm.id( *row + i - 1, *col + i - 1 ) == _id ) != black )
          d[i][j] = false;

  if( d[0][1] ) { --(*row); return true; }
  if( d[2][1] ) { ++(*row); return true; }
  if( d[1][0] ) { --(*col); return true; }
  if( d[1][2] ) { ++(*row); return true; }
  if( d[0][0] ) { --(*row); --(*col); return true; }
  if( d[0][2] ) { --(*row); ++(*col); return true; }
  if( d[2][0] ) { ++(*row); --(*col); return true; }
  if( d[2][2] ) { ++(*row); ++(*col); return true; }
  return false;
  }
*/

int Block::seek_left( int row, int col, bool black ) const throw()
  {
  const Blockmap & bm = *_blockmap;

  for( ; col > left(); --col )
    if( ( bm.id( row, col - 1 ) == _id ) == black ) break;
  return col;
  }


int Block::seek_top( int row, int col, bool black ) const throw()
  {
  const Blockmap & bm = *_blockmap;

  for( ; row > top(); --row )
    if( ( bm.id( row - 1, col ) == _id ) == black ) break;
  return row;
  }


int Block::seek_right( int row, int col, bool black ) const throw()
  {
  const Blockmap & bm = *_blockmap;

  for( ; col < right(); ++col )
    if( ( bm.id( row, col + 1 ) == _id ) == black ) break;
  return col;
  }


int Block::seek_bottom( int row, int col, bool black ) const throw()
  {
  const Blockmap & bm = *_blockmap;

  for( ; row < bottom(); ++row )
    if( ( bm.id( row + 1, col ) == _id ) == black ) break;
  return row;
  }


// Looks for an inverted-U-shaped curve near the top, then tests wich of
// the vertical bars goes deeper
bool Block::top_hook( int *hdiff ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  int row, lcol = 0, rcol = 0, black_section = 0;

  for( row = top() + 1; row < vcenter(); ++row )
    {
    bool prev_black = false;
    black_section = 0;
    for( int col = left(); col <= right(); ++col )
      {
      bool black = ( bm.id( row, col ) == _id );
      if( black ) { if( !prev_black && ++black_section == 2 ) rcol = col; }
      else if( prev_black && black_section == 1 ) lcol = col - 1;
      prev_black = black;
      }
    if( black_section >= 2 ) break;
    }

  if( black_section != 2 ) return false;
  if( escape_top( row, lcol + 1 ) ) return false;
  *hdiff = follow_bottom( row, lcol ) - follow_bottom( row, rcol );
  return true;
  }


// Looks for an U-shaped curve near the bottom, then tests wich of
// the vertical bars is taller
bool Block::bottom_hook( int *hdiff ) const throw()
  {
  const Blockmap & bm = *_blockmap;
  int row, lcol = 0, rcol = 0, black_section = 0;

  for( row = bottom() - 1; row > vcenter(); --row )
    {
    bool prev_black = false;
    black_section = 0;
    for( int col = left(); col <= right(); ++col )
      {
      bool black = ( bm.id( row, col ) == _id );
      if( black ) { if( !prev_black && ++black_section == 2 ) rcol = col; }
      else if( prev_black && black_section == 1 ) lcol = col - 1;
      prev_black = black;
      }
    if( black_section >= 2 ) break;
    }

  if( black_section != 2 ) return false;
  if( escape_bottom( row, lcol + 1 ) ) return false;
  *hdiff = follow_top( row, lcol ) - follow_top( row, rcol );
  return true;
  }


void Block::print( FILE * outfile, int sp ) const throw()
  {
  char ch = _id > 0 ? 'O' : '-';
  for( int row = top(); row <= bottom(); ++row)
    {
    for( int i = 0; i < sp; ++i ) fputc( ' ', outfile );
    for( int col = left(); col <= right(); ++col)
      {
      if( _blockmap->id( row, col ) == _id )
        fprintf( outfile, " %c", ch );
      else fprintf( outfile, " " );
      }
    fprintf( outfile, "\n" );
    }
  fprintf( outfile, "\n" );

  if( sp >= 0 )
    {
    std::list< Block >::const_iterator p = _block_list.begin();
    for( ; p != _block_list.end(); ++p )
      p->print( outfile, sp + 8 );
    }
  }
