/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <algorithm>
#include <cctype>
#include <cstdio>
#include <stack>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "vrhomboid.h"
#include "track.h"
#include "bitmap.h"


namespace {

char pbm_getrawbyte( FILE * f ) throw( Bitmap::Error )
  {
  int ch = std::fgetc( f );

  if( ch == EOF )
    throw Bitmap::Error( "end-of-file reading pbm file.\n" );

  return static_cast< char > (ch);
  }


char pbm_getc( FILE * f ) throw()
  {
  char ch;
  bool comment = false;

  do {
    ch = pbm_getrawbyte( f );
    if( ch == '#' ) comment = true;
    else if( ch == '\n' ) comment = false;
    }
  while( comment );
  return ch;
  }


int pbm_getint( FILE * f ) throw( Bitmap::Error )
  {
  char ch;
  int i = 0;

  do ch = pbm_getc( f ); while( std::isspace( ch ) );
  if( !std::isdigit( ch ) )
    throw Bitmap::Error( "junk in pbm file where an integer should be.\n" );
  do { i = (i * 10) + (ch - '0'); ch = pbm_getc( f ); }
  while( std::isdigit( ch ) );
  return i;
  }


bool pbm_getbit( FILE * f ) throw( Bitmap::Error )
  {
  char ch;

  do ch = pbm_getc( f ); while( std::isspace( ch ) );

  if( ch == '0' ) return false;
  if( ch == '1' ) return true;
  throw Bitmap::Error( "junk in pbm file where bits should be.\n" );
  }


void mirror_left_right( std::vector< std::vector< bool > > data ) throw()
  {
  int height = data.size();
  for( int row = 0; row < height; ++row )
    reverse( data[row].begin(), data[row].end() );
  }


void mirror_top_bottom( std::vector< std::vector< bool > > data ) throw()
  {
  for( int u = 0, d = data.size() - 1; u < d; ++u, --d )
    swap( data[u], data[d] );
  }

} // end namespace


int Bitmap::find_columns( int rindex ) throw()
  {
  const int colmin = ( _width > 100 ) ? _width / 10 : 10;
  const int gapmin = ( _width > 200 ) ? _width / 100 : 2;
  int columns = 0;
  const Rectangle & r = _rectangle_vector[rindex];
  std::vector< int > h_outline;
  h_outline.resize( r.width() );

  int total_dots = 0;
  for( int col = 0; col < r.width(); ++col )
    {
    h_outline[col] = 0;
    for( int row = 0; row < r.height(); ++row )
      if( data[r.top()+row][r.left()+col] ) ++h_outline[col];
    total_dots += h_outline[col];
    }
  if( 10 * total_dots > 8 * r.height() * r.width() )	// eliminates images
    { _rectangle_vector.erase( _rectangle_vector.begin() + rindex );
    return true; }

  int threshold_col = ( total_dots / ( r.width() * 4 ) ) + 6;
  int threshold_gap = ( total_dots / ( r.width() * 20 ) ) + 1;

  int left, right, right_end, col_width, gap_width;
  for( left = 0; left < r.width(); ++left )	// cut left border
    if( h_outline[left] >= threshold_gap ) break;
  if( left < _width / 10 )			// cut left border noise
    {
    int left2 = left + gapmin;
    for( ; left2 < r.width(); ++left2 )
      if( h_outline[left2] >= threshold_gap ) break;
    if( left2 > left + ( 2 * gapmin ) ) left = left2;
    }
  right = left + 1;
  left = ( left > gapmin ) ? left - gapmin : 0;

  for( right_end = r.width(); right_end > left; --right_end ) // cut right border
    if( h_outline[right_end-1] >= threshold_gap ) break;
  right_end = ( right_end + gapmin < r.width() ) ? right_end + gapmin : r.width();

  while( true )
    {
    const Rectangle & r = _rectangle_vector[rindex];
    for( col_width = 0; right < right_end; ++right )  // look for text
      {
      if( h_outline[right] > threshold_col ) ++col_width;
      else if( col_width < colmin ) col_width = 0;
      else break;
      }
    for( gap_width = 0; right < right_end; ++right )  // look for gap
      {
      if( h_outline[right] < threshold_gap ) ++gap_width;
      else if( gap_width < gapmin ) gap_width = 0;
      else break;
      }
    if( right >= right_end )
      {
      if( columns == 0 ) break;
      ++columns;
      Rectangle r1( r.left() + left, r.top(), r.right(), r.bottom() );
      _rectangle_vector.insert( _rectangle_vector.begin() + rindex + columns, r1 );
      break;
      }
    if( right >= right_end - colmin && columns == 0 ) break;
    ++columns;
    Rectangle r1( r.left() + left, r.top(), r.left() + right - ( gap_width / 2 ), r.bottom() );
    _rectangle_vector.insert( _rectangle_vector.begin() + rindex + columns, r1 );
    left = right - ( gap_width / 2 );
    }
  if( columns > 0 )
    _rectangle_vector.erase( _rectangle_vector.begin() + rindex );
  return columns;
  }


bool Bitmap::find_rows_and_columns() throw()
  {
  const int rowmin = ( _height > 100 ) ? _height / 10 : 10;
  const int gapmin = ( _height > 200 ) ? _height / 100 : 2;
  bool split = false;
  for( int rindex = 0; rindex < rectangles(); )
    {
    Rectangle & r = _rectangle_vector[rindex];
    if( r.height() < ( 2 * rowmin ) + gapmin )
      {
      int c = find_columns( rindex );
      if( c > 0 ) { rindex += c; split = true; } else ++rindex;
      continue;
      }
    std::vector< int > v_outline;
    v_outline.resize( r.height() );

    int total_dots = 0;
    for( int row = 0; row < r.height(); ++row )
      {
      v_outline[row] = 0;
      for( int col = 0; col < r.width(); ++col )
        if( data[r.top()+row][r.left()+col] ) ++v_outline[row];
      total_dots += v_outline[row];
      }
    if( 10 * total_dots > 8 * r.height() * r.width() )	// eliminates images
      { _rectangle_vector.erase( _rectangle_vector.begin() + rindex );
      continue; }

//    int threshold_row = ( total_dots / ( r.height() * 4 ) ) + 6;
    int threshold_gap = ( total_dots / ( r.height() * 20 ) ) + 1;

    int top, bottom;
    for( top = 0; top < r.height(); ++top )	// cut top border
      if( v_outline[top] > threshold_gap ) break;
    for( bottom = r.height() - 1; bottom > top; --bottom ) // cut bottom border
      if( v_outline[bottom] > threshold_gap ) break;
    if( bottom - top < 2 * rowmin )
      {
      int c = find_columns( rindex );
      if( c > 0 ) { rindex += c; split = true; } else ++rindex;
      continue;
      }

    int t, b;
    for( t = b = top + gapmin; t < bottom - gapmin; ++t )
      if( v_outline[t] < threshold_gap )	// look for top of gap
        {
        for( b = t + 1; b < bottom; ++b )	// look for bottom of gap
          if( v_outline[b] > threshold_gap ) break;
        if( b - t >= gapmin ) break;
        t = b;
        }
    if( b - t >= gapmin )
      {
      // cut by a minimum near the center
      int mid = ( b + t ) / 2, half = ( b - t ) / 2, imin = mid;
      for( int i = 0; i <= half && v_outline[imin] > 0; ++i )
        {
        if( v_outline[mid+i] < v_outline[imin] ) imin = mid + i;
        if( v_outline[mid-i] < v_outline[imin] ) imin = mid - i;
        }
      Rectangle r1( r.left(), r.top() + imin, r.right(), r.bottom() );
      r.bottom( r.top() + imin );
      _rectangle_vector.insert( _rectangle_vector.begin() + rindex + 1, r1 );
      if( find_columns( rindex ) == 0 ) ++rindex;
      split = true;
      }
    else if( find_columns( rindex ) > 0 ) split = true;
    else ++rindex;
    }
  return split;
  }


void Bitmap::mirror_diagonal() throw()
  {
  int size = std::max( _width, _height );

  if( _height < size )
    {
    data.resize( size );
    for( int row = _height; row < size; ++row )
      data[row].resize( size );
    }
  else if( _width < size )
    for( int row = 0; row < _height; ++row )
      data[row].resize( size );

  for( int row = 0; row < size; ++row )
    for( int col = 0; col < row; ++col )
      {
      bool b = data[row][col];
      data[row][col] = data[col][row]; data[col][row] = b;
      }

  int tmp = _width; _width = _height; _height = tmp;
  if( _height < size ) data.resize( _height );
  else if( _width < size )
    for( int row = 0; row < _height; ++row )
      data[row].resize( _width );
  }


void Bitmap::floodfill4( const Bitmap & source, Bitmap & dest,
                         int row, int col ) throw()
  {
  std::stack< int, std::vector< int > > st;

  if( source.data[row][col] == false && dest.data[row][col] == false )
    { dest.data[row][col] = true; st.push(row); st.push(col); }

  while( !st.empty() )
    {
    col = st.top(); st.pop();
    row = st.top(); st.pop();
    if( col > 0 && !source.data[row][col-1] && !dest.data[row][col-1] )
      { dest.data[row][col-1] = true; st.push(row); st.push(col-1); }
    if( col < dest._width-1 && !source.data[row][col+1] && !dest.data[row][col+1] )
      { dest.data[row][col+1] = true; st.push(row); st.push(col+1); }
    if( row > 0 && !source.data[row-1][col] && !dest.data[row-1][col] )
      { dest.data[row-1][col] = true; st.push(row-1); st.push(col); }
    if( row < dest._height-1 && !source.data[row+1][col] && !dest.data[row+1][col] )
      { dest.data[row+1][col] = true; st.push(row+1); st.push(col); }
    }
  }


Bitmap::Bitmap( FILE * f, bool invert ) throw( Error )
  {
  char filetype = 0;

  if( pbm_getrawbyte( f ) == 'P' )
    {
    char ch = pbm_getrawbyte( f );
    if( ch == '1' || ch == '4' ) filetype = ch;
    }
  if( filetype == 0 )
    throw Error( "bad magic number - not a pbm file.\n" );

  _width = pbm_getint( f );
  _height = pbm_getint( f );

  data.resize( _height );

  if( filetype == '1' )
    {
    for( int row = 0; row < _height; ++row )
      {
      data[row].resize( _width );
      if( invert )
        for( int col = 0; col < _width; ++col )
          data[row][col] = !pbm_getbit( f );
      else
        for( int col = 0; col < _width; ++col )
          data[row][col] = pbm_getbit( f );
      }
    }
  else
    {
    for( int row = 0; row < _height; ++row )
      {
      data[row].resize( _width );
      unsigned char byte = 0, mask = 0;
      if( invert )
        for( int col = 0; col < _width; ++col )
          {
          if( mask == 0 ) { mask = 0x80; byte = pbm_getrawbyte( f ); }
          data[row][col] = !( byte & mask );
          mask >>= 1;
          }
      else
        for( int col = 0; col < _width; ++col )
          {
          if( mask == 0 ) { mask = 0x80; byte = pbm_getrawbyte( f ); }
          data[row][col] = ( byte & mask );
          mask >>= 1;
          }
      }
    }
  }


Bitmap::Bitmap( const Bitmap & source, const Rectangle & r ) throw()
  {
  if( r.left() < 0 || r.top() < 0 ||
      r.right() >= source._width || r.bottom() >= source._height )
    Ocrad::internal_error( "bad parameter building a Bitmap from another one" );

  _height = r.height(); _width = r.width();

  data.resize( _height );
  for( int row = 0; row < _height; ++row )
    {
    data[row].resize( _width );

    for( int col = 0; col < _width; ++col )
      data[row][col] = source.data[r.top()+row][r.left()+col];
    }
  }


Bitmap::Bitmap( const Bitmap & source, int scale, int th ) throw()
  {
  if( scale < 2 || scale > source._width || scale > source._height ||
      th >= scale * scale )
    Ocrad::internal_error( "bad parameter building a reduced Bitmap" );

  if( th < 0 )
    {
    long dots = 0;
    for( int row = 0; row < source._height; ++row )
      for( int col = 0; col < source._width; ++col )
        if( source.data[row][col] ) ++dots;
    th = ( (scale * scale * dots) - 1 ) / (source._width * source._height);
    std::fprintf( stderr, "automatic threshold = %d\n", th );
    }

  _height = source._height / scale;
  _width = source._width / scale;

  data.resize( _height );
  for( int row = 0; row < _height; ++row )
    {
    int srow = (row * scale) + scale;
    data[row].resize( _width );
    for( int col = 0; col < _width; ++col )
      {
      int scol = (col * scale) + scale;
      int counter = 0;
      for( int i = srow - scale; i < srow; ++i )
        for( int j = scol - scale; j < scol; ++j )
          if( source.data[i][j] && ++counter > th ) goto L1;
      L1: data[row][col] = ( counter > th );
      }
    }
  }


Bitmap::Bitmap( const Bitmap & source, Type t ) throw( Error )
  {
  _height = source._height; _width = source._width;

  data.resize( _height );
  for( int row = 0; row < _height; ++row ) data[row].resize( _width );

  if( t == vertical_histogram )
    {
    for( int col = 0; col < _width; ++col )
      for( int row = 0, y = _height; row < _height; ++row )
        if( source.data[row][col] ) data[--y][col] = true;
    }
  else if( t == horizontal_histogram )
    {
    for( int row = 0; row < _height; ++row )
      for( int col = 0, x = 0; col < _width; ++col )
        if( source.data[row][col] ) data[row][x++] = true;
    }
  else if( t == connected_ground )
    {
    int row, col;
    for( row = 0; row < _height; ++row )
      {
      for( col = 0; col < _width; ++col )
        if( source.data[row][col] ) break;
      if( col == _width ) break;
      }
    if( row == _height ) throw Error( "image is not clear enough.\n" );
    floodfill4( source, *this, row, _width/2 );
    }
  else Ocrad::internal_error( "type not implemented building a 'special' Bitmap" );
  }


Bitmap::Bitmap( const std::vector< std::vector< int > > & matrix ) throw()
  {
  _height = matrix.size();
  if( _height == 0 || ( _width = matrix[0].size() ) == 0 )
    Ocrad::internal_error( "bad size building a Bitmap from an int matrix" );

  data.resize( _height );
  for( int row = 0; row < _height; ++row )
    {
    data[row].resize( _width );
    const std::vector< int > & v1 = matrix[row];
    std::vector< bool > & v2 = data[row];
    for( int col = 0; col < _width; ++col ) v2[col] = ( v1[col] > 0 );
    }
  }


void Bitmap::draw_rectangle( const Rectangle & re ) throw()
  {
  int l = std::max( 0, re.left() );
  int t = std::max( 0, re.top() );
  int r = std::min( _width - 1, re.right() );
  int b = std::min( _height - 1, re.bottom() );
  if( l == re.left() ) for( int row = t; row <= b; ++row ) data[row][l] = true;
  if( t == re.top() ) for( int col = l; col <= r; ++col ) data[t][col] = true;
  if( r == re.right() ) for( int row = t; row <= b; ++row ) data[row][r] = true;
  if( b == re.bottom() ) for( int col = l; col <= r; ++col ) data[b][col] = true;
  }


void Bitmap::draw_track( const Track & tr ) throw()
  {
  int l = std::max( 0, tr.left() );
  int r = std::min( _width - 1, tr.right() );
  if( l == r ) return;
  if( l == tr.left() )
    for( int row = tr.top(l); row <= tr.bottom(l); ++row )
      if( row >= 0 && row < _height ) data[row][l] = true;
  if( r == tr.right() )
    for( int row = tr.top(r); row <= tr.bottom(r); ++row )
      if( row >= 0 && row < _height ) data[row][r] = true;
  for( int col = l; col <= r; ++col )
    {
    int row = tr.top( col );
    if( row >= 0 && row < _height ) data[row][col] = true;
    row = tr.bottom( col );
    if( row >= 0 && row < _height ) data[row][col] = true;
    }
  }


int Bitmap::analyse_layout( int layout_level ) throw()
  {
  if( _rectangle_vector.size() != 0 )
    Ocrad::internal_error( "analyse_layout called twice" );

  if( layout_level > 0 && layout_level <= 2 && _width > 200 && _height > 200 )
    {
    Bitmap reduced( *this, 10, 9 );
    Rectangle r( 0, 0, reduced._width - 1, reduced._height - 1 );
    reduced._rectangle_vector.push_back( r );
    reduced.find_columns( 0 );
    if( layout_level >= 2 ) reduced.find_rows_and_columns();
    if( reduced.rectangles() > 1 )
      for( int i = 0; i < reduced.rectangles(); ++i )
        {
        const Rectangle & r = reduced._rectangle_vector[i];
        Rectangle r1( 10*r.left(), 10*r.top(), 10*r.right(), 10*r.bottom() );
        _rectangle_vector.push_back( r1 );
        }
    }
  if( _rectangle_vector.size() == 0 )
    {
    Rectangle r( 0, 0, _width - 1, _height - 1 );
    _rectangle_vector.push_back( r );
    }
  return _rectangle_vector.size();
  }


bool Bitmap::only_this_rectangle( int i ) throw()
  {
  if( i < 0 || i >= (int)_rectangle_vector.size() ) return false;
  if( _rectangle_vector.size() > 1 )
    {
    Rectangle r = _rectangle_vector[i];
    _rectangle_vector.clear(); _rectangle_vector.push_back( r );
    }
  return true;
  }


void Bitmap::save( FILE * f, char filetype ) const throw()
  {
  if( filetype != '1' && filetype != '4' ) filetype = '4';
  std::fprintf( f, "P%c\n%d %d\n", filetype, _width, _height );

  if( filetype == '1' )
    {
    for( int row = 0; row < _height; ++row )
      {
      for( int col = 0; col < _width; ++col )
        std::putc( data[row][col] ? '1' : '0', f );
      std::putc( '\n', f );
      }
    }
  else	// filetype == '4'
    {
    for( int row = 0; row < _height; ++row )
      {
      unsigned char byte = 0, mask = 0x80;
      for( int col = 0; col < _width; ++col )
        {
        if( data[row][col] ) byte |= mask;
        mask >>= 1;
        if( mask == 0 ) { std::putc( byte, f ); byte = 0; mask = 0x80; }
        }
      if( mask != 0x80 ) std::putc( byte, f ); // incomplete byte at end of row
      }
    }
  }


void Bitmap::transform( const Transformation & t ) throw()
  {
  switch( t.type() )
    {
    case Transformation::none:
      break;
    case Transformation::rotate90:
      mirror_diagonal(); mirror_top_bottom( data ); break;
    case Transformation::rotate180:
      mirror_left_right( data ); mirror_top_bottom( data ); break;
    case Transformation::rotate270:
      mirror_diagonal(); mirror_left_right( data ); break;
    case Transformation::mirror_lr: mirror_left_right( data ); break;
    case Transformation::mirror_tb: mirror_top_bottom( data ); break;
    case Transformation::mirror_d1: mirror_diagonal(); break;
    case Transformation::mirror_d2:
      mirror_diagonal(); mirror_left_right( data ); mirror_top_bottom( data ); break;
    }
  }
