/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "blockmap.h"
#include "profile.h"


Profile::Profile( const Block & b, type t ) throw()
  : _block( &b ), _blockmap( b.blockmap() ), _type(t), valid(false),
    valid_mean(false), valid_range(false), valid_isconcave(false),
    valid_isconvex(false), valid_isflat(false), valid_isflats(false),
    valid_ispit(false), valid_isvpit(false), valid_istip(false),
    valid_isctip(false) {}


void Profile::initialize() throw()
  {
  const Block & b = *_block;
  valid = true;
  switch( _type )
    {
    case left :
      data.resize( b.height() ); limit = b.width();
      for( int i = 0; i < samples(); ++i )
        {
        int j = 0, row = b.top() + i;
        while( j < b.width() && b.blockmap()->id( row, b.left() + j ) != b.id() )
          ++j;
        data[i] = j;
        } break;
    case top :
      data.resize( b.width() ); limit = b.height();
      for( int i = 0; i < samples(); ++i )
        {
        int j = 0, col = b.left() + i;
        while( j < b.height() && b.blockmap()->id( b.top() + j, col ) != b.id() )
          ++j;
        data[i] = j;
        } break;
    case right :
      data.resize( b.height() ); limit = b.width();
      for( int i = 0; i < samples(); ++i )
        {
        int j = 0, row = b.top() + i;
        while( j < b.width() && b.blockmap()->id( row, b.right() - j ) != b.id() )
          ++j;
        data[i] = j;
        } break;
    case bottom :
      data.resize( b.width() ); limit = b.height();
      for( int i = 0; i < samples(); ++i )
        {
        int j = 0, col = b.left() + i;
        while( j < b.height() && b.blockmap()->id( b.bottom() - j, col ) != b.id() )
          ++j;
        data[i] = j;
        } break;
    case height :
      data.resize( b.width() ); limit = b.height();
      for( int i = 0; i < samples(); ++i )
        {
        int j1 = 0, j2 = 0, col = b.left() + i;
        while( j1 < b.height() && b.blockmap()->id( b.top() + j1, col ) != b.id() )
          ++j1;
        while( j2 < b.height() && b.blockmap()->id( b.bottom() - j2, col ) != b.id() )
          ++j2;
        data[i] = (j1 + j2 < b.height()) ? b.height() - j1 - j2 : 0;
        } break;
    case width :
      data.resize( b.height() ); limit = b.width();
      for( int i = 0; i < samples(); ++i )
        {
        int j1 = 0, j2 = 0, row = b.top() + i;
        while( j1 < b.width() && b.blockmap()->id( row, b.left() + j1 ) != b.id() )
          ++j1;
        while( j2 < b.width() && b.blockmap()->id( row, b.right() - j2 ) != b.id() )
          ++j2;
        data[i] = (j1 + j2 < b.width()) ? b.width() - j1 - j2 : 0;
        } break;
    case zheight :
      data.resize( b.width() ); limit = b.height();
      for( int i = 0; i < samples(); ++i )
        {
        int col = b.left() + i;
        data[i] = 0;
        for( int j = 0; j < b.height(); ++j )
          if( b.blockmap()->id( b.top() + j, col ) == b.id() ) ++data[i];
        } break;
    case zwidth :
      data.resize( b.height() ); limit = b.width();
      for( int i = 0; i < samples(); ++i )
        {
        int row = b.top() + i;
        data[i] = 0;
        for( int j = 0; j < b.width(); ++j )
          if( b.blockmap()->id( row, b.left() + j ) == b.id() ) ++data[i];
        } break;
    }
  }


int Profile::mean() throw()
  {
  if( !valid_mean )
    {
    valid_mean = true; if( !valid ) initialize();
    _mean = 0;
    for( int i = 0; i < samples(); ++i ) _mean += data[i];
    if( samples() > 1 ) _mean /= samples();
    }
  return _mean;
  }


int Profile::range() throw()
  {
  if( !valid_range )
    {
    valid_range = true; if( !valid ) initialize();
    _range = 0;
    for( int i = 0; i < samples(); ++i ) if( data[i] > _range )
      _range = data[i];
    }
  return _range;
  }


int Profile::operator[]( int i ) throw()
  {
  if( !valid ) initialize();
  if( i < 0 ) i = 0;
  else if( i >= samples() ) i = samples() - 1;
  return data[i];
  }

/*
bool Profile::equal( Profile & p ) throw()
  {
  if( !valid ) initialize();
  if( !p.valid ) p.initialize();
  if( samples() != p.samples() ) return false;
  for( int i = 0; i < samples(); ++i )
    if( data[i] != p.data[i] ) return false;
  return true;
  }
*/

bool Profile::increasing( int i ) throw()
  {
  if( !valid ) initialize();
  if( i < 0 || i > samples() - 2 || data[i] >= data[samples()-1] )
    return false;
  for( ++i; i < samples(); ++i ) if( data[i] < data[i-1] ) return false;
  return true;
  }


bool Profile::decreasing( int i ) throw()
  {
  if( !valid ) initialize();
  if( i < 0 || i > samples() - 2 || data[i] <= data[samples()-1] )
    return false;
  for( ++i; i < samples(); ++i ) if( data[i] > data[i-1] ) return false;
  return true;
  }

/*
bool Profile::isconcave() throw()
  {
  if( !valid_isconcave )
    {
    valid_isconcave = true; _isconcave = false; if( !valid ) initialize();
    int max = limit, imax = 0;

    for( int i = 0; i < samples(); ++i )
      if( data[i] > max ) { max = data[i]; imax = i; }
    if( imax < samples() / 4 || imax > (samples() * 3) / 4 ) return _isconcave;

    int min = limit;
    for( int i = 1; i < samples(); ++i )
      {
      if( data[i] - data[i-1] < min ) min = data[i] - data[i-1];
      else if( data[i] - data[i-1] > min + 1 ) return _isconcave;
      }
    _isconcave = true;
    }
  return _isconcave;
  }
*/

bool Profile::isconvex() throw()
  {
  if( !valid_isconvex )
    {
    valid_isconvex = true; _isconvex = false; if( !valid ) initialize();
    int min = limit, lmin = limit, rmax = -limit, imin = 0, l = 0, r = 0;

    for( int i = 1; i < samples(); ++i )
      {
      int d = data[i] - data[i-1];
      if( d < lmin ) { lmin = d; l = i; }
      if( d >= rmax ) { rmax = d; r = i; }
      if( data[i] < min ) { min = data[i]; imin = i; }
      }
    if( imin < samples() / 4 || imin > (samples() * 3) / 4 ) return _isconvex;
    if( l >= r ) return _isconvex;
    if( l >= samples() / 4 ) return _isconvex;
    if( r <= (samples() * 3) / 4 ) return _isconvex;

    int dmax = -limit;
    for( int i = l; i <= r; ++i )
      {
      int d = data[i] - data[i-1];
      if( d > dmax ) dmax = d;
      else if( d < dmax - 2 || ( d < dmax - 1 && d != 0 ) ) return _isconvex;
      }
    if( dmax > 0 ) _isconvex = true;
    }
  return _isconvex;
  }


bool Profile::isflat() throw()
  {
  if( !valid_isflat )
    {
    valid_isflat = true; _isflat = false; if( !valid ) initialize();
    int th = ( mean() < 1 ) ? 1 : mean();
    for( int i = 2; i < samples() - 2; ++i )
      if( abs( data[i] - th ) > 1 ) return _isflat;
    _isflat = true;
    }
  return _isflat;
  }


bool Profile::isflats() throw()
  {
  if( !valid_isflats )
    {
    valid_isflats = true; _isflats = false; if( !valid ) initialize();
    int begin = (samples() + 9) / 10;
    int end = (samples() * 9) / 10;
    if( begin >= end ) return _isflats;

    int mean1 = 0;
    for( int i = begin; i < end; ++i ) mean1 += data[i];
    mean1 /= ( end - begin );

    for( int i = 1; i < begin; ++i )
      if( data[i] - mean1 > 1 ) return _isflats;
    for( int i = begin; i < end; ++i )
      if( abs( data[i] - mean1 ) > 1 ) return _isflats;
    for( int i = end; i < samples() - 1; ++i )
      if( data[i] - mean1 > 1 ) return _isflats;
    _isflats = true;
    }
  return _isflats;
  }


bool Profile::ispit() throw()
  {
  if( !valid_ispit )
    {
    valid_ispit = true; _ispit = false; if( !valid ) initialize();
    if( samples() < 5 ) return _ispit;

    int th = ( mean() < 2 && range() > 2 ) ? 2 : mean();
    if( data[0] <= th && data[1] <= th ) return _ispit;
    if( data[samples()-1] <= th && data[samples()-2] <= th ) return _ispit;
    int status = 0;
    for( int i = 2; i < samples() - 2; ++i )
      switch( status )
        {
        case 0: if( data[i] < th ) status = 1; break;
        case 1: if( data[i] > th ) status = 2; break;
        case 2: if( data[i] < th ) return _ispit;
        }
    _ispit = ( status >= 1 );
    }
  return _ispit;
  }


bool Profile::isvpit() throw()
  {
  if( !valid_isvpit )
    {
    valid_isvpit = true; if( !valid ) initialize();
    if( !ispit() ) { _isvpit = false; return _isvpit; }
    int count1 = 0, count2 = 0;
    for( int i = 0; i < samples(); ++i )
      if( data[i] <= 2 ) { if( data[i] < 2 ) ++count1; else ++count2; }
    _isvpit = ( count1 * 4 < samples() ||
                ( count1 * 3 < samples() && count2 <= 2 ) );
    }
  return _isvpit;
  }


bool Profile::istip() throw()
  {
  if( !valid_istip )
    {
    valid_istip = true; _istip = false; if( !valid ) initialize();
    if( samples() < 5 ) return _istip;

    int th = ( mean() < 2 && range() > 2 ) ? 2 : mean();
    if( data[0] >= th && data[1] >= th ) return _istip;
    if( data[samples()-1] >= th && data[samples()-2] >= th ) return _istip;
    int status = 0;
    for( int i = 2; i < samples() - 2; ++i )
      switch( status )
        {
        case 0: if( data[i] > th ) status = 1; break;
        case 1: if( data[i] < th ) status = 2; break;
        case 2: if( data[i] > th ) return _istip;
        }
    _istip = ( status >= 1 );
    }
  return _istip;
  }


bool Profile::isctip() throw()
  {
  if( !valid_isctip )
    {
    valid_isctip = true; _isctip = false; if( !valid ) initialize();
    if( samples() < 5 ) return _isctip;
    int th = ( mean() < 2 ) ? 2 : mean();
    int imax = -1, mid = samples() / 2;

    for( int i = 0; i < samples() / 4; ++i )
      {
      if( data[mid+i] > th ) { imax = mid + i; break; }
      if( data[mid-i-1] > th ) { imax = mid - i - 1; break; }
      }
    if( imax < 0 ) return _isctip;

    for( int i = imax + 1; i < samples(); ++i )
      if( data[i] < th )
        {
        for( int j = imax - 1; j >= 0; --j )
          if( data[j] < th ) { _isctip = true; return _isctip; }
        break;
        }
    }
  return _isctip;
  }


int Profile::imaximum() throw()
  {
  if( !valid ) initialize();
  int mbegin = 0, mend, mvalue = 0;
  for( int i = 0; i < samples(); ++i )
    if( data[i] > mvalue ) { mvalue = data[i]; mbegin = i; }
  for( mend = mbegin + 1; mend < samples(); ++mend )
    if( data[mend] < mvalue ) break;
  return ( mbegin + mend - 1 ) / 2;
  }


int Profile::iminimum() throw()
  {
  if( !valid ) initialize();
  int mbegin = 0, mend, mvalue = limit + 1;
  for( int i = 0; i < samples(); ++i )
    if( data[i] < mvalue ) { mvalue = data[i]; mbegin = i; }
  for( mend = mbegin + 1; mend < samples(); ++mend )
    if( data[mend] > mvalue ) break;
  return ( mbegin + mend - 1 ) / 2;
  }


int Profile::maxima() throw()
  {
  if( !valid ) initialize();
  int maxima = ( data[0] > mean() ) ? 1 : 0;
  for( int i = 1; i < samples(); ++i )
    if( data[i] > mean() && data[i-1] <= mean() ) ++maxima;
  return maxima;
  }


int Profile::minima( int th ) throw()
  {
  if( !valid ) initialize();
  if( samples() < 1 ) return false;
  if( th < 0 ) th = ( mean() < 2 ) ? 2 : mean();
  int minima = ( data[0] < th ) ? 1 : 0;
  int status = ( minima ) ? 1 : 0;

  for( int i = 1; i < samples(); ++i )
    switch( status )
      {
      case 0: if( data[i] < th ) { status = 1; ++minima; } break;
      case 1: if( data[i] > th ) status = 0; break;
      }
  return minima;
  }


bool Profile::straight( int & slope ) throw()
  {
  if( !valid ) initialize();
  if( samples() < 5 ) return false;
  int total_dif = data[samples()-2] - data[1];
  int mean_dif = total_dif / ( samples() - 3 );

  for( int i = 2; i <= samples() - 2; ++i )
    if( abs( mean_dif - ( data[i] - data[i-1] ) ) > 1 ) return false;
  slope = total_dif;
  return true;
  }
