/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "blockmap.h"
#include "profile.h"
#include "features.h"


Features::Features( const Block & b ) throw()
  : _block( &b ), _blockmap( b.blockmap() ), _hbars( -1 ), _vbars( -1 ),
    hscan_valid( false ), vscan_valid( false ), lp( b, Profile::left ),
    tp( b, Profile::top ), rp( b, Profile::right ), bp( b, Profile::bottom ),
    hp( b, Profile::height ), wp( b, Profile::width ),
    zhp( b, Profile::zheight ), zwp( b, Profile::zwidth )
  {}


int Features::hbars() const throw()
  {
  if( _hbars < 0 )
    {
    const Block & b = *_block;
    const Blockmap & bm = *_blockmap;
    int state = 0, begin = 0, l = 0, r = 0, limit = b.width() / 2;
    _hbars = 0;

    for( int row = b.top(); row <= b.bottom(); ++row )
      {
      int maxcount = 0, count = 0, lt = 0;
      int col;
      for( col = b.left(); col <= b.right(); ++col )
        {
        if( bm.id( row, col ) == b.id() ) ++count;
        else
          { if( count > maxcount ) { maxcount = count; lt = col - count; }
          count = 0; }
        }
      if( maxcount > count ) count = maxcount; else lt = col - count;
      switch( state )
        {
        case 0: if( count > limit )
                  { state = 1; begin = row; l = lt; r = l + count - 1; }
                break;
        case 1: if( count < limit )
                  { _hbar.push_back( Rectangle( l, begin, r, row - 1 ) );
                  ++_hbars; state = 0; break; }
                if( lt < l ) l = lt;
                if( l + count - 1 > r ) r = l + count - 1;
                if( row == b.bottom() )
                  { _hbar.push_back( Rectangle( l, begin, r, row ) );
                  ++_hbars; state = 0; break; }
        }
      }
    }
  return _hbars;
  }


int Features::vbars() const throw()
  {
  if( _vbars < 0 )
    {
    const Block & b = *_block;
    const Blockmap & bm = *_blockmap;
    int state = 0, begin = 0, limit = b.height() - 3;
    _vbars = 0;

    for( int col = b.left(); col <= b.right(); ++col )
      {
      int maxcount = 0, count = 0;
      for( int row = b.top() + 1; row < b.bottom(); ++row )
        {
        if( bm.id( row, col ) == b.id() ) ++count;
        else { if( count > maxcount ) maxcount = count; count = 0; }
        }
      if( maxcount > count ) count = maxcount;
      switch( state )
        {
        case 0: if( count >= limit ) { state = 3; begin = col; }
                else if( count * 4 >= limit * 3 ) { state = 2; begin = col; }
                else if( count * 3 >= limit * 2 ) { state = 1; begin = col; }
                break;
        case 1: if( count >= limit ) state = 3;
                else if( count * 4 >= limit * 3 ) state = 2;
                else if( count * 3 < limit * 2 ) state = 0;
                else begin = col;
                break;
        case 2: if( count >= limit ) state = 3;
                else if( count * 3 < limit * 2 ) state = 0;
                else if( count * 4 < limit * 3 ) state = 1;
                break;
        case 3: if( count * 3 < limit * 2 )
                  { _vbar.push_back( Rectangle( begin, b.top(), col - 1, b.bottom() ) );
                  ++_vbars; state = 0; }
                else if( col == b.right() )
                  { _vbar.push_back( Rectangle( begin, b.top(), col, b.bottom() ) );
                  ++_vbars; state = 0; }
        }
      }
    }
  return _vbars;
  }


// return the number of horizontal traces crossing every column
const std::vector< int > & Features::hscan() const throw()
  {
  if( !hscan_valid )
    {
    const Block & b = *_block;
    const Blockmap & bm = *_blockmap;
    int state = 0;
    hscan_valid = true; _hscan.resize( b.width() );

    for( int col = b.left(); col <= b.right(); ++col )
      {
      _hscan[col-b.left()] = 0;
      for( int row = b.top(); row <= b.bottom(); ++row )
        {
        bool black = ( bm.id( row, col ) == b.id() );
        switch( state )
          {
          case 0: if( black ) state = 1; break;
          case 1: if( !black || row == b.bottom() )
                    { state = 0; ++_hscan[col-b.left()]; }
          }
        }
      }
    }
  return _hscan;
  }


// return the number of vertical traces crossing every row
const std::vector< int > & Features::vscan() const throw()
  {
  if( !vscan_valid )
    {
    const Block & b = *_block;
    const Blockmap & bm = *_blockmap;
    int state = 0;
    vscan_valid = true; _vscan.resize( b.height() );

    for( int row = b.top(); row <= b.bottom(); ++row )
      {
      _vscan[row-b.top()] = 0;
      for( int col = b.left(); col <= b.right(); ++col )
        {
        bool black = ( bm.id( row, col ) == b.id() );
        switch( state )
          {
          case 0: if( black ) state = 1; break;
          case 1: if( !black || col == b.right() )
                    { state = 0; ++_vscan[row-b.top()]; }
          }
        }
      }
    }
  return _vscan;
  }


bool Features::escape_left( int row, int col ) const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == b.id() ) return false;
  bool array[b.height()];

  for( int r = row; r > b.top(); --r )
    {
    if( bm.id( r, col ) != b.id() ) array[r-b.top()] = true;
    else { for( int i = r - b.top(); i >= 0; --i ) array[i] = false; break; }
    }
  for( int r = row + 1; r < b.bottom(); ++r )
    {
    if( bm.id( r, col ) != b.id() ) array[r-b.top()] = true;
    else { for( int i = r - b.top(); i < b.height(); ++i ) array[i] = false; break; }
    }

  while( --col >= b.left() )
    {
    bool alive = false;
    for( int i = 0; i < b.height(); ++i ) if( array[i] )
      { if( bm.id( b.top() + i, col ) == b.id() ) array[i] = false;
      else alive = true; }
    if( !alive ) return false;

    for( int i = 2; i < b.height() - 1; ++i )
      if( array[i-1] && !array[i] &&
          bm.id( b.top() + i, col ) != b.id() ) array[i] = true;
    for( int i = b.height() - 3; i > 0; --i )
      if( array[i+1] && !array[i] &&
          bm.id( b.top() + i, col ) != b.id() ) array[i] = true;
    }
  return true;
  }


bool Features::escape_top( int row, int col ) const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == b.id() ) return false;
  int l, r;

  for( l = col; l > b.left(); --l )
    if( bm.id( row, l - 1 ) == b.id() ) break;
  for( r = col; r < b.right(); ++r )
    if( bm.id( row, r + 1 ) == b.id() ) break;
  while( l <= r && --row >= b.top() )
    {
    while( l <= r && bm.id( row, l ) == b.id() ) ++l;
    while( l <= r && bm.id( row, r ) == b.id() ) --r;
    }
  return ( row < b.top() );
  }


bool Features::escape_right( int row, int col ) const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == b.id() ) return false;
  bool array[b.height()];

  for( int r = row; r > b.top(); --r )
    {
    if( bm.id( r, col ) != b.id() ) array[r-b.top()] = true;
    else { for( int i = r - b.top(); i >= 0; --i ) array[i] = false; break; }
    }
  for( int r = row + 1; r < b.bottom(); ++r )
    {
    if( bm.id( r, col ) != b.id() ) array[r-b.top()] = true;
    else { for( int i = r - b.top(); i < b.height(); ++i ) array[i] = false; break; }
    }

  while( ++col <= b.right() )
    {
    bool alive = false;
    for( int i = 0; i < b.height(); ++i ) if( array[i] )
      { if( bm.id( b.top() + i, col ) == b.id() ) array[i] = false;
      else alive = true; }
    if( !alive ) return false;

    for( int i = 2; i < b.height() - 1; ++i )
      if( array[i-1] && !array[i] &&
          bm.id( b.top() + i, col ) != b.id() ) array[i] = true;
    for( int i = b.height() - 3; i > 0; --i )
      if( array[i+1] && !array[i] &&
          bm.id( b.top() + i, col ) != b.id() ) array[i] = true;
    }
  return true;
  }


bool Features::escape_bottom( int row, int col ) const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;
  if( bm.id( row, col ) == b.id() ) return false;
  int l, r;

  for( l = col; l > b.left(); --l )
    if( bm.id( row, l - 1 ) == b.id() ) break;
  for( r = col; r < b.right(); ++r )
    if( bm.id( row, r + 1 ) == b.id() ) break;
  while( l <= r && ++row <= b.bottom() )
    {
    while( l <= r && bm.id( row, l ) == b.id() ) ++l;
    while( l <= r && bm.id( row, r ) == b.id() ) --r;
    }
  return ( row > b.bottom() );
  }


int Features::seek_left( int row, int col ) const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;

  for( ; col > b.left(); --col )
    if( bm.id( row, col - 1 ) == b.id() ) break;
  return col;
  }


int Features::seek_top( int row, int col ) const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;

  for( ; row > b.top(); --row )
    if( bm.id( row - 1, col ) == b.id() ) break;
  return row;
  }


int Features::seek_right( int row, int col ) const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;

  for( ; col < b.right(); ++col )
    if( bm.id( row, col + 1 ) == b.id() ) break;
  return col;
  }


int Features::seek_bottom( int row, int col ) const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;

  for( ; row < b.bottom(); ++row )
    if( bm.id( row + 1, col ) == b.id() ) break;
  return row;
  }


// Looks for three black sections in column hcenter()  1, then tests if
// upper  and lower gaps are open to the right or to the left
char Features::test_235sz() const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;
  int urow2 = 0, lrow1 = 0, lrow2 = 0, col, black_section = 0;

  for( col = b.hcenter() - 1; col <= b.hcenter() + 1; ++col )
    {
    bool prev_black = false;
    black_section = 0;
    for( int row = b.top(); row <= b.bottom(); ++row )
      {
      bool black = ( bm.id( row, col ) == b.id() );
      if( black ) { if( !prev_black && ++black_section == 2 ) urow2 = row - 1; }
      else if( prev_black )
        { if( black_section == 1 ) lrow1 = row;
          else if( black_section == 2 ) lrow2 = row; }
      prev_black = black;
      }
    if( black_section == 3 ) break;
    }

  if( black_section != 3 ) return 0;
  if( escape_left( lrow2, col ) )
    {
    if( escape_left( urow2, col ) )
      {
      if( escape_bottom( lrow2, col ) && escape_top( urow2, col ) ) return'*';
      return '3';
      }
    if( escape_right( urow2, col ) )
      {
      if( !tp.isconvex() && seek_right( lrow1 + 1, col ) >= b.right() )
        return '5';
      if( urow2 > b.vcenter() + 1 && lrow2 > b.bottom() - (b.height() / 5) &&
          seek_right( urow2 - 1, col ) < b.right() )
        return '';
      return 's';
      }
    }
  else if( escape_right( lrow2, col ) )
    {
    if( escape_right( urow2, col ) )
      {
      if( escape_bottom( lrow2, col ) && escape_top( urow2, col ) ) return'*';
      return 'E';
      }
    if( escape_left( urow2, col ) && bp.isflat() )
      {
      if( tp.isflat() ) return 'z';
//      if( hbars() >= 1 && hbar(0).top() <= b.top() + 1 ) return 'z';
      if( tp.isconvex() ) return '2';
      }
    }
  return 0;
  }


// Looks for the nearest frontier in column hcenter(), then tests if
// gap is open downwards
char Features::test_hn() const throw()
  {
  const Block & b = *_block;
  const Blockmap & bm = *_blockmap;
  int row = b.vcenter();

  if( bm.id( row, b.hcenter() ) == b.id() )
    {
    while( ++row <= b.bottom() && bm.id( row, b.hcenter() ) == b.id() );
    if( row > b.bottom() ) return 0;
    }
  else
    {
    while( row > b.top() && bm.id( row - 1, b.hcenter() ) != b.id() )
      --row;
    if( row <= b.top() ) return 0;
    }
  if( !escape_bottom( row, b.hcenter() ) ) return 0;
  if( tp.isconvex() ) return '^';
  if( similar( row - b.top(), b.vcenter() - b.top(), 16 ) )
    { if( tp.minima() == 2 ) return 'W'; else return 'h'; }
  if( similar( b.height(), b.width(), 40 ) && row > b.vcenter() &&
      tp.minima() == 2 ) return 'w';
  if( similar( b.height(), b.width(), 40 ) || bp.range() > b.height() / 2 )
    return 'n';
  if( b.width() > 2 * b.height() ) return '~';
  return 0;
  }


// Recognizes single line, non-rectangular characters without holes.
// '/<>\^`
char Features::test_line() const throw()
  {
  const Block & b = *_block;
  int slope1, slope2;

  if( lp.straight( slope1 ) && rp.straight( slope2 ) )
    {
    if( slope1 < 0 )
      {
      if( slope2 < 0 && bp.minima() == 2 ) return '^';
      if( 4 * b.area() < b.height() * b.width() ) return '/';
      return '\'';
      }
    if( slope1 > 0 && slope2 < 0 )
      {
      if( 4 * b.area() < b.height() * b.width() ) return '\\';
      return '`';
      }
    }
  if( tp.straight( slope1 ) && bp.straight( slope2 ) )
    {
    if( slope1 < 0 && slope2 < 0 ) return '<';
    if( slope1 > 0 && slope2 > 0 ) return '>';
    }
  return 0;
  }
