/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "bitmap.h"
#include "block.h"
#include "blockmap.h"


void Blockmap::create_block( const Rectangle & r, const int id ) throw()
  {
  Block b( r, *this, id );
  _block_list.push_back( b );
  }

void Blockmap::delete_block( const int id ) throw( Internal_error )
  {
  std::list< Block >::reverse_iterator p = _block_list.rbegin();
  for( ; p != _block_list.rend(); ++p )
    if( id == p->id() )
      {
      for( int row = p->top(); row <= p->bottom(); ++row )
        for( int col = p->left(); col <= p->right(); ++col )
          if( data[row][col] == id ) data[row][col] = 0;
      _block_list.erase( --(p.base()) ); return;
      }

  throw Internal_error( "lost block (delete).\n" );
  }

void Blockmap::join_blocks( int id1, int id2 ) throw( Internal_error )
  {
  if( id1 > id2 )
    { int temp = id1; id1 = id2; id2 = temp; }

  std::list< Block >::reverse_iterator p1 = _block_list.rbegin();
  std::list< Block >::reverse_iterator p2 = _block_list.rbegin();
  while( p1 != _block_list.rend() && id1 != p1->id() ) ++p1;
  while( p2 != _block_list.rend() && id2 != p2->id() ) ++p2;
  if( p1 == _block_list.rend() || p2 == _block_list.rend() )
    throw Internal_error( "lost block (join).\n" );

  for( int row = p2->top(); row <= p2->bottom(); ++row )
    for( int col = p2->left(); col <= p2->right(); ++col )
      if( data[row][col] == id2 ) data[row][col] = id1;
  p1->add_point( p2->top(), p2->left() );
  p1->add_point( p2->bottom(), p2->right() );
  _block_list.erase( --(p2.base()) );
  }

void Blockmap::add_point_to_block( const int row, const int col, const int id )
                                                        throw( Internal_error )
  {
  std::list< Block >::reverse_iterator p = _block_list.rbegin();
  for( ; p != _block_list.rend(); ++p )
    if( id == p->id() ) { p->add_point( row, col ); return; }

  throw Internal_error( "lost block (add_point).\n" );
  }


int Blockmap::left_neighbor( const Rectangle & r , const int distance )
							const throw()
  {
  int limit = distance >= r.left() ? 0 : r.left() - distance;

  for( int col = r.left(); col > limit; )
    {
    --col;
    for( int row = r.top(); row <= r.bottom(); ++row )
      if( data[row][col] != 0 ) return data[row][col];
    }
  return 0;
  }

int Blockmap::top_neighbor( const Rectangle & r , const int distance )
								const throw()
  {
  int limit = distance >= r.top() ? 0 : r.top() - distance;

  for( int row = r.top(); row > limit; )
    {
    --row;
    for( int col = r.left(); col <= r.right(); ++col )
      if( data[row][col] != 0 ) return data[row][col];
    }
  return 0;
  }

int Blockmap::right_neighbor( const Rectangle & r , const int distance )
								const throw()
  {
  int limit = r.right() + distance + 1;
  if( limit > _width ) limit = _width;

  for( int col = r.right() + 1; col < limit; ++col )
    for( int row = r.top(); row <= r.bottom(); ++row )
      if( data[row][col] != 0 ) return data[row][col];

  return 0;
  }

int Blockmap::bottom_neighbor( const Rectangle & r , const int distance )
								const throw()
  {
  int limit = r.bottom() + distance + 1;
  if( limit > _height ) limit = _height;

  for( int row = r.bottom() + 1; row < limit; ++row )
    for( int col = r.left(); col <= r.right(); ++col )
      if( data[row][col] != 0 ) return data[row][col];

  return 0;
  }


void delete_small_blocks( std::list< Block > & block_list ) throw()
  {
  std::list< Block >::iterator p1 = block_list.begin();
  while( p1 != block_list.end() )
    {
    std::list< Block >::iterator next = p1;
    ++next;
    if( p1->height() <= 2 && p1->width() <= 2 /*&& p1->id() > 0*/ )
      block_list.erase( p1 );
    p1 = next;
    }
  }

/*
void hierarchize_blocks( std::list< Block > & block_list ) throw()
  {
  std::list< Block >::iterator p1 = block_list.begin();
  for( ; p1 != block_list.end(); ++p1 )
    {
    std::list< Block >::iterator p2 = block_list.begin();
    while( p2 != block_list.end() )
      {
      std::list< Block >::iterator next = p2;
      ++next;
      if( p1 != p2 && p1->includes( *p2 ) )
        {
        int id = p2->id();
        p1->_block_list.splice( p1->_block_list.end(), block_list, p2 );
        if( id > 0 ) hierarchize_blocks( p1->_block_list );
        }
      p2 = next;
      }
    }
  }
*/

void hierarchize_blocks( std::list< Block > & block_list ) throw()
  {
  std::list< Block >::iterator p1 = block_list.begin();
  for( ; p1 != block_list.end(); ++p1 ) if( p1->id() > 0 )
    {
    std::list< Block >::iterator p2 = block_list.begin();
    while( p2 != block_list.end() )
      {
      std::list< Block >::iterator next = p2;
      ++next;
      if( p1 != p2 && p1->includes( *p2 ) )
        {
        if( p2->id() < 0 )
          p1->_block_list.splice( p1->_block_list.end(), block_list, p2 );
        else
          {
          std::list< Block >::iterator p3 = p1->_block_list.begin();
          for( ; p3 != p1->_block_list.end(); ++p3 )
            if( p3->includes( *p2 ) )
              p3->_block_list.splice( p3->_block_list.end(), block_list, p2 );
          }
        }
      p2 = next;
      }
    }
  }


void order_blocks( std::list< Block > & block_list ) throw()
  {
  std::list< Block >::iterator p1 = block_list.begin();
  while( p1 != block_list.end() )
    {
    std::list< Block >::iterator best = p1;
    std::list< Block >::iterator p2 = p1;
    while( ++p2 != block_list.end() && p2->top() <= best->bottom() )
      if( p2->left() < best->left() ) best = p2;
    if( best != p1 ) block_list.splice( p1, block_list, best );
    else ++p1;
    }
  }


int generate_black_id() { static int next = 0; return ++next; }

int generate_white_id() { static int next = 0; return --next; }

Blockmap::Blockmap( const Bitmap & page_image, const int debug_level )
						throw( Internal_error )
  {
  _height = page_image.height();
  _width = page_image.width();

  data.resize( _height );
  for( int row = 0; row < _height; ++row )
    {
    data[row].resize( _width );
    for( int col = 0; col < _width; ++col )
      {
      if( page_image.get_bit( row, col ) == true )	// point is black
        {
        if( row > 0 && data[row-1][col] > 0 )
          {
          data[row][col] = data[row-1][col];
          add_point_to_block( row, col, data[row][col] );
          if( col > 0 && data[row][col-1] > 0 && data[row][col-1] != data[row-1][col] )
            join_blocks( data[row-1][col], data[row][col-1] );
          }
        else if( col > 0 && data[row][col-1] > 0 )
          {
          data[row][col] = data[row][col-1];
          add_point_to_block( row, col, data[row][col] );
          }
        else
          {
          data[row][col] = generate_black_id();
          create_block( Rectangle( col, row, col, row ), data[row][col] );
          }
        }
      else						// point is white
        {
        if( row == 0 || col == 0 ) data[row][col] = 0;
        else if( data[row-1][col] == 0 || data[row][col-1] == 0 )
          {
          data[row][col] = 0;
          if( data[row-1][col] < 0 ) delete_block( data[row-1][col] );
          else if( data[row][col-1] < 0 ) delete_block( data[row][col-1] );
          }
        else if( data[row-1][col] < 0 )
          {
          data[row][col] = data[row-1][col];
          add_point_to_block( row, col, data[row][col] );
          if( data[row][col-1] < 0 && data[row][col-1] != data[row-1][col] )
            join_blocks( data[row-1][col], data[row][col-1] );
          }
        else if( data[row][col-1] < 0 )
          {
          data[row][col] = data[row][col-1];
          add_point_to_block( row, col, data[row][col] );
          }
        else
          {
          data[row][col] = generate_white_id();
          create_block( Rectangle( col, row, col, row ), data[row][col] );
          }
        }
      }
    }
  delete_small_blocks( _block_list );
  if( debug_level <= 99 ) hierarchize_blocks( _block_list );
  if( debug_level <= 97 ) order_blocks( _block_list );
  }
