/*
   Kickshaw - A Menu Editor for Openbox

   Copyright (c) 2010–2018        Marcus Schätzle

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along 
   with Kickshaw. If not, see http://www.gnu.org/licenses/.
*/

#include <gtk/gtk.h>

#if !(GLIB_CHECK_VERSION(2,6,0))
#error "Kickshaw needs GLib version 2.6 or higher to compile. Please make sure to have at least GLib version 2.6 installed. Compilation aborted."
#endif
#if !(GTK_CHECK_VERSION(2,6,0))
#error "Kickshaw needs GLib version 2.6 or higher to compile. The first version based upon a version of GLib >= 2.6 is GTK 2.6. Please make sure to have at least GTK version 2.6 installed. Compilation aborted."
#endif

#include <errno.h>
#include <locale.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <unistd.h>
#if !(GLIB_CHECK_VERSION(2,14,0))
#include <regex.h>
#endif

#include "definitions_and_enumerations.h"
#include "kickshaw.h"

ks_data ks = {
#if !(GTK_CHECK_VERSION(2,16,0))
     .menu_item_txts = { "Show Menu ID column", "Show Execute column", "Show Element Visibility column", "Keep highlighting", 
                         "Don't keep highlighting", "Show icons", "Set off separators", "Draw rows in altern. colours (theme dep.)", 
                         "Show tree lines", "No grid lines", "Horizontal", "Vertical", "Both", "Create backup before overwriting menu", 
                         "Sort execute/startupnotify options", "Always notify about execute opt. conversions" }, 
#endif
    .options_label_txts = { " Prompt: ", " Command: ", " Startupnotify " }, 

    .actions = { "Execute", "Exit", "Reconfigure", "Restart", "SessionLogout" }, 

    .execute_displayed_txts = { "Prompt", "Command", "Startupnotify" }, 
    .execute_options = { "prompt", "command", "startupnotify" },

    .startupnotify_options = { "enabled", "name", "wmclass", "icon" }, 
    .startupnotify_displayed_txts = { "Enabled", "Name", "WM_CLASS", "Icon" },

    .column_header_txts = { "Menu Element", "Type", "Value", "Menu ID", "Execute", "Element Visibility" }, 

    .enable_list = {{ "STRING", GTK_TARGET_SAME_WIDGET, 0 }}
};

static void general_initialisiation (void);
static void add_button_content (GtkWidget *button, gchar *label_text);
static void pack_entry_field (GtkWidget *widget, guint8 index);
static void set_column_attributes (G_GNUC_UNUSED GtkTreeViewColumn *cell_column, 
                                                 GtkCellRenderer   *txt_renderer,
                                                 GtkTreeModel      *cell_model, 
                                                 GtkTreeIter       *cell_iter, 
                                                 gpointer           column_number_pointer);
static void change_view_and_options (gpointer activated_menu_item_pointer);
void set_status_of_expand_and_collapse_buttons_and_menu_items (void);
static void about (void);
gboolean continue_despite_unsaved_changes (void);
void clear_global_data (void);
static void new_menu (void);
static void quit_program (gpointer ask_before_quitting);
void activate_change_done (void);
static void write_settings (void);
void set_filename_and_window_title (gchar *new_filename);
void show_errmsg (gchar *errmsg_raw_txt);
void show_msg_in_statusbar (gchar *message);

int main (G_GNUC_UNUSED int argc, char *argv[])
{
    // --- Startup checks --- 


    // ### Display program version. ###

    if (G_UNLIKELY (STREQ (argv[1], "--version"))) {
        g_print ("Kickshaw " KICKSHAW_VERSION
"\nCopyright (C) 2010-2018    Marcus Schaetzle\
\n\nKickshaw comes with ABSOLUTELY NO WARRANTY.\
\nYou may redistribute copies of Kickshaw\
\nunder the terms of the GNU General Public License.\
\nFor more information about these matters, see the file named COPYING.\n");
        exit (EXIT_SUCCESS);
    }

    // ### Check if a windowing system is running. ###

    if (G_UNLIKELY (!g_getenv ("DISPLAY"))) {
        g_printerr ("No windowing system currently running, program aborted.\n");
        exit (EXIT_FAILURE);
    }


    // ### Prevent multiple instances of Kickshaw ###

    ks.lock_file_path = g_strconcat (g_getenv ("HOME"), "/.kickshaw.lck", NULL);
    if (G_UNLIKELY (!(ks.lock_file = fopen (ks.lock_file_path, "ab+")))) {
        g_printerr ("Could not open lock file for Kickshaw!\n");

        exit (EXIT_FAILURE);
    }

    if (g_file_test (ks.lock_file_path, G_FILE_TEST_EXISTS) && 
        flock (fileno (ks.lock_file), LOCK_EX|LOCK_NB) == -1 && 
        errno == EWOULDBLOCK) {
        g_printerr ("Kickshaw is already open!\n");

        fclose (ks.lock_file);

        exit (EXIT_FAILURE);
    }

    if (flock (fileno (ks.lock_file), LOCK_EX) == -1) {
        gchar *error = NULL; // Initialization avoids compiler warning.
        gchar *error_msg;

        switch (errno) {
            case EBADF:
                error = "The first argument passed to flock () is not an open file descriptor.";
                break;
            case EINTR:
                error = "While waiting to acquire a lock, the call was interrupted by delivery of a signal caught by a handler.";
                break;
            case EINVAL:
                error = "The operation is invalid.";
                break;
            case ENOLCK:
                error = "The kernel ran out of memory for allocating lock records.";
        }

        error_msg = g_strdup_printf ("Could not create a lock file for Kickshaw!\nError: %s\n", error); 
        g_printerr ("%s", error_msg);

        fclose (ks.lock_file);

        exit (EXIT_FAILURE);
    }


    // --- Initialise GTK, create the GUI and set up everything else. ---


	gtk_init (NULL, NULL); // Commandline handling is not required, so NULL is passed for argc and argv.

	general_initialisiation ();

	gtk_main ();
}

/* 

    Creates GUI and signals, also loads settings and standard menu file, if they exist.

*/

static void general_initialisiation (void)
{
    GtkWidget *main_box;

    GtkWidget *menubar;
    GSList *element_visibility_menu_item_group = NULL, *grid_menu_item_group = NULL;
    GtkWidget *mb_file, *mb_find, *mb_view, *mb_help,
              *mb_show_element_visibility_column, *mb_show_grid, *mb_about;
    enum { FILE_MENU, EDIT_MENU, SEARCH_MENU, VIEW_MENU, OPTIONS_MENU, HELP_MENU, 
           SHOW_ELEMENT_VISIBILITY_COLUMN_MENU, SHOW_GRID_MENU, NUMBER_OF_SUBMENUS };
    GtkWidget *submenus[NUMBER_OF_SUBMENUS];
    GtkMenuShell *view_submenu;

    GtkAccelGroup *accel_group = NULL;
    guint accel_key;
    GdkModifierType accel_mod;

    gchar *file_menu_item_txts[] = { "_New", "_Open", "_Save", "Save As", "", "_Quit" };
    gchar *file_menu_item_accelerators[] = { "<Ctl>N", "<Ctl>O", "<Ctl>S", "<Shift><Ctl>S", "", "<Ctl>Q" };

    gchar *visibility_txts[] = { "Show Element Visibility column", "Keep highlighting", "Don't keep highlighting" };
    gchar *grid_txts[] = { "No grid lines", "Horizontal", "Vertical", "Both" };
    guint default_checked[] = { SHOW_MENU_ID_COL, SHOW_EXECUTE_COL, SHOW_ICONS, SET_OFF_SEPARATORS, SHOW_TREE_LINES };
    guint8 txts_cnt;

    GtkWidget *toolbar;
    gchar *button_IDs[] = { GTK_STOCK_NEW, GTK_STOCK_OPEN, GTK_STOCK_SAVE, GTK_STOCK_SAVE_AS, GTK_STOCK_GO_UP, 
                            GTK_STOCK_GO_DOWN, GTK_STOCK_REMOVE, GTK_STOCK_FIND, GTK_STOCK_ZOOM_IN, GTK_STOCK_ZOOM_OUT, 
                            GTK_STOCK_QUIT };
    gchar *tb_tooltips[] = { "New menu", "Open menu", "Save menu", "Save menu as...", "Move up", 
                             "Move down", "Remove", "Find", "Expand all", "Collapse all", "Quit" };
    GtkToolItem *tb_separator;

    // Label text of the last button is set dynamically dependent on the type of the selected row.
    gchar *bt_add_txts[] = { "_Menu", "_Pipe Menu", "_Item", "Sepa_rator", "" };
    gchar *add_txts[] = { "menu", "pipe menu", "item", "separator" };

    enum { CHANGE_VALUES_CANCEL, CHANGE_VALUES_RESET, CHANGE_VALUES_DONE, NUMBER_OF_CHANGE_VALUES_BUTTONS };
    GtkWidget *change_values_buttons[NUMBER_OF_CHANGE_VALUES_BUTTONS];
    gchar *change_values_button_txts[NUMBER_OF_CHANGE_VALUES_BUTTONS] = { "_Cancel", "_Reset", "_Done" };
    GtkWidget *separator_change_values_label_bottom_alignment, *change_values_buttons_box_alignment;

    GtkWidget *options_check_button_box, *suboptions_table_event_box;

    gchar *find_entry_buttons_imgs[] = { GTK_STOCK_CLOSE, GTK_STOCK_GO_BACK, GTK_STOCK_GO_FORWARD };
    gchar *find_entry_buttons_tooltips[] = { "Close", "Previous", "Next" };
    enum { ENTRY, COLUMNS_SELECTION, SPECIAL_OPTIONS, NUMBER_OF_FIND_SUBBOXES };
    GtkWidget *find_subboxes[NUMBER_OF_FIND_SUBBOXES];
    GtkWidget *find_buttons_box, *find_buttons_alignment;

#if !(GTK_CHECK_VERSION(2,12,0))
    GtkTooltips *tooltips = gtk_tooltips_new ();
    GtkToolItem *tb_items[NUMBER_OF_TB_BUTTONS];
    GtkWidget *tb_event_boxes[NUMBER_OF_TB_BUTTONS];
#endif

    GtkWidget *scrolled_window;
    GtkTreeSelection *selection;

    GtkWidget *menu_element_or_value_entry_box, *inner_icon_buttons_box, *icon_buttons_alignment;

    gchar *settings_file_path;

    gchar *new_filename;

    guint8 highlighted_index;
    guint8 buttons_cnt, columns_cnt, entry_fields_cnt, grids_cnt, mb_menu_items_cnt, 
           options_cnt, snotify_opts_cnt, submenus_cnt, view_and_opts_cnt, widgets_cnt;


    // Also needed to ensure that the search functionality uses a utf-8 locale on systems with GLib < 2.14-
    setlocale (LC_ALL,"en_US.utf8");


    // --- Creating the GUI. ---


    // ### Create a new window. ###

    ks.window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

    gtk_window_set_title (GTK_WINDOW (ks.window), "Kickshaw");

    gtk_window_set_position (GTK_WINDOW (ks.window), GTK_WIN_POS_CENTER);

    g_signal_connect (ks.window, "delete-event", G_CALLBACK (quit_program), GUINT_TO_POINTER (FALSE));

    // ### Create a new grid that will contain all elements. ###

    main_box = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (ks.window), main_box);

    // ### Create a menu bar. ###

    menubar = gtk_menu_bar_new ();
    gtk_box_pack_start (GTK_BOX (main_box), menubar, FALSE, FALSE, 0);

    // Submenus
    for (submenus_cnt = 0; submenus_cnt < NUMBER_OF_SUBMENUS; submenus_cnt++) {
        submenus[submenus_cnt] = gtk_menu_new ();
    }

    // Accelerator Group
    accel_group = gtk_accel_group_new ();
    gtk_window_add_accel_group (GTK_WINDOW (ks.window), accel_group);

    // File
    mb_file = gtk_menu_item_new_with_mnemonic ("_File");

    for (mb_menu_items_cnt = MB_NEW; mb_menu_items_cnt <= MB_QUIT; mb_menu_items_cnt++) {
        if (mb_menu_items_cnt != MB_SEPARATOR_FILE) {
            ks.mb_file_menu_items[mb_menu_items_cnt] = gtk_menu_item_new_with_mnemonic (file_menu_item_txts[mb_menu_items_cnt]);
            gtk_accelerator_parse (file_menu_item_accelerators[mb_menu_items_cnt], &accel_key, &accel_mod);
            gtk_widget_add_accelerator (ks.mb_file_menu_items[mb_menu_items_cnt], "activate", accel_group, 
                                        accel_key, accel_mod, GTK_ACCEL_VISIBLE);
        }
        else {
            ks.mb_file_menu_items[MB_SEPARATOR_FILE] = gtk_separator_menu_item_new ();
        }
    }

    for (mb_menu_items_cnt = 0; mb_menu_items_cnt < NUMBER_OF_FILE_MENU_ITEMS; mb_menu_items_cnt++) {
        gtk_menu_shell_append (GTK_MENU_SHELL (submenus[FILE_MENU]), ks.mb_file_menu_items[mb_menu_items_cnt]);
    }

    gtk_menu_item_set_submenu (GTK_MENU_ITEM (mb_file), submenus[FILE_MENU]);
    gtk_menu_shell_append (GTK_MENU_SHELL (menubar), mb_file);

    // Edit
    ks.mb_edit = gtk_menu_item_new_with_mnemonic ("_Edit");

    ks.mb_edit_menu_items[MB_MOVE_TOP] = gtk_menu_item_new_with_label ("Move to top");
    gtk_accelerator_parse ("<Ctl>T", &accel_key, &accel_mod);
    gtk_widget_add_accelerator (ks.mb_edit_menu_items[MB_MOVE_TOP], "activate", accel_group, accel_key, accel_mod, 
                                GTK_ACCEL_VISIBLE);
    ks.mb_edit_menu_items[MB_MOVE_UP] = gtk_menu_item_new_with_label ("Move up");
    gtk_accelerator_parse ("<Ctl>minus", &accel_key, &accel_mod);
    gtk_widget_add_accelerator (ks.mb_edit_menu_items[MB_MOVE_UP], "activate", accel_group, accel_key, accel_mod, 
                                GTK_ACCEL_VISIBLE);
    ks.mb_edit_menu_items[MB_MOVE_DOWN] = gtk_menu_item_new_with_label ("Move down");
    gtk_accelerator_parse ("<Ctl>plus", &accel_key, &accel_mod);
    gtk_widget_add_accelerator (ks.mb_edit_menu_items[MB_MOVE_DOWN], "activate", accel_group, accel_key, accel_mod, 
                                GTK_ACCEL_VISIBLE);
    ks.mb_edit_menu_items[MB_MOVE_BOTTOM] = gtk_menu_item_new_with_label ("Move to bottom");
    gtk_accelerator_parse ("<Ctl>B", &accel_key, &accel_mod);
    gtk_widget_add_accelerator (ks.mb_edit_menu_items[MB_MOVE_BOTTOM], "activate", accel_group, accel_key, accel_mod, 
                                GTK_ACCEL_VISIBLE);
    ks.mb_edit_menu_items[MB_REMOVE] = gtk_menu_item_new_with_label ("Remove");
    gtk_accelerator_parse ("<Ctl>R", &accel_key, &accel_mod);
    gtk_widget_add_accelerator (ks.mb_edit_menu_items[MB_REMOVE], "activate", accel_group, accel_key, accel_mod, 
                                GTK_ACCEL_VISIBLE);
    ks.mb_edit_menu_items[MB_SEPARATOR_EDIT1] = gtk_separator_menu_item_new ();
    ks.mb_edit_menu_items[MB_REMOVE_ALL_CHILDREN] = gtk_menu_item_new_with_label ("Remove all children");
    gtk_accelerator_parse ("<Ctl><Shift>R", &accel_key, &accel_mod);
    gtk_widget_add_accelerator (ks.mb_edit_menu_items[MB_REMOVE_ALL_CHILDREN], "activate", accel_group, accel_key, accel_mod, 
                                GTK_ACCEL_VISIBLE);
    ks.mb_edit_menu_items[MB_SEPARATOR_EDIT2] = gtk_separator_menu_item_new ();
    ks.mb_edit_menu_items[MB_VISUALISE] = gtk_menu_item_new_with_label ("Visualise");
    ks.mb_edit_menu_items[MB_VISUALISE_RECURSIVELY] = gtk_menu_item_new_with_label ("Visualise recursively");

    for (mb_menu_items_cnt = 0; mb_menu_items_cnt < NUMBER_OF_EDIT_MENU_ITEMS; mb_menu_items_cnt++) {
        gtk_menu_shell_append (GTK_MENU_SHELL (submenus[EDIT_MENU]), ks.mb_edit_menu_items[mb_menu_items_cnt]);
    }
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (ks.mb_edit), submenus[EDIT_MENU]);
    gtk_menu_shell_append (GTK_MENU_SHELL (menubar), ks.mb_edit);

    // Search
    ks.mb_search = gtk_menu_item_new_with_mnemonic ("_Search");

    mb_find = gtk_menu_item_new_with_mnemonic ("_Find");
    gtk_accelerator_parse ("<Ctl>F", &accel_key, &accel_mod);
    gtk_widget_add_accelerator (mb_find, "activate", accel_group, accel_key, accel_mod, GTK_ACCEL_VISIBLE);

    gtk_menu_shell_append (GTK_MENU_SHELL (submenus[SEARCH_MENU]), mb_find);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (ks.mb_search), submenus[SEARCH_MENU]);
    gtk_menu_shell_append (GTK_MENU_SHELL (menubar), ks.mb_search);

    // View
    mb_view = gtk_menu_item_new_with_mnemonic ("_View");

    ks.mb_expand_all_nodes = gtk_menu_item_new_with_label ("Expand all nodes");
    gtk_accelerator_parse ("<Ctl><Shift>X", &accel_key, &accel_mod);
    gtk_widget_add_accelerator (ks.mb_expand_all_nodes, "activate", accel_group, accel_key, accel_mod, GTK_ACCEL_VISIBLE);
    ks.mb_collapse_all_nodes = gtk_menu_item_new_with_label ("Collapse all nodes");
    gtk_accelerator_parse ("<Ctl><Shift>C", &accel_key, &accel_mod);
    gtk_widget_add_accelerator (ks.mb_collapse_all_nodes, "activate", accel_group, accel_key, accel_mod, GTK_ACCEL_VISIBLE);
    ks.mb_view_and_options[SHOW_MENU_ID_COL] = gtk_check_menu_item_new_with_label ("Show Menu ID column");
    ks.mb_view_and_options[SHOW_EXECUTE_COL] = gtk_check_menu_item_new_with_label ("Show Execute column");
    mb_show_element_visibility_column = gtk_menu_item_new_with_label ("Show Element Visibility column");
    ks.mb_view_and_options[SHOW_ICONS] = gtk_check_menu_item_new_with_label ("Show icons");
    ks.mb_view_and_options[SET_OFF_SEPARATORS] = gtk_check_menu_item_new_with_label ("Set off separators");
    ks.mb_view_and_options[DRAW_ROWS_IN_ALT_COLOURS] = 
        gtk_check_menu_item_new_with_label ("Draw rows in altern. colours (theme dep.)");
    ks.mb_view_and_options[SHOW_TREE_LINES] = gtk_check_menu_item_new_with_label ("Show tree lines"); 
    mb_show_grid = gtk_menu_item_new_with_label ("Show grid");

    for (mb_menu_items_cnt = SHOW_ELEMENT_VISIBILITY_COL_ACTVTD, txts_cnt = 0;
         mb_menu_items_cnt <= SHOW_ELEMENT_VISIBILITY_COL_DONT_KEEP_HIGHL; 
         mb_menu_items_cnt++, txts_cnt++) {
        if (mb_menu_items_cnt == SHOW_ELEMENT_VISIBILITY_COL_ACTVTD) {
            ks.mb_view_and_options[mb_menu_items_cnt] = gtk_check_menu_item_new_with_label (visibility_txts[txts_cnt]);
        }
        else {
            ks.mb_view_and_options[mb_menu_items_cnt] = gtk_radio_menu_item_new_with_label (element_visibility_menu_item_group, 
                                                                                            visibility_txts[txts_cnt]);
            element_visibility_menu_item_group =
                gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (ks.mb_view_and_options[mb_menu_items_cnt]));
        }
        gtk_menu_shell_append (GTK_MENU_SHELL (submenus[SHOW_ELEMENT_VISIBILITY_COLUMN_MENU]), 
                               ks.mb_view_and_options[mb_menu_items_cnt]);
    }
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (mb_show_element_visibility_column), 
        submenus[SHOW_ELEMENT_VISIBILITY_COLUMN_MENU]);

    for (mb_menu_items_cnt = NO_GRID_LINES, txts_cnt = 0;
         mb_menu_items_cnt <= BOTH;
         mb_menu_items_cnt++, txts_cnt++) {
        ks.mb_view_and_options[mb_menu_items_cnt] = gtk_radio_menu_item_new_with_label (grid_menu_item_group, 
                                                    grid_txts[txts_cnt]);
        grid_menu_item_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (ks.mb_view_and_options[mb_menu_items_cnt]));
        gtk_menu_shell_append (GTK_MENU_SHELL (submenus[SHOW_GRID_MENU]), ks.mb_view_and_options[mb_menu_items_cnt]);
    }
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (mb_show_grid), submenus[SHOW_GRID_MENU]);

    view_submenu = GTK_MENU_SHELL (submenus[VIEW_MENU]);
    gtk_menu_shell_append (view_submenu, ks.mb_expand_all_nodes);
    gtk_menu_shell_append (view_submenu, ks.mb_collapse_all_nodes);
    gtk_menu_shell_append (view_submenu, gtk_separator_menu_item_new ());
    gtk_menu_shell_append (view_submenu, ks.mb_view_and_options[SHOW_MENU_ID_COL]);
    gtk_menu_shell_append (view_submenu, ks.mb_view_and_options[SHOW_EXECUTE_COL]);
    gtk_menu_shell_append (view_submenu, mb_show_element_visibility_column);
    gtk_menu_shell_append (view_submenu, gtk_separator_menu_item_new ());
    gtk_menu_shell_append (view_submenu, ks.mb_view_and_options[SHOW_ICONS]);
    gtk_menu_shell_append (view_submenu, gtk_separator_menu_item_new ());
    gtk_menu_shell_append (view_submenu, ks.mb_view_and_options[SET_OFF_SEPARATORS]);
    gtk_menu_shell_append (view_submenu, gtk_separator_menu_item_new ());
    gtk_menu_shell_append (view_submenu, ks.mb_view_and_options[DRAW_ROWS_IN_ALT_COLOURS]);
    gtk_menu_shell_append (view_submenu, ks.mb_view_and_options[SHOW_TREE_LINES]);
    gtk_menu_shell_append (view_submenu, mb_show_grid);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (mb_view), submenus[VIEW_MENU]);
    gtk_menu_shell_append (GTK_MENU_SHELL (menubar), mb_view);

    // Default settings
    for (mb_menu_items_cnt = 0; mb_menu_items_cnt < G_N_ELEMENTS (default_checked); mb_menu_items_cnt++) {
        gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (ks.mb_view_and_options[default_checked[mb_menu_items_cnt]]), TRUE);
    }
    gtk_widget_set_sensitive (ks.mb_view_and_options[SHOW_ELEMENT_VISIBILITY_COL_KEEP_HIGHL], FALSE);
    gtk_widget_set_sensitive (ks.mb_view_and_options[SHOW_ELEMENT_VISIBILITY_COL_DONT_KEEP_HIGHL], FALSE);

    // Options
    ks.mb_options = gtk_menu_item_new_with_mnemonic ("_Options");
    ks.mb_view_and_options[CREATE_BACKUP_BEFORE_OVERWRITING_MENU] = 
        gtk_check_menu_item_new_with_label ("Create backup before overwriting menu");
    ks.mb_view_and_options[SORT_EXECUTE_AND_STARTUPN_OPTIONS] = 
        gtk_check_menu_item_new_with_label ("Sort execute/startupnotify options");
    ks.mb_view_and_options[NOTIFY_ABOUT_EXECUTE_OPT_CONVERSIONS] = 
        gtk_check_menu_item_new_with_label ("Always notify about execute opt. conversions");

    gtk_menu_shell_append (GTK_MENU_SHELL (submenus[OPTIONS_MENU]), ks.mb_view_and_options[CREATE_BACKUP_BEFORE_OVERWRITING_MENU]);
    gtk_menu_shell_append (GTK_MENU_SHELL (submenus[OPTIONS_MENU]), ks.mb_view_and_options[SORT_EXECUTE_AND_STARTUPN_OPTIONS]);
    gtk_menu_shell_append (GTK_MENU_SHELL (submenus[OPTIONS_MENU]), ks.mb_view_and_options[NOTIFY_ABOUT_EXECUTE_OPT_CONVERSIONS]);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (ks.mb_options), submenus[OPTIONS_MENU]);
    gtk_menu_shell_append (GTK_MENU_SHELL (menubar), ks.mb_options);

    // Default settings
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (ks.mb_view_and_options[CREATE_BACKUP_BEFORE_OVERWRITING_MENU]), TRUE);
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (ks.mb_view_and_options[NOTIFY_ABOUT_EXECUTE_OPT_CONVERSIONS]), TRUE);

    // Help
    mb_help = gtk_menu_item_new_with_mnemonic ("_Help");

    mb_about = gtk_menu_item_new_with_label ("About");

    gtk_menu_shell_append (GTK_MENU_SHELL (submenus[HELP_MENU]), mb_about);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (mb_help), submenus[HELP_MENU]);
    gtk_menu_shell_append (GTK_MENU_SHELL (menubar), mb_help);

    // ### Create toolbar. ###

    toolbar = gtk_toolbar_new ();
    gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_ICONS);
    gtk_box_pack_start (GTK_BOX (main_box), toolbar, FALSE, FALSE, 0);

    gtk_container_set_border_width (GTK_CONTAINER (toolbar), 2);
    for (buttons_cnt = 0; buttons_cnt < NUMBER_OF_TB_BUTTONS; buttons_cnt++) {
        switch (buttons_cnt) {
            case TB_MOVE_UP:
            case TB_FIND:
            case TB_QUIT:
                tb_separator = gtk_separator_tool_item_new ();
                gtk_toolbar_insert (GTK_TOOLBAR (toolbar), tb_separator, -1);
        }

        ks.tb[buttons_cnt] = gtk_tool_button_new_from_stock (button_IDs[buttons_cnt]);

#if GTK_CHECK_VERSION(2,12,0)
        gtk_widget_set_tooltip_text (GTK_WIDGET (ks.tb[buttons_cnt]), tb_tooltips[buttons_cnt]);
        gtk_toolbar_insert (GTK_TOOLBAR (toolbar), ks.tb[buttons_cnt], -1);
#else
        tb_items[buttons_cnt] = gtk_tool_item_new ();
        tb_event_boxes[buttons_cnt] = gtk_event_box_new ();
        gtk_container_add (GTK_CONTAINER (tb_items[buttons_cnt]), tb_event_boxes[buttons_cnt]);
        gtk_container_add (GTK_CONTAINER (tb_event_boxes[buttons_cnt]), GTK_WIDGET (ks.tb[buttons_cnt]));
        gtk_tooltips_set_tip (tooltips, GTK_WIDGET (tb_event_boxes[buttons_cnt]), tb_tooltips[buttons_cnt], NULL);
        gtk_toolbar_insert (GTK_TOOLBAR (toolbar), tb_items[buttons_cnt], -1);
#endif
    }

    // ### Create Button Bar. ###

    ks.button_box = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main_box), ks.button_box, FALSE, FALSE, 0);

    ks.add_image = gtk_image_new_from_stock (GTK_STOCK_ADD, GTK_ICON_SIZE_BUTTON);
    gtk_box_pack_start (GTK_BOX (ks.button_box), ks.add_image, FALSE, FALSE, 0);

    // Label text is set dynamically dependent on whether it is possible to add a row or not.
    ks.bt_bar_label = gtk_label_new (NULL);
    gtk_box_pack_start (GTK_BOX (ks.button_box), ks.bt_bar_label, FALSE, FALSE, 0);

    // Label text is set dynamically dependent on the type of the selected row.
    ks.bt_add_action_option_label = gtk_label_new (NULL);
    gtk_label_set_use_underline (GTK_LABEL (ks.bt_add_action_option_label), TRUE);

    for (buttons_cnt = 0; buttons_cnt < NUMBER_OF_ADD_BUTTONS; buttons_cnt++) {
        ks.bt_add[buttons_cnt] = gtk_button_new ();
        add_button_content (ks.bt_add[buttons_cnt], bt_add_txts[buttons_cnt]);
        gtk_box_pack_start (GTK_BOX (ks.button_box), ks.bt_add[buttons_cnt], FALSE, FALSE, 0);
    }

    ks.separator_change_values_label_top = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (main_box), ks.separator_change_values_label_top, FALSE, FALSE, 5);

    ks.change_values_label = gtk_label_new ("");
    gtk_box_pack_start (GTK_BOX (main_box), ks.change_values_label, FALSE, FALSE, 0);

    ks.separator_change_values_label_bottom = gtk_hseparator_new ();
    separator_change_values_label_bottom_alignment = gtk_alignment_new (0.0, 0.0, 1.0, 0.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (separator_change_values_label_bottom_alignment), 5, 12, 0, 0);
    gtk_container_add (GTK_CONTAINER (separator_change_values_label_bottom_alignment), ks.separator_change_values_label_bottom);
    gtk_box_pack_start (GTK_BOX (main_box), separator_change_values_label_bottom_alignment, FALSE, FALSE, 0);


    ks.subbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (main_box), ks.subbox);


    // ### Box for entering the values of new rows. ###


    ks.action_option_box = gtk_vbox_new (FALSE, 0);
    ks.action_option_box_alignment = gtk_alignment_new (0.0, 0.0, 1.0, 0.0);
    gtk_container_add (GTK_CONTAINER (ks.action_option_box_alignment), ks.action_option_box);
    gtk_box_pack_start (GTK_BOX (ks.subbox), ks.action_option_box_alignment, FALSE, FALSE, 0);
    // List store and model for action/option combo box; the combo box isn't yet created here, only when needed.
    ks.action_option_combo_box_liststore = gtk_list_store_new (NUMBER_OF_ACTION_OPTION_COMBO_ELEMENTS, G_TYPE_STRING);
    ks.action_option_combo_box_model = GTK_TREE_MODEL (ks.action_option_combo_box_liststore);

    ks.new_action_option_table = gtk_table_new (1, 7, FALSE);
    ks.new_action_option_table_alignment = gtk_alignment_new (0.0, 0.0, 1.0, 0.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (ks.new_action_option_table_alignment), 5, 5, 0, 0);
    gtk_container_add (GTK_CONTAINER (ks.new_action_option_table_alignment), ks.new_action_option_table);
    gtk_container_add (GTK_CONTAINER (ks.action_option_box), ks.new_action_option_table_alignment);

    ks.new_action_option_widgets[INSIDE_MENU_LABEL] = gtk_label_new (" Inside menu ");

    ks.new_action_option_widgets[INSIDE_MENU_CHECK_BUTTON] = gtk_check_button_new ();

    ks.new_action_option_widgets[INCLUDING_ACTION_LABEL] = gtk_label_new (" Incl. action ");

    ks.new_action_option_widgets[INCLUDING_ACTION_CHECK_BUTTON] = gtk_check_button_new ();
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ks.new_action_option_widgets[INCLUDING_ACTION_CHECK_BUTTON]), TRUE);

    ks.new_action_option_widgets[ACTION_OPTION_DONE] = gtk_button_new ();
    add_button_content (ks.new_action_option_widgets[ACTION_OPTION_DONE], "_Done");

    ks.new_action_option_widgets[ACTION_OPTION_CANCEL] = gtk_button_new ();
    add_button_content (ks.new_action_option_widgets[ACTION_OPTION_CANCEL], "_Cancel");

    for (widgets_cnt = 0; widgets_cnt < NUMBER_OF_NEW_ACTION_OPTION_WIDGETS; widgets_cnt++) {
        if (widgets_cnt != NEW_ACTION_OPTION_COMBO_BOX) {
            gtk_table_attach (GTK_TABLE (ks.new_action_option_table), ks.new_action_option_widgets[widgets_cnt], 
                              widgets_cnt, widgets_cnt + 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
        }
    }

    // Execute options
    ks.options_table = gtk_table_new (3, 2, FALSE);
    gtk_container_add (GTK_CONTAINER (ks.action_option_box), ks.options_table);

    for (options_cnt = 0; options_cnt < SN_OR_PROMPT; options_cnt++) {
        ks.options_labels[options_cnt] = gtk_label_new (ks.options_label_txts[options_cnt]);

        gtk_misc_set_alignment (GTK_MISC (ks.options_labels[options_cnt]), 0.0, 0.5);

        ks.options_fields[options_cnt] = gtk_entry_new ();

        highlighted_index = (options_cnt == PROMPT) ? PROMPT_ENTRY_HL : COMMAND_ENTRY_HL;

        pack_entry_field (ks.options_fields[options_cnt], highlighted_index);

        gtk_table_attach (GTK_TABLE (ks.options_table), ks.options_labels[options_cnt], 0, 1, 
                          options_cnt, options_cnt + 1, GTK_SHRINK | GTK_FILL, GTK_FILL, 0, 0);
        gtk_table_attach (GTK_TABLE (ks.options_table), ks.highlighted_widgets[highlighted_index][EVENTBOX], 
                          1, 2, options_cnt, options_cnt + 1, GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
    }

    options_check_button_box = gtk_hbox_new (FALSE, 0);
    gtk_table_attach (GTK_TABLE (ks.options_table), options_check_button_box, 0, 1, 2, 3, 
		              GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);

    ks.options_labels[SN_OR_PROMPT] = gtk_label_new ("Startupnotify ");
    gtk_box_pack_start (GTK_BOX (options_check_button_box), ks.options_labels[SN_OR_PROMPT], FALSE, FALSE, 0);

    ks.options_fields[SN_OR_PROMPT] = gtk_check_button_new ();
    gtk_box_pack_start (GTK_BOX (options_check_button_box), ks.options_fields[SN_OR_PROMPT], FALSE, FALSE, 0);

    ks.suboptions_table = gtk_table_new (4, 2, FALSE);
    ks.suboptions_table_alignment = gtk_alignment_new (0.0, 0.0, 1.0, 0.0);
    suboptions_table_event_box = gtk_event_box_new ();
    gtk_alignment_set_padding (GTK_ALIGNMENT (ks.suboptions_table_alignment), 0, 0, 2, 0);
    gtk_container_add (GTK_CONTAINER (ks.suboptions_table_alignment), ks.suboptions_table);
    gtk_container_add (GTK_CONTAINER (suboptions_table_event_box), ks.suboptions_table_alignment);
    gtk_table_attach (GTK_TABLE (ks.options_table), suboptions_table_event_box, 1, 2, 2, 3, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

    for (snotify_opts_cnt = ENABLED; snotify_opts_cnt < NUMBER_OF_STARTUPNOTIFY_OPTS; snotify_opts_cnt++) {
        ks.suboptions_labels[snotify_opts_cnt] = gtk_label_new (NULL); // the actual label text is set later.
        gtk_misc_set_alignment (GTK_MISC (ks.suboptions_labels[snotify_opts_cnt]), 0.0, 0.5);

        gtk_table_attach (GTK_TABLE (ks.suboptions_table), ks.suboptions_labels[snotify_opts_cnt], 
		                  0, 1, snotify_opts_cnt, snotify_opts_cnt + 1, GTK_FILL | GTK_SHRINK, GTK_FILL, 0, 0);

        ks.suboptions_fields[snotify_opts_cnt] = (snotify_opts_cnt == ENABLED) ? gtk_check_button_new () : gtk_entry_new ();

        if (snotify_opts_cnt > ENABLED) {
            highlighted_index = get_highlighted_snotify_index (snotify_opts_cnt);
            pack_entry_field (ks.suboptions_fields[snotify_opts_cnt], highlighted_index);
        }

        gtk_table_attach (GTK_TABLE (ks.suboptions_table), 
                          (snotify_opts_cnt > ENABLED) ? ks.highlighted_widgets[highlighted_index][EVENTBOX] : 
                          ks.suboptions_fields[ENABLED], 
                          1, 2, snotify_opts_cnt, snotify_opts_cnt + 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
    }

    ks.mandatory = gtk_label_new (NULL);
    gtk_label_set_markup (GTK_LABEL (ks.mandatory), "(<span foreground='darkred'>*</span>) = mandatory");
    ks.mandatory_alignment = gtk_alignment_new (0.5, 0.0, 0.0, 0.0);
    gtk_container_add (GTK_CONTAINER (ks.mandatory_alignment), ks.mandatory);
    gtk_box_pack_start (GTK_BOX (ks.subbox), ks.mandatory_alignment, FALSE, FALSE, 0);

    ks.separator_change_values_buttons_box = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (ks.subbox), ks.separator_change_values_buttons_box, FALSE, FALSE, 10);

    ks.change_values_buttons_box = gtk_hbox_new (FALSE, 10);
    for (buttons_cnt = 0; buttons_cnt < NUMBER_OF_CHANGE_VALUES_BUTTONS; buttons_cnt++) {
        change_values_buttons[buttons_cnt] = gtk_button_new_with_mnemonic (change_values_button_txts[buttons_cnt]);  
        gtk_container_add (GTK_CONTAINER (ks.change_values_buttons_box), change_values_buttons[buttons_cnt]);  
    }
    change_values_buttons_box_alignment = gtk_alignment_new (0.5, 0.0, 0.0, 0.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (change_values_buttons_box_alignment), 0, 10, 0, 0);
    gtk_container_add (GTK_CONTAINER (change_values_buttons_box_alignment), ks.change_values_buttons_box);
    gtk_box_pack_start (GTK_BOX (ks.subbox), change_values_buttons_box_alignment, FALSE, FALSE, 0);

    // ### Create find box. ###

    ks.find_box = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (ks.subbox), ks.find_box, FALSE, FALSE, 0);

    for (grids_cnt = 0; grids_cnt < NUMBER_OF_FIND_SUBBOXES; grids_cnt++) {
        find_subboxes[grids_cnt] = gtk_hbox_new (FALSE, 0);
        gtk_container_add (GTK_CONTAINER (ks.find_box), find_subboxes[grids_cnt]);
    }

    ks.find_entry = gtk_entry_new ();
    pack_entry_field (ks.find_entry, FIND_ENTRY_HL);
    gtk_box_pack_end (GTK_BOX (find_subboxes[ENTRY]), ks.highlighted_widgets[FIND_ENTRY_HL][EVENTBOX], TRUE, TRUE, 0);

    find_buttons_alignment = gtk_alignment_new (0.0, 0.5, 0.0, 0.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (find_buttons_alignment), 0, 0, 0, 0);
    find_buttons_box = gtk_hbox_new (FALSE, 0);
    for (buttons_cnt = 0; buttons_cnt < NUMBER_OF_FIND_ENTRY_BUTTONS; buttons_cnt++) {
        ks.find_entry_buttons[buttons_cnt] = gtk_button_new ();
        gtk_container_add (GTK_CONTAINER (ks.find_entry_buttons[buttons_cnt]), 
                           gtk_image_new_from_stock (find_entry_buttons_imgs[buttons_cnt], GTK_ICON_SIZE_BUTTON));
#if GTK_CHECK_VERSION(2,12,0)
        gtk_widget_set_tooltip_text (GTK_WIDGET (ks.find_entry_buttons[buttons_cnt]), find_entry_buttons_tooltips[buttons_cnt]);
#else
        gtk_tooltips_set_tip (tooltips, GTK_WIDGET (ks.find_entry_buttons[buttons_cnt]), 
                              find_entry_buttons_tooltips[buttons_cnt], NULL);
#endif
        gtk_box_pack_start (GTK_BOX (find_buttons_box), ks.find_entry_buttons[buttons_cnt], FALSE, FALSE, 0);
    }
    gtk_container_add (GTK_CONTAINER (find_buttons_alignment), find_buttons_box);
    gtk_box_pack_start (GTK_BOX (find_subboxes[ENTRY]), find_buttons_alignment, FALSE, FALSE, 0);

    for (columns_cnt = 0; columns_cnt < NUMBER_OF_COLUMNS - 1; columns_cnt++) {
        ks.find_in_columns[columns_cnt] = gtk_check_button_new_with_label (ks.column_header_txts[columns_cnt]);
        gtk_box_pack_start (GTK_BOX (find_subboxes[COLUMNS_SELECTION]), ks.find_in_columns[columns_cnt], FALSE, FALSE, 0);
    }

    ks.find_in_all_columns = gtk_check_button_new_with_label ("All columns");
    gtk_box_pack_start (GTK_BOX (find_subboxes[COLUMNS_SELECTION]), ks.find_in_all_columns, FALSE, FALSE, 0);

    ks.find_match_case = gtk_check_button_new_with_label ("Match case");
    gtk_box_pack_start (GTK_BOX (find_subboxes[SPECIAL_OPTIONS]), ks.find_match_case, FALSE, FALSE, 0);

#if GLIB_CHECK_VERSION(2,14,0)
    ks.find_regular_expression = gtk_check_button_new_with_label ("Regular expression (PCRE)");
#else
    ks.find_regular_expression = gtk_check_button_new_with_label ("Regular expression (POSIX ERE)");
#endif
    gtk_box_pack_start (GTK_BOX (find_subboxes[SPECIAL_OPTIONS]), ks.find_regular_expression, FALSE, FALSE, 0);

    // Default setting
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ks.find_regular_expression), TRUE);

    // ### Initialise Tree View. ###

    ks.treeview = gtk_tree_view_new ();
#if GTK_CHECK_VERSION(2,10,0)
    gtk_tree_view_set_enable_tree_lines (GTK_TREE_VIEW (ks.treeview), TRUE); // Default
#endif

    // Create columns with cell renderers and add them to the treeview.

    for (columns_cnt = 0; columns_cnt < NUMBER_OF_COLUMNS; columns_cnt++) {
        ks.columns[columns_cnt] = gtk_tree_view_column_new ();
        gtk_tree_view_column_set_title (ks.columns[columns_cnt], ks.column_header_txts[columns_cnt]);

        if (columns_cnt == COL_MENU_ELEMENT) {
            ks.renderers[PIXBUF_RENDERER] = gtk_cell_renderer_pixbuf_new ();
            ks.renderers[EXCL_TXT_RENDERER] = gtk_cell_renderer_text_new ();
            gtk_tree_view_column_pack_start (ks.columns[COL_MENU_ELEMENT], ks.renderers[PIXBUF_RENDERER], FALSE);
            gtk_tree_view_column_pack_start (ks.columns[COL_MENU_ELEMENT], ks.renderers[EXCL_TXT_RENDERER], FALSE);
            gtk_tree_view_column_set_attributes (ks.columns[COL_MENU_ELEMENT], ks.renderers[PIXBUF_RENDERER], 
                                                 "pixbuf", TS_ICON_IMG, NULL);
        }
        else if (columns_cnt == COL_VALUE) {
            ks.renderers[BOOL_RENDERER] = gtk_cell_renderer_toggle_new ();
            g_signal_connect (ks.renderers[BOOL_RENDERER], "toggled", G_CALLBACK (boolean_toggled), NULL);
            gtk_tree_view_column_pack_start (ks.columns[COL_VALUE], ks.renderers[BOOL_RENDERER], FALSE);
        }

        ks.renderers[TXT_RENDERER] = gtk_cell_renderer_text_new ();
        g_signal_connect (ks.renderers[TXT_RENDERER], "edited", G_CALLBACK (cell_edited), GUINT_TO_POINTER ((guint) columns_cnt));
        gtk_tree_view_column_pack_start (ks.columns[columns_cnt], ks.renderers[TXT_RENDERER], FALSE);
        // TREEVIEW_COLUMN_OFFSET is used to skip the icon related columns in the treestore. 
        gtk_tree_view_column_set_attributes (ks.columns[columns_cnt], ks.renderers[TXT_RENDERER], 
                                             "text", columns_cnt + TREEVIEW_COLUMN_OFFSET, NULL);

        gtk_tree_view_column_set_cell_data_func (ks.columns[columns_cnt], ks.renderers[TXT_RENDERER], 
                                                 (GtkTreeCellDataFunc) set_column_attributes, 
                                                 GUINT_TO_POINTER ((guint) columns_cnt), 
                                                 NULL); // NULL -> no destroy notify for user data.*/
        gtk_tree_view_column_set_resizable (ks.columns[columns_cnt], TRUE);
        gtk_tree_view_append_column (GTK_TREE_VIEW (ks.treeview), ks.columns[columns_cnt]);
    }

    // Default setting
    gtk_tree_view_column_set_visible (ks.columns[COL_ELEMENT_VISIBILITY], FALSE);

    // Set treestore and model.

    /*
        Order: icon img, icon img status, icon img modification time, icon path, 
               menu element, type, value, menu ID, execute, element visibility
    */
    ks.treestore = gtk_tree_store_new (NUMBER_OF_TS_ELEMENTS, GDK_TYPE_PIXBUF, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING, 
                                       G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, 
                                       G_TYPE_STRING);

    gtk_tree_view_set_model (GTK_TREE_VIEW (ks.treeview), GTK_TREE_MODEL (ks.treestore));
    ks.model = gtk_tree_view_get_model (GTK_TREE_VIEW (ks.treeview));
    g_object_unref (ks.treestore);

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (ks.treeview));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

    // Set scrolled window that contains the treeview.
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_add (GTK_CONTAINER (scrolled_window), ks.treeview);

    gtk_box_pack_start (GTK_BOX (ks.subbox), scrolled_window, TRUE, TRUE, 0);

    // Set drag and drop destination parameters.
    gtk_tree_view_enable_model_drag_dest (GTK_TREE_VIEW (ks.treeview), ks.enable_list, 1, GDK_ACTION_MOVE);

    // ### Create entry box. ###

    ks.entry_table = gtk_table_new (3, 2, FALSE);
    gtk_box_pack_start (GTK_BOX (ks.subbox), ks.entry_table, FALSE, FALSE, 0);

    ks.entry_labels[MENU_ELEMENT_ENTRY] = gtk_label_new (" Label: ");
    gtk_misc_set_alignment (GTK_MISC (ks.entry_labels[MENU_ELEMENT_ENTRY]), 0.0, 0.5);

    gtk_table_attach (GTK_TABLE (ks.entry_table), ks.entry_labels[MENU_ELEMENT_ENTRY], 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);

    for (entry_fields_cnt = MENU_ELEMENT_ENTRY; entry_fields_cnt <= EXECUTE_ENTRY; entry_fields_cnt++) {
        ks.entry_fields[entry_fields_cnt] = gtk_entry_new ();
        pack_entry_field (ks.entry_fields[entry_fields_cnt], entry_fields_cnt);
    }

    ks.entry_labels[ICON_PATH_ENTRY] = gtk_label_new (" Icon: ");

    inner_icon_buttons_box = gtk_hbox_new (FALSE, 0);

    ks.icon_chooser = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (ks.icon_chooser), gtk_image_new_from_stock (GTK_STOCK_OPEN, GTK_ICON_SIZE_BUTTON));
#if GTK_CHECK_VERSION(2,12,0)
    gtk_widget_set_tooltip_text (GTK_WIDGET (ks.icon_chooser), "Add/Change icon");
#else
     gtk_tooltips_set_tip (tooltips, GTK_WIDGET (ks.icon_chooser), "Add/Change icon", NULL);
#endif
    gtk_container_add (GTK_CONTAINER (inner_icon_buttons_box), ks.icon_chooser);

    ks.remove_icon = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (ks.remove_icon), gtk_image_new_from_stock (GTK_STOCK_REMOVE, GTK_ICON_SIZE_BUTTON));
#if GTK_CHECK_VERSION(2,12,0)
    gtk_widget_set_tooltip_text (GTK_WIDGET (ks.remove_icon), "Remove icon");
#else
    gtk_tooltips_set_tip (tooltips, GTK_WIDGET (ks.remove_icon), "Remove icon", NULL);
#endif
    gtk_container_add (GTK_CONTAINER (inner_icon_buttons_box), ks.remove_icon);

    icon_buttons_alignment = gtk_alignment_new (0.0, 0.5, 0.0, 0.0);
    gtk_container_add (GTK_CONTAINER (icon_buttons_alignment), inner_icon_buttons_box);

    ks.icon_buttons_box = gtk_hbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (ks.icon_buttons_box), icon_buttons_alignment);

    ks.icon_box = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (ks.icon_box), ks.entry_labels[ICON_PATH_ENTRY], FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (ks.icon_box), ks.icon_buttons_box, FALSE, FALSE, 0);   

    menu_element_or_value_entry_box = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (menu_element_or_value_entry_box), 
                        ks.highlighted_widgets[MENU_ELEMENT_ENTRY_HL][EVENTBOX], TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (menu_element_or_value_entry_box), ks.icon_box, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (menu_element_or_value_entry_box), ks.highlighted_widgets[ICON_PATH_ENTRY_HL][EVENTBOX], TRUE, TRUE, 0);

    gtk_table_attach (GTK_TABLE (ks.entry_table), menu_element_or_value_entry_box, 1, 2, 0, 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

    for (entry_fields_cnt = MENU_ID_OR_VALUE_ENTRY; entry_fields_cnt <= EXECUTE_ENTRY; entry_fields_cnt++) {
        // Label text is set dynamically dependent on the type of the selected row.
        ks.entry_labels[entry_fields_cnt] = gtk_label_new (NULL);
        gtk_misc_set_alignment (GTK_MISC (ks.entry_labels[entry_fields_cnt]), 0.0, 0.5);

        gtk_table_attach (GTK_TABLE (ks.entry_table), ks.entry_labels[entry_fields_cnt], 
                          0, 1, entry_fields_cnt - 1, entry_fields_cnt, GTK_FILL, GTK_FILL, 0, 0);

        gtk_table_attach (GTK_TABLE (ks.entry_table), ks.highlighted_widgets[entry_fields_cnt][EVENTBOX], 
                          1, 2, entry_fields_cnt - 1, entry_fields_cnt, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
    }

    // ### Statusbar ###

    ks.statusbar = gtk_statusbar_new ();
    gtk_box_pack_start (GTK_BOX (main_box), ks.statusbar, FALSE, FALSE, 0);

    // --- Create signals for all buttons and relevant events. ---


    ks.handler_id_row_selected = g_signal_connect (selection, "changed", G_CALLBACK (row_selected), NULL);
    g_signal_connect (ks.treeview, "row-expanded", G_CALLBACK (set_status_of_expand_and_collapse_buttons_and_menu_items),
                      NULL);
    g_signal_connect (ks.treeview, "row-collapsed", G_CALLBACK (set_status_of_expand_and_collapse_buttons_and_menu_items), 
                      NULL);
    g_signal_connect (ks.treeview, "button-press-event", G_CALLBACK (mouse_pressed), NULL);
    g_signal_connect (ks.treeview, "button-release-event", G_CALLBACK (mouse_released), NULL);

    g_signal_connect (ks.treeview, "key-press-event", G_CALLBACK (key_pressed), NULL);

    g_signal_connect (ks.treeview, "drag-motion", G_CALLBACK (drag_motion_handler), NULL);
    g_signal_connect (ks.treeview, "drag-leave", G_CALLBACK (drag_leave_handler), NULL);
    g_signal_connect (ks.treeview, "drag-drop", G_CALLBACK (drag_drop_handler), NULL);

    g_signal_connect (ks.mb_file_menu_items[MB_NEW], "activate", G_CALLBACK (new_menu), NULL);
    g_signal_connect (ks.mb_file_menu_items[MB_OPEN], "activate", G_CALLBACK (open_menu), NULL);
    g_signal_connect_swapped (ks.mb_file_menu_items[MB_SAVE], "activate", G_CALLBACK (save_menu), NULL);
    g_signal_connect (ks.mb_file_menu_items[MB_SAVE_AS], "activate", G_CALLBACK (save_menu_as), NULL);

    g_signal_connect (ks.mb_file_menu_items[MB_QUIT], "activate", G_CALLBACK (quit_program), GUINT_TO_POINTER (TRUE));

    for (mb_menu_items_cnt = TOP; mb_menu_items_cnt <= BOTTOM; mb_menu_items_cnt++) {
        g_signal_connect_swapped (ks.mb_edit_menu_items[mb_menu_items_cnt], "activate", 
                                  G_CALLBACK (move_selection), GUINT_TO_POINTER ((guint) mb_menu_items_cnt));
    }
    g_signal_connect_swapped (ks.mb_edit_menu_items[MB_REMOVE], "activate", G_CALLBACK (remove_rows), "menu bar");
    g_signal_connect (ks.mb_edit_menu_items[MB_REMOVE_ALL_CHILDREN], "activate", G_CALLBACK (remove_all_children), NULL);
    g_signal_connect_swapped (ks.mb_edit_menu_items[MB_VISUALISE], "activate", 
                              G_CALLBACK (visualise_menus_items_and_separators), GUINT_TO_POINTER (FALSE));
    g_signal_connect_swapped (ks.mb_edit_menu_items[MB_VISUALISE_RECURSIVELY], "activate", 
                              G_CALLBACK (visualise_menus_items_and_separators), GUINT_TO_POINTER (TRUE));

    g_signal_connect (mb_find, "activate", G_CALLBACK (show_or_hide_find_box), NULL);

    g_signal_connect_swapped (ks.mb_expand_all_nodes, "activate", G_CALLBACK (expand_or_collapse_all), GUINT_TO_POINTER (TRUE));
    g_signal_connect_swapped (ks.mb_collapse_all_nodes, "activate", G_CALLBACK (expand_or_collapse_all), GUINT_TO_POINTER (FALSE));
    for (mb_menu_items_cnt = 0; mb_menu_items_cnt < NUMBER_OF_VIEW_AND_OPTIONS; mb_menu_items_cnt++) {
        g_signal_connect_swapped (ks.mb_view_and_options[mb_menu_items_cnt], "activate", 
                                  G_CALLBACK (change_view_and_options), GUINT_TO_POINTER ((guint) mb_menu_items_cnt));
    }

    g_signal_connect (mb_about, "activate", G_CALLBACK (about), NULL);

    g_signal_connect (ks.tb[TB_NEW], "clicked", G_CALLBACK (new_menu), NULL);
    g_signal_connect (ks.tb[TB_OPEN], "clicked", G_CALLBACK (open_menu), NULL);

    g_signal_connect_swapped (ks.tb[TB_SAVE], "clicked", G_CALLBACK (save_menu), NULL);
    g_signal_connect (ks.tb[TB_SAVE_AS], "clicked", G_CALLBACK (save_menu_as), NULL);
    g_signal_connect_swapped (ks.tb[TB_MOVE_UP], "clicked", G_CALLBACK (move_selection), GUINT_TO_POINTER (UP));
    g_signal_connect_swapped (ks.tb[TB_MOVE_DOWN], "clicked", G_CALLBACK (move_selection), GUINT_TO_POINTER (DOWN));
    g_signal_connect_swapped (ks.tb[TB_REMOVE], "clicked", G_CALLBACK (remove_rows), "toolbar");
    g_signal_connect (ks.tb[TB_FIND], "clicked", G_CALLBACK (show_or_hide_find_box), NULL);
    g_signal_connect_swapped (ks.tb[TB_EXPAND_ALL], "clicked", G_CALLBACK (expand_or_collapse_all), GUINT_TO_POINTER (TRUE));
    g_signal_connect_swapped (ks.tb[TB_COLLAPSE_ALL], "clicked", G_CALLBACK (expand_or_collapse_all), GUINT_TO_POINTER (FALSE));
    g_signal_connect (ks.tb[TB_QUIT], "clicked", G_CALLBACK (quit_program), GUINT_TO_POINTER (TRUE));

    for (buttons_cnt = 0; buttons_cnt < ACTION_OR_OPTION; buttons_cnt++) {
        g_signal_connect_swapped (ks.bt_add[buttons_cnt], "clicked", G_CALLBACK (add_new), add_txts[buttons_cnt]);
    }

    ks.handler_id_including_action_check_button = g_signal_connect_swapped (ks.new_action_option_widgets[INCLUDING_ACTION_CHECK_BUTTON], 
                                                                            "clicked", 
                                                                            G_CALLBACK (one_of_the_change_values_buttons_pressed), 
                                                                            "incl. action");
    g_signal_connect_swapped (change_values_buttons[CHANGE_VALUES_RESET], "clicked", 
                              G_CALLBACK (one_of_the_change_values_buttons_pressed), "reset");
    g_signal_connect_swapped (change_values_buttons[CHANGE_VALUES_DONE], "clicked", 
                              G_CALLBACK (one_of_the_change_values_buttons_pressed), "done");

    // This "simulates" a click on another row.
    g_signal_connect (change_values_buttons[CHANGE_VALUES_CANCEL], "clicked", G_CALLBACK (row_selected), NULL);
    g_signal_connect_swapped (ks.new_action_option_widgets[ACTION_OPTION_DONE], "clicked", 
                              G_CALLBACK (action_option_insert), "by combo box");
    g_signal_connect_swapped (ks.new_action_option_widgets[ACTION_OPTION_CANCEL], "clicked", 
                              G_CALLBACK (hide_action_option_box), "cancel button");
    ks.handler_id_show_or_hide_startupnotify_options = g_signal_connect (ks.options_fields[SN_OR_PROMPT], "toggled", 
                                                                         G_CALLBACK (show_or_hide_startupnotify_options), NULL);

    g_signal_connect (ks.find_entry_buttons[CLOSE], "clicked", G_CALLBACK (show_or_hide_find_box), NULL);
    for (buttons_cnt = BACK; buttons_cnt <= FORWARD; buttons_cnt++) {
        g_signal_connect_swapped (ks.find_entry_buttons[buttons_cnt], "clicked", 
                                  G_CALLBACK (jump_to_previous_or_next_occurrence),
                                  GUINT_TO_POINTER ((buttons_cnt == FORWARD)));
    }
    g_signal_connect (ks.find_entry, "activate", G_CALLBACK (run_search), NULL);
    for (columns_cnt = 0; columns_cnt < NUMBER_OF_COLUMNS - 1; columns_cnt++) {
        ks.handler_id_find_in_columns[columns_cnt] = g_signal_connect_swapped (ks.find_in_columns[columns_cnt], "clicked", 
                                                                               G_CALLBACK (find_buttons_management), "specif");
    }
    g_signal_connect_swapped (ks.find_in_all_columns, "clicked", G_CALLBACK (find_buttons_management), "all");
    g_signal_connect_swapped (ks.find_match_case, "clicked", G_CALLBACK (find_buttons_management), NULL);
    g_signal_connect_swapped (ks.find_regular_expression, "clicked", G_CALLBACK (find_buttons_management), NULL);

    for (entry_fields_cnt = 0; entry_fields_cnt < NUMBER_OF_ENTRY_FIELDS; entry_fields_cnt++) {
        ks.handler_id_entry_fields[entry_fields_cnt] = g_signal_connect (ks.entry_fields[entry_fields_cnt], "activate", 
                                                                         G_CALLBACK (change_row), NULL);
    }
    g_signal_connect (ks.icon_chooser, "clicked", G_CALLBACK (icon_choosing_by_button_or_context_menu), NULL);
    g_signal_connect (ks.remove_icon, "clicked", G_CALLBACK (remove_icons_from_menus_or_items), NULL);

    g_signal_connect (ks.options_fields[PROMPT], "activate", G_CALLBACK (single_field_entry), NULL);
    g_signal_connect (ks.options_fields[COMMAND], "activate", G_CALLBACK (single_field_entry), NULL);
    for (snotify_opts_cnt = NAME; snotify_opts_cnt < NUMBER_OF_STARTUPNOTIFY_OPTS; snotify_opts_cnt++) {
        g_signal_connect (ks.suboptions_fields[snotify_opts_cnt], "activate", G_CALLBACK (single_field_entry), NULL);
    }

    // --- Default settings ---

    g_object_get (gtk_settings_get_default (), "gtk-font-name", &ks.font_desc, NULL);
    ks.font_size = get_font_size (); // Get the default font size ...
    g_object_get (gtk_settings_get_default (), "gtk-icon-theme-name", &ks.icon_theme_name, NULL); // ... and the icon theme name.
    create_invalid_icon_imgs (); // Create broken/invalid path icon images suitable for that font size.
    ks.search_term = g_string_new (NULL);
    // Has to be placed here because the signal to deactivate all other column check boxes has to be triggered.
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ks.find_in_all_columns), TRUE); // Deactivate user defined column search.


    gtk_widget_show_all (ks.window);

    // This works only properly after gtk_widget_show_all (). The choice of the entry widget is arbitrary.
    ks.bg_normal_entry_eventbox = gtk_widget_get_style (ks.highlighted_widgets[FIND_ENTRY_HL][EVENTBOX])->bg[GTK_STATE_NORMAL];

    /*
        The height of the message label is set to be identical to the one of the buttons, so the button grid doesn't 
        shrink if the buttons are missing. This can only be done after all widgets have already been added to the grid, 
        because only then the height of the button grid (with buttons shown) can be determined correctly.
    */
    gtk_widget_set_size_request (ks.bt_bar_label, -1, ks.bt_add[0]->allocation.height);

    /*
        Usually the height of the window with all widgets shown will exceed 550 px, 
        so using gtk_window_set_default_size (GTK_WINDOW (ks.window), 650, 550) at the beginning 
        won't work as desired for a height value set to 550.
    */
    gtk_window_resize (GTK_WINDOW (ks.window), 650, 550);
    gtk_window_set_gravity (GTK_WINDOW (ks.window), GDK_GRAVITY_CENTER);
    /*
        By calling row_selected (), settings for menu- and toolbar are adjusted.
        The following widgets are hidden by this function:
        * ks.separator_change_values_label_top
        * ks_change_values_label
        * ks.separator_change_values_label_bottom
        * ks.mandatory
        * ks.separator
        * ks.change_values_buttons_box
        * ks.new_action_option_widgets[INSIDE_MENU_LABEL]
        * ks.new_action_option_widgets[INSIDE_MENU_CHECK_BUTTON]
        * ks.new_action_option_widgets[INCLUDING_ACTION_LABEL]
        * ks.new_action_option_widgets[INCLUDING_ACTION_CHECK_BUTTON]
    */
    row_selected ();
    /*
        ks.find_box is not hidden by row_selected (). It would be hidden later if there is a menu file, 
        but since that might not be the case, for simplicity it is done here anyway.
    */
    gtk_widget_hide (ks.find_box);

#if !(GTK_CHECK_VERSION(2,10,0))
    gtk_widget_hide (ks.mb_view_and_options[SHOW_TREE_LINES]);
    gtk_widget_hide (mb_show_grid);
#endif

    // --- Load settings and standard menu file, if existent. ---


    settings_file_path = g_strconcat (g_getenv ("HOME"), "/.kickshawrc", NULL);
    if (G_LIKELY (g_file_test (settings_file_path, G_FILE_TEST_EXISTS))) {
        GKeyFile *settings_file = g_key_file_new ();
        GError *settings_file_error = NULL;

        if (G_LIKELY (g_key_file_load_from_file (settings_file, settings_file_path, 
                      G_KEY_FILE_KEEP_COMMENTS, &settings_file_error))) {
            gchar *comment = NULL; // Initialization avoids compiler warning.
            // Versions up to 0.5.7 don't have a comment in the settings file.
            if (G_LIKELY ((comment = g_key_file_get_comment (settings_file, NULL, NULL, NULL)))) { 
                for (view_and_opts_cnt = 0; view_and_opts_cnt < NUMBER_OF_VIEW_AND_OPTIONS; view_and_opts_cnt++) {
                    gtk_check_menu_item_set_active 
                        (GTK_CHECK_MENU_ITEM (ks.mb_view_and_options[view_and_opts_cnt]), 
                                              g_key_file_get_boolean (settings_file, 
                                              (view_and_opts_cnt < CREATE_BACKUP_BEFORE_OVERWRITING_MENU) ? "VIEW" : "OPTIONS", 
#if GTK_CHECK_VERSION(2,16,0)
                                              gtk_menu_item_get_label ((GtkMenuItem *) ks.mb_view_and_options[view_and_opts_cnt]), 
#else
                                              ks.menu_item_txts[view_and_opts_cnt], 
#endif
                                              NULL));
                }
                // Cleanup
                g_free (comment);
            }
            else {
                GtkWidget *dialog;

                create_dialog (&dialog, "Settings reset due to change in settings setup", 
                               GTK_STOCK_DIALOG_INFO, 
                               "Due to a change in the setup of the settings the settings file has been rewritten "
                               "and the settings have been reset to defaults. Use \"View\" and \"Options\" " 
                               "in the menu bar to readjust the settings.",  
                               "Close", NULL, NULL, TRUE);

                write_settings ();

                gtk_dialog_run (GTK_DIALOG (dialog));
                gtk_widget_destroy (dialog);   
            }
        }
        else {
            gchar *errmsg_txt = g_strdup_printf ("<b>Could not open settings file</b>\n<tt>%s</tt>\n<b>for reading!\n\n"
                                                 "<span foreground='darkred'>%s</span></b>", 
                                                 settings_file_path, settings_file_error->message);

            show_errmsg (errmsg_txt);

            // Cleanup
            g_error_free (settings_file_error);
            g_free (errmsg_txt);
        }

        // Cleanup
        g_key_file_free (settings_file);

    }
    else {
        write_settings ();
    }
    // Cleanup
    g_free (settings_file_path);


    // --- Load standard menu file, if existent. ---


    if (G_LIKELY (g_file_test (new_filename = g_strconcat (g_getenv ("HOME"), "/.config/openbox/menu.xml", NULL), 
                  G_FILE_TEST_EXISTS))) {
        get_menu_elements_from_file (new_filename);
    }
    else {
        // Cleanup
        g_free (new_filename);
    }

    /*
        This is another "convenience call" of row_selected (), this time for setting the sensivity of    
        * mb_file_menu_items[MB_NEW]
        * tb[TB_NEW]
    */
    row_selected ();

    gtk_widget_grab_focus (ks.treeview);
}

/* 

    Creates a label for an "Add new" button, adds it to a grid with additional margin, which is added to the button.

*/

static void add_button_content (GtkWidget *button, gchar *label_text)
{
    GtkWidget *box = gtk_hbox_new (FALSE, 0);
    GtkWidget *label = (*label_text) ? gtk_label_new_with_mnemonic (label_text) : ks.bt_add_action_option_label;

    gtk_container_set_border_width (GTK_CONTAINER (box), 2);
    gtk_container_add (GTK_CONTAINER (box), label);
    gtk_container_add (GTK_CONTAINER (button), box);
}

/*

    Packs an entry field into an alignment widget and event box.

*/

static void pack_entry_field (GtkWidget *widget, guint8 index)
{
    guint left_padding;

    switch (index) {
        case MENU_ELEMENT_ENTRY_HL:
        case MENU_ID_OR_VALUE_ENTRY_HL:
        case EXECUTE_ENTRY_HL:
            left_padding = 0;
            break;
        default:
            left_padding = 2;
    }

    ks.highlighted_widgets[index][ALIGNMENT] = gtk_alignment_new (0.0, 0.0, 1.0, 1.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (ks.highlighted_widgets[index][ALIGNMENT]), 0, 0, left_padding, 0);
    ks.highlighted_widgets[index][EVENTBOX] = gtk_event_box_new ();
    gtk_container_add (GTK_CONTAINER (ks.highlighted_widgets[index][ALIGNMENT]), widget);
    gtk_container_add (GTK_CONTAINER (ks.highlighted_widgets[index][EVENTBOX]), ks.highlighted_widgets[index][ALIGNMENT]);
}

/* 

    Sets attributes like foreground and background colour, visibility of cell renderers and 
    editability of cells according to certain conditions. Also highlights search results.

*/

static void set_column_attributes (G_GNUC_UNUSED GtkTreeViewColumn *cell_column, 
                                                 GtkCellRenderer   *txt_renderer,
                                                 GtkTreeModel      *cell_model, 
                                                 GtkTreeIter       *cell_iter, 
                                                 gpointer           column_number_pointer)
{
    GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (ks.treeview));

    gboolean row_is_selected = gtk_tree_selection_iter_is_selected (selection, cell_iter);

    guint column_number = GPOINTER_TO_UINT (column_number_pointer);
    GtkTreePath *cell_path = gtk_tree_model_get_path (cell_model, cell_iter);
    guint cell_data_icon_img_status;
    gchar *cell_data[NUMBER_OF_TXT_FIELDS];

    // Defaults
    gboolean visualise_txt_renderer = TRUE;
    gboolean visualise_bool_renderer = FALSE;

    // Defaults
    gchar *background = NULL;
    gboolean background_set = FALSE;
    GString *txt_with_markup = NULL;

    gboolean show_icons = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (ks.mb_view_and_options[SHOW_ICONS]));
    gboolean set_off_separators = 
        gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (ks.mb_view_and_options[SET_OFF_SEPARATORS]));
    gboolean keep_highlighting = 
        gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM 
                                        (ks.mb_view_and_options[SHOW_ELEMENT_VISIBILITY_COL_KEEP_HIGHL]));

    gtk_tree_model_get (cell_model, cell_iter, 
                        TS_ICON_IMG_STATUS, &cell_data_icon_img_status, 
                        TS_ICON_PATH, &cell_data[ICON_PATH_TXT], 
                        TS_MENU_ELEMENT, &cell_data[MENU_ELEMENT_TXT], 
                        TS_TYPE, &cell_data[TYPE_TXT], 
                        TS_VALUE, &cell_data[VALUE_TXT], 
                        TS_MENU_ID, &cell_data[MENU_ID_TXT], 
                        TS_EXECUTE, &cell_data[EXECUTE_TXT], 
                        TS_ELEMENT_VISIBILITY, &cell_data[ELEMENT_VISIBILITY_TXT], 
                        -1);

    gboolean icon_to_be_shown = show_icons && cell_data[ICON_PATH_TXT];

    /*
        Set the cell renderer type of the "Value" column to toggle if it is a "prompt" option of a non-Execute action or 
        an "enabled" option of a "startupnotify" option block.
    */

    if (column_number == COL_VALUE && STREQ (cell_data[TYPE_TXT], "option") && 
        streq_any (cell_data[MENU_ELEMENT_TXT], "prompt", "enabled", NULL)) {
        GtkTreeIter parent;
        gchar *cell_data_menu_element_parent_txt;

        gtk_tree_model_iter_parent (cell_model, &parent, cell_iter);
        gtk_tree_model_get (ks.model, &parent, TS_MENU_ELEMENT, &cell_data_menu_element_parent_txt, -1);
        if (!STREQ (cell_data_menu_element_parent_txt, "Execute")) {
            visualise_txt_renderer = FALSE;
            visualise_bool_renderer = TRUE;
            g_object_set (ks.renderers[BOOL_RENDERER], "active", STREQ (cell_data[VALUE_TXT], "yes"), NULL);
        }
        // Cleanup
        g_free (cell_data_menu_element_parent_txt);
    }

    g_object_set (txt_renderer, "visible", visualise_txt_renderer, NULL);
    g_object_set (ks.renderers[BOOL_RENDERER], "visible", visualise_bool_renderer, NULL);
    g_object_set (ks.renderers[PIXBUF_RENDERER], "visible", icon_to_be_shown, NULL);
    g_object_set (ks.renderers[EXCL_TXT_RENDERER], "visible", 
                  G_UNLIKELY (icon_to_be_shown && cell_data_icon_img_status), NULL);

    /*
        If the icon is one of the two built-in types that indicate an invalid path or icon image, 
        set two red exclamation marks behind it to clearly distinguish this icon from icons of valid image files.
    */ 
    if (G_UNLIKELY (column_number == COL_MENU_ELEMENT && cell_data_icon_img_status)) {
        g_object_set (ks.renderers[EXCL_TXT_RENDERER], "markup", "<span foreground='darkred'><b>!!</b></span>", NULL);
    }

    // Emphasis that a menu, pipe menu or item has no label (=invisible).
    if (G_UNLIKELY (column_number == COL_MENU_ELEMENT && 
                    streq_any (cell_data[TYPE_TXT], "menu", "pipe menu", "item", NULL) && !cell_data[MENU_ELEMENT_TXT])) {
        g_object_set (txt_renderer, "text", "(No label)", NULL);
    }

    if (keep_highlighting) {
        guint8 visibility_of_parent = NONE_OR_VISIBLE_ANCESTOR; // Default
        if (G_UNLIKELY ((cell_data[ELEMENT_VISIBILITY_TXT] && !STREQ (cell_data[ELEMENT_VISIBILITY_TXT], "visible")) || 
                        (visibility_of_parent = check_if_invisible_ancestor_exists (cell_model, cell_path)))) {
            background = ((cell_data[ELEMENT_VISIBILITY_TXT] && 
                          g_str_has_suffix (cell_data[ELEMENT_VISIBILITY_TXT], "orphaned menu")) || 
                          visibility_of_parent == INVISIBLE_ORPHANED_ANCESTOR) ? "#364074" : "#656772";
            background_set = TRUE;
        }
    }

    // If a search is going on, highlight all matches.
#if GTK_CHECK_VERSION(2,18,0)
    if (*ks.search_term->str && column_number < COL_ELEMENT_VISIBILITY && !gtk_widget_get_visible (ks.action_option_box) && 
#else
    if (*ks.search_term->str && column_number < COL_ELEMENT_VISIBILITY && !GTK_WIDGET_VISIBLE (ks.action_option_box) && 
#endif
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ks.find_in_columns[column_number])) && 
        check_for_match (cell_iter, column_number)) {

        enum { KS_START_POS, KS_END_POS };
        GArray *positions[2];
#if GLIB_CHECK_VERSION(2,14,0)
        positions[KS_START_POS] = g_array_new (FALSE, FALSE, sizeof (gint));
        positions[KS_END_POS] = g_array_new (FALSE, FALSE, sizeof (gint));
#else
        positions[KS_START_POS] = g_array_new (FALSE, FALSE, sizeof (regoff_t));
        positions[KS_END_POS] = g_array_new (FALSE, FALSE, sizeof (regoff_t));
#endif

        // cell_data starts with ICON_PATH, but this is not part of the treeview.
        gchar *original_cell_txt = cell_data[column_number + 1]; // Only introduced for better readability.
        gchar *search_term_str_escaped = (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ks.find_regular_expression))) ? 
                                         NULL : g_regex_escape_string (ks.search_term->str, -1);

#if GLIB_CHECK_VERSION(2,14,0)
        GRegex *regex = g_regex_new ((search_term_str_escaped) ? search_term_str_escaped : ks.search_term->str, 
                                     (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ks.find_match_case))) ? 
                                     0 : G_REGEX_CASELESS, 0, NULL);
        GMatchInfo *match_info;

        gint start_position, end_position;

        g_regex_match (regex, original_cell_txt, 0, &match_info);

        while (g_match_info_matches (match_info)) {
            g_match_info_fetch_pos (match_info, 0, &start_position, &end_position);
            g_array_append_val (positions[KS_START_POS], start_position);
            g_array_append_val (positions[KS_END_POS], end_position);
            g_match_info_next (match_info, NULL);
        }
#else
        regex_t regex;
        regmatch_t match_positions;

        gint error;
        guint index_counter = 0;
        regoff_t current_end_pos, start_pos, end_pos;

        regcomp (&regex, (search_term_str_escaped) ? search_term_str_escaped : ks.search_term->str, 
                 (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ks.find_match_case))) ? REG_EXTENDED : REG_EXTENDED | REG_ICASE);

        error = regexec (&regex, original_cell_txt, 1, &match_positions, 0);
        g_array_append_val (positions[KS_START_POS], match_positions.rm_so);
        g_array_append_val (positions[KS_END_POS], match_positions.rm_eo);
        while (!error) { // While matches found.
            /*
                Substring found between match_positions.rm_so and match_positions.rm_eo.
                This call to regexec() finds the next match.
            */
            current_end_pos = g_array_index (positions[KS_END_POS], regoff_t, index_counter);
            error = regexec (&regex, original_cell_txt + current_end_pos, 1, &match_positions, REG_NOTBOL);
            if (!error) {
                start_pos = current_end_pos + match_positions.rm_so;
                // g_array_append_val is a macro, that's why the arithmetics are done before.
                g_array_append_val (positions[KS_START_POS], start_pos);
                end_pos = current_end_pos + match_positions.rm_eo;
                g_array_append_val (positions[KS_END_POS], end_pos);
            }

            index_counter++;
        }
#endif

        gchar *counter_char = original_cell_txt; // counter_char will move through all the characters of original_cell_txt.
        gint counter;

        gunichar unichar;
        gchar utf8_char[6]; // Six bytes is the buffer size needed later by g_unichar_to_utf8 (). 
        gint utf8_length;
        gchar *utf8_escaped;

        txt_with_markup = g_string_new ((!background_set) ? "" : "<span foreground='white'>");

        do {
            unichar = g_utf8_get_char (counter_char);
            counter = counter_char - original_cell_txt; // pointer arithmetic

#if GLIB_CHECK_VERSION(2,14,0)
            if (counter == g_array_index (positions[KS_END_POS], gint, 0)) {
#else
            if (counter == g_array_index (positions[KS_END_POS], regoff_t, 0)) {
#endif
                txt_with_markup = g_string_append (txt_with_markup, "</span>");
                // It's simpler to always access the first element instead of looping through the whole array.
                g_array_remove_index (positions[KS_END_POS], 0);
            }
            /*
                No "else if" is used here, since if there is a search for a single character going on and  
                such a character appears double as 'm' in "command", between both m's a span tag has to be 
                closed and opened at the same position.
            */
#if GLIB_CHECK_VERSION(2,14,0)
            if (counter == g_array_index (positions[KS_START_POS], gint, 0)) {
#else
            if (counter == g_array_index (positions[KS_START_POS], regoff_t, 0)) {
#endif
                txt_with_markup = g_string_append (txt_with_markup, 
                                                  (row_is_selected) ? 
                                                  "<span background='black' foreground='white'>" : 
                                                  "<span background='yellow' foreground='black'>");
                // See the comment for the similar instruction above.
                g_array_remove_index (positions[KS_START_POS], 0);
            }

            utf8_length = g_unichar_to_utf8 (unichar, utf8_char);
            /*
                Instead of using a switch statement to check whether the current character needs to be escaped, 
                for simplicity the character is sent to the escape function regardless of whether there will be 
                any escaping done by it or not.
            */
            utf8_escaped = g_markup_escape_text (utf8_char, utf8_length);

            txt_with_markup = g_string_append (txt_with_markup, utf8_escaped);

            // Cleanup
            g_free (utf8_escaped);

            counter_char = g_utf8_find_next_char (counter_char, NULL);
        } while (*counter_char != '\0');

        /*
            There is a '</span>' to set at the end; because the end position is one position after the string size
            this couldn't be done inside the preceding loop.
        */            
        if (positions[KS_END_POS]->len) {
            g_string_append (txt_with_markup, "</span>");
        }

        if (background_set) {
            txt_with_markup = g_string_append (txt_with_markup, "</span>");
        }

        g_object_set (txt_renderer, "markup", txt_with_markup->str, NULL);

        // Cleanup
        g_free (search_term_str_escaped);
        g_array_free (positions[KS_START_POS], TRUE);
        g_array_free (positions[KS_END_POS], TRUE);
#if GLIB_CHECK_VERSION(2,14,0)
        g_regex_unref (regex);
        g_match_info_free (match_info);
#else
        regfree (&regex);
#endif
    }

    // Set foreground and background font and cell colours. Also set editability of cells.
    g_object_set (txt_renderer, "weight", (set_off_separators && STREQ (cell_data[TYPE_TXT], "separator")) ? 1000 : 400, 

                  "family", (STREQ (cell_data[TYPE_TXT], "separator") && set_off_separators) ? 
                            "monospace, courier new, courier" : "sans, sans-serif, arial, helvetica",
 
                  "foreground", "white", "foreground-set", (row_is_selected || (background_set && !txt_with_markup)),
                  "background", background, "background-set", background_set, 

                  "editable", 
                  (((column_number == COL_MENU_ELEMENT && 
                  (STREQ (cell_data[TYPE_TXT], "separator") || 
                  (cell_data[MENU_ELEMENT_TXT] &&
                  streq_any (cell_data[TYPE_TXT], "menu", "pipe menu", "item", NULL)))) || 
                  (column_number == COL_VALUE && STREQ (cell_data[TYPE_TXT], "option")) || 
                  (column_number == COL_MENU_ID && streq_any (cell_data[TYPE_TXT], "menu", "pipe menu", NULL)) ||
                  (column_number == COL_EXECUTE && STREQ (cell_data[TYPE_TXT], "pipe menu")))), 

                  NULL);

    for (guint8 renderer_cnt = EXCL_TXT_RENDERER; renderer_cnt < NUMBER_OF_RENDERERS; renderer_cnt++) {
        g_object_set (ks.renderers[renderer_cnt], "cell-background", background, "cell-background-set", background_set, NULL);
    }
#if GTK_CHECK_VERSION(2,18,0)
    if (txt_with_markup && gtk_cell_renderer_get_visible (ks.renderers[BOOL_RENDERER])) {
#else
    if (txt_with_markup && column_number == COL_VALUE && 
        STREQ (cell_data[TYPE_TXT], "option") && STREQ (cell_data[MENU_ELEMENT_TXT], "enabled")) {
#endif
        g_object_set (ks.renderers[BOOL_RENDERER], "cell-background", "yellow", "cell-background-set", TRUE, NULL);
    }

    // Cleanup
    if (txt_with_markup) {
        g_string_free (txt_with_markup, TRUE);
    }
    gtk_tree_path_free (cell_path);
    free_elements_of_static_string_array (cell_data, NUMBER_OF_TXT_FIELDS, FALSE);
}

/* 

    Changes certain aspects of the tree view or misc. settings.

*/

static void change_view_and_options (gpointer activated_menu_item_pointer)
{
    guint8 activated_menu_item = GPOINTER_TO_UINT (activated_menu_item_pointer);

    gboolean menu_item_activated = 
        (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (ks.mb_view_and_options[activated_menu_item])));

    if (activated_menu_item <= SHOW_ELEMENT_VISIBILITY_COL_ACTVTD) {
        /*
            SHOW_MENU_ID_COL (0)                   + 3 = COL_MENU_ID (3)
            SHOW_EXECUTE_COL (1)                   + 3 = COL_EXECUTE (4)
            SHOW_ELEMENT_VISIBILITY_COL_ACTVTD (2) + 3 = COL_ELEMENT_VISIBILITY (5)
        */
        guint8 column_position = activated_menu_item + 3;

        gtk_tree_view_column_set_visible (ks.columns[column_position], menu_item_activated);

        if (column_position == COL_ELEMENT_VISIBILITY) {
            if (!menu_item_activated) {
                gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM 
                                                (ks.mb_view_and_options[SHOW_ELEMENT_VISIBILITY_COL_KEEP_HIGHL]), TRUE);
            }
            gtk_widget_set_sensitive (ks.mb_view_and_options[SHOW_ELEMENT_VISIBILITY_COL_KEEP_HIGHL], menu_item_activated);
            gtk_widget_set_sensitive (ks.mb_view_and_options[SHOW_ELEMENT_VISIBILITY_COL_DONT_KEEP_HIGHL], menu_item_activated);
        }
    }
    else if (activated_menu_item == DRAW_ROWS_IN_ALT_COLOURS) {
        g_object_set (ks.treeview, "rules-hint", menu_item_activated, NULL);
    }
#if GTK_CHECK_VERSION(2,10,0)
    else if (activated_menu_item >= NO_GRID_LINES && activated_menu_item <= BOTH) {
        guint8 grid_settings_cnt, grid_lines_type_cnt;

        for (grid_settings_cnt = NO_GRID_LINES, grid_lines_type_cnt = GTK_TREE_VIEW_GRID_LINES_NONE; 
             grid_settings_cnt <= BOTH; 
             grid_settings_cnt++, grid_lines_type_cnt++) {
            if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (ks.mb_view_and_options[grid_settings_cnt]))) {
                gtk_tree_view_set_grid_lines (GTK_TREE_VIEW (ks.treeview), grid_lines_type_cnt);
                break;
            }
        }
    }
    else if (activated_menu_item == SHOW_TREE_LINES) {
        gtk_tree_view_set_enable_tree_lines (GTK_TREE_VIEW (ks.treeview), menu_item_activated);
    }
#endif
    else if (activated_menu_item == SORT_EXECUTE_AND_STARTUPN_OPTIONS) {
        if ((ks.autosort_options = menu_item_activated)) {
            gtk_tree_model_foreach (ks.model, (GtkTreeModelForeachFunc) sort_loop_after_sorting_activation, NULL);
        }
        /*
            A change of the activation status of autosorting requires 
            (de)activation of the movement arrows of the menu- and toolbar.
        */
        row_selected ();
    }
    else {
        gtk_tree_view_columns_autosize (GTK_TREE_VIEW (ks.treeview)); // If icon visibility has been switched on.
    }

    write_settings ();
}

/*

    Sets the sensivity of the expand/collapse menu items and toolbar buttons
    according to the status of the nodes of the treeview.

*/

void set_status_of_expand_and_collapse_buttons_and_menu_items (void)
{
    gboolean expansion_statuses_of_nodes[NUMBER_OF_EXPANSION_STATUSES] = { FALSE };

    gtk_tree_model_foreach (ks.model, (GtkTreeModelForeachFunc) check_expansion_statuses_of_nodes, expansion_statuses_of_nodes);

    gtk_widget_set_sensitive (ks.mb_collapse_all_nodes, expansion_statuses_of_nodes[AT_LEAST_ONE_IS_EXPANDED]);
    gtk_widget_set_sensitive ((GtkWidget *) ks.tb[TB_COLLAPSE_ALL], 
                              expansion_statuses_of_nodes[AT_LEAST_ONE_IS_EXPANDED]);
    gtk_widget_set_sensitive (ks.mb_expand_all_nodes, expansion_statuses_of_nodes[AT_LEAST_ONE_IS_COLLAPSED]);
    gtk_widget_set_sensitive ((GtkWidget *) ks.tb[TB_EXPAND_ALL], 
                              expansion_statuses_of_nodes[AT_LEAST_ONE_IS_COLLAPSED]);
}

/* 

    Dialog window that shows the program name and version, a short description, the website, author and license.

*/

static void about (void) 
{
    gtk_show_about_dialog (GTK_WINDOW (ks.window), 
                           /*
                               There is no icon that comes with Kickshaw; "logo-icon-name" instead of "logo" is 
                               used to avoid a broken default icon that would be shown in the about dialog. 
                           */
                           "logo-icon-name", NULL, 
#if GTK_CHECK_VERSION(2,12,0)
                           "program-name", "Kickshaw", 
#else
                           "name", "Kickshaw", 
#endif
                           "version", KICKSHAW_VERSION, 
                           "comments", "A menu editor for Openbox", 
                           "website", "https://savannah.nongnu.org/projects/obladi/", 
                           "website_label", "Project page at GNU Savannah", 
                           "copyright", "© 2010–2018 Marcus Schätzle", 
                           "license", "Copyright (c) 2010–2018        Marcus Schätzle\n\n"
                                      "This program is free software; you can redistribute it and/or modify\n"
                                      "it under the terms of the GNU General Public License as published by\n"
                                      "the Free Software Foundation; either version 2 of the License, or\n"
                                      "(at your option) any later version.\n\n"
                                      "This program is distributed in the hope that it will be useful,\n"
                                      "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
                                      "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the\n"
                                      "GNU General Public License for more details.\n\n"
                                      "You should have received a copy of the GNU General Public License along\n"
                                      "with Kickshaw. If not, see http://www.gnu.org/licenses/ .", 

                           "authors", (gchar *[]) { "Marcus Schätzle", NULL }, 
                           NULL);
}

/* 

    Asks whether to continue if there are unsaved changes.

*/

gboolean continue_despite_unsaved_changes (void)
{
    GtkWidget *dialog;

    gint result;

    #define CONTINUE_DESPITE_UNSAVED_CHANGES 2

    create_dialog (&dialog, "Menu has unsaved changes", GTK_STOCK_DIALOG_WARNING, 
                   "<b>This menu has unsaved changes. Continue anyway?</b>", "_Cancel", "_Yes", NULL, TRUE);
    gtk_window_resize (GTK_WINDOW (dialog), 570, 1);

    result = gtk_dialog_run (GTK_DIALOG (dialog));

    gtk_widget_destroy (dialog);

    return (result == CONTINUE_DESPITE_UNSAVED_CHANGES);
}

/* 

    Clears the data that is held throughout the running time of the program.

*/

void clear_global_data (void)
{
    GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (ks.treeview));

    FREE_AND_REASSIGN (ks.filename, NULL);
#if GLIB_CHECK_VERSION(2,28,0)
    g_slist_free_full (ks.menu_ids, (GDestroyNotify) g_free);
#else
    g_slist_foreach (ks.menu_ids, (GFunc) g_free, NULL);
    g_slist_free (ks.menu_ids);
#endif
    ks.menu_ids = NULL;
    if (ks.rows_with_icons) {
        stop_timeout ();
    }
#if GTK_CHECK_VERSION(2,18,0)
    if (gtk_widget_get_visible (ks.find_box)) {
#else
    if (GTK_WIDGET_VISIBLE (ks.find_box)) {
#endif
        show_or_hide_find_box ();
    }
    g_signal_handler_block (selection, ks.handler_id_row_selected);
    gtk_tree_store_clear (ks.treestore);
    g_signal_handler_unblock (selection, ks.handler_id_row_selected);
    ks.statusbar_msg_shown = FALSE;
}

/* 

    Clears treestore and global variables, also resets window title and menu-/tool-/button bar widgets accordingly.

*/

static void new_menu (void)
{
    if (ks.change_done && !continue_despite_unsaved_changes ()) {
        return;
    }

    gtk_window_set_title (GTK_WINDOW (ks.window), "Kickshaw");

    clear_global_data ();
    ks.change_done = FALSE;

    gtk_tree_view_columns_autosize (GTK_TREE_VIEW (ks.treeview));
    row_selected (); // Switches the settings for menu- and toolbar to that of an empty menu.
}

/* 

    Quits program, if there are unsaved changes a confirmation dialog window is shown.

*/

static void quit_program (gpointer ask_before_quitting)
{
    if (GPOINTER_TO_UINT (ask_before_quitting) && ks.change_done && !continue_despite_unsaved_changes ()) {
        return;
    }

    unlink (ks.lock_file_path);
    flock (fileno (ks.lock_file), LOCK_UN);

    exit (EXIT_SUCCESS);
}

/* 

    Activates "Save" menubar item/toolbar button (provided that there is a filename) if a change has been done.
    Also sets a global veriable so a program-wide check for a change is possible.
    If a search is still active, the list of results is recreated.     
    A list of rows with icons is (re)created, too.

*/

void activate_change_done (void)
{
    if (ks.filename) {
        gtk_widget_set_sensitive (ks.mb_file_menu_items[MB_SAVE], TRUE);
        gtk_widget_set_sensitive ((GtkWidget *) ks.tb[TB_SAVE], TRUE);
    }

    if (*ks.search_term->str) {
        create_list_of_rows_with_found_occurrences ();
    }

    if (ks.rows_with_icons) {
        stop_timeout ();
    }
    create_list_of_icon_occurrences ();

    ks.change_done = TRUE;
}

/* 

    Writes all view and option settings into a file.

*/

static void write_settings (void)
{
    FILE *settings_file;
    gchar *settings_file_path = g_strconcat (g_getenv ("HOME"), "/.kickshawrc", NULL);

    if (G_UNLIKELY (!(settings_file = fopen (settings_file_path, "w")))) {
        gchar *errmsg_txt = g_strdup_printf ("<b>Could not open settings file</b>\n<tt>%s</tt>\n<b>for writing!</b>", 
                                             settings_file_path);

        show_errmsg (errmsg_txt);

        // Cleanup
        g_free (errmsg_txt);
        g_free (settings_file_path);

        return;
    }

    // Cleanup
    g_free (settings_file_path);

    fputs ("# Generated by Kickshaw " KICKSHAW_VERSION "\n\n[VIEW]\n\n", settings_file);

    for (guint8 view_and_opts_cnt = 0; view_and_opts_cnt < NUMBER_OF_VIEW_AND_OPTIONS; view_and_opts_cnt++) {
        if (view_and_opts_cnt == CREATE_BACKUP_BEFORE_OVERWRITING_MENU) {
            fputs ("\n[OPTIONS]\n\n", settings_file);
        }
#if GTK_CHECK_VERSION(2,16,0)
        fprintf (settings_file, "%s=%s\n", gtk_menu_item_get_label ((GtkMenuItem *) ks.mb_view_and_options[view_and_opts_cnt]), 
#else
        fprintf (settings_file, "%s=%s\n", ks.menu_item_txts[view_and_opts_cnt], 
#endif
                 (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (ks.mb_view_and_options[view_and_opts_cnt]))) ? 
                 "true" : "false");
    }

    fclose (settings_file);
}

/* 

    Replaces the filename and window title.

*/

void set_filename_and_window_title (gchar *new_filename)
{
    gchar *basename;
    gchar *window_title;

    FREE_AND_REASSIGN (ks.filename, new_filename);
    basename = g_path_get_basename (ks.filename);
    window_title = g_strconcat ("Kickshaw - ", basename, NULL);
    gtk_window_set_title (GTK_WINDOW (ks.window), window_title);

    // Cleanup
    g_free (basename);
    g_free (window_title);
}

/* 

    Shows an error message dialog.

*/

void show_errmsg (gchar *errmsg_raw_txt)
{
    GtkWidget *dialog;
    gchar *label_txt = g_strdup_printf ((!strstr (errmsg_raw_txt, "<b>")) ? "<b>%s</b>" : "%s", errmsg_raw_txt);

    create_dialog (&dialog, "Error", GTK_STOCK_DIALOG_ERROR, label_txt, "_Close", NULL, NULL, TRUE);

    // Cleanup
    g_free (label_txt);

    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
}

/* 

    Shows a message in the statusbar at the botton for information.

*/

void show_msg_in_statusbar (gchar *message)
{
#if GTK_CHECK_VERSION(2,22,0)
    gtk_statusbar_remove_all (GTK_STATUSBAR (ks.statusbar), 1);
    gtk_statusbar_push (GTK_STATUSBAR (ks.statusbar), 1, message); // Only one context (indicated by 1) with one message.
#else
    if (ks.message_ID) {
        gtk_statusbar_remove (GTK_STATUSBAR (ks.statusbar), 1, ks.message_ID);
    }
    ks.message_ID = gtk_statusbar_push (GTK_STATUSBAR (ks.statusbar), 1, message); // Only one context (indicated by 1) with one message.
#endif

    ks.statusbar_msg_shown = TRUE;
}
