/*
--          This file is part of the New World OS and Objectify projects
--         Copyright (C) 2004, 2005, 2006, 2007, 2009, 2012  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--  In addition, as a special exception, QRW Software Company gives permission
--  to link the code of this program with any version of the OpenSSL library
--  which is distributed under a license identical to that listed in the
--  included LICENSE.OpenSSL file, and distribute linked combinations including
--  the two.  You must obey the GNU General Public License in all respects
--  for all of the code used other than OpenSSL.  If you modify this file,
--  you may extend this exception to your version of the file, but you are not
--  obligated to do so.  If you do not wish to do so, delete this exception
--  statement from your version.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2012-04-17 22:39:49 -0600 (Tue, 17 Apr 2012) $
--   $Revision: 5002 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Since this file was originally
--   copied from the file in the root directory I have left the CVS log from
--   that file below.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
--
-- Revision 1.49  2006/12/01 05:16:46  jsedwards
-- Added a test in get_object_class to make sure it is not a reference list
-- before trying to compute the header checksum.
--
-- Revision 1.48  2006/12/01 05:11:48  jsedwards
-- Add a strlcpy function only if compiling on Linux because it doesn't have
-- it in the library.
--
-- Revision 1.47  2006/11/27 14:28:26  jsedwards
-- Added test for unencrypted objects in read_and_decrypt and write_and_encrypt
-- functions and skip over encryption stuff if set.
--
-- Revision 1.46  2006/11/27 13:46:46  jsedwards
-- Added functions to encode and decode variable sized counts.
--
-- Revision 1.45  2006/11/19 13:44:30  jsedwards
-- Add time stamp to disk header so we can tell when it was last changed.
--
-- Revision 1.44  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.43  2006/11/18 14:26:47  jsedwards
-- Changed to read and write objects larger than FILE_BLOCK_SIZE (256 bytes).
--
-- Revision 1.42  2006/11/17 12:18:11  jsedwards
-- Added "read_only" flag to keep it from trying to write the header back when
-- read only.
--
-- Revision 1.41  2006/11/11 15:20:37  jsedwards
-- Moved all reference list code into a separate file: reference_list.c.
--
-- Revision 1.40  2006/11/11 14:24:37  jsedwards
-- Made nwos_malloc and nwos_free public (not static) and moved
-- find_class_definiton function to class_definition.c.
--
-- Revision 1.39  2006/11/11 13:58:00  jsedwards
-- Add new generate_new_public_id function.
--
-- Revision 1.38  2006/11/11 13:34:32  jsedwards
-- Made disk header a static global that is read in initialize(), updated with
-- next public reference and written back out in terminate().  Also added a
-- new global variable with the next public reference.
--
-- Revision 1.37  2006/11/11 12:01:05  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.36  2006/11/10 13:39:16  jsedwards
-- Uncommented write_block in write_public_object_to_disk and removed print
-- statements.
--
-- Revision 1.35  2006/11/10 13:32:05  jsedwards
-- Moved Ref_List_First_Block and Ref_List_Extra_Block definitions from
-- objectify.c to objectify_private.h so that import_public_csv can use
-- them also.
--
-- Revision 1.34  2006/11/10 04:11:02  jsedwards
-- Add "write_public_object_to_disk" function and fix hash to deal with
-- public references.
--
-- Revision 1.33  2006/11/04 18:51:07  jsedwards
-- Made a #define in the config.h file for the log file.
--
-- Revision 1.32  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.31  2006/10/26 01:22:51  jsedwards
-- Replace the main truck version (1.30) with the latest version from the
-- alpha_05_branch.
--
-- Revision 1.24.2.57  2006/10/25 12:22:29  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.24.2.56  2006/10/22 13:05:16  jsedwards
-- Changed so that blocks on disk and reserved public blocks is variable
-- based on values store in disk header.
--
-- Revision 1.24.2.55  2006/10/22 12:40:47  jsedwards
-- Corrected version string error message.
--
-- Revision 1.24.2.54  2006/10/22 00:16:43  jsedwards
-- More fixes for the changing to the separate next_block_ptr to the add to
-- reference list and write reference list.
--
-- Revision 1.24.2.53  2006/10/20 13:25:58  jsedwards
-- Change to have a separate next_block_ptr (for linked list) instead of
-- using the first 4 (unused) bytes of each reference list block as the
-- pointer to the next block.  This wasn't happy when pointers aren't 4 bytes.
--
-- Revision 1.24.2.52  2006/10/17 12:38:37  jsedwards
-- Changed formats in printf statementss to be more compatible with different
-- types (size_t, off64_t) being different sizes.  Also changed format for
-- uint32 to be unsigned instead of long unsigned (unsigned is 32 bits on
-- both 32 and 64 bit machines, whereas long unsigned is 64 on amd64).
--
-- Revision 1.24.2.51  2006/10/16 13:01:05  jsedwards
-- Add code to rewind disk after indexing a compressed file so reading the
-- header works correctly.
--
-- Revision 1.24.2.50  2006/10/14 13:13:45  jsedwards
-- Added code to check the disk header and make sure it is a nwos disk, the
-- version string matches, and the disk size matches what was compiled in.
--
-- Revision 1.24.2.49  2006/10/13 11:47:44  jsedwards
-- Comment out print statement of setting bits in block maps.
--
-- Revision 1.24.2.48  2006/10/12 12:12:22  jsedwards
-- Fixed bug in last ref for generating references in sequence.  Also added
-- function to flush dirty block maps back to disk.
--
-- Revision 1.24.2.47  2006/10/11 13:24:50  jsedwards
-- Add code to set the bits in the bit maps when a block is written.  Note:
-- this version does not use the bit maps for anything.
--
-- Revision 1.24.2.46  2006/10/10 07:30:35  jsedwards
-- Change to 8192 bytes per bit map instead of 256 bytes.
--
-- Revision 1.24.2.45  2006/10/07 22:27:02  jsedwards
-- Changed the generate id function so that it checks to see if we are
-- operating in the sequential mode and it calls the correct function.
--
-- Revision 1.24.2.44  2006/10/06 04:39:24  jsedwards
-- Changed the names of some of the public and private, blocks and id
-- definitions and added new function to generate ids that scan across
-- the disk drive, instead of totally random.
--
-- Revision 1.24.2.43  2006/10/03 12:50:16  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.24.2.42  2006/10/02 14:02:45  jsedwards
-- Changed to just read the file while indexing, instead of doing the seeks.
--
-- Revision 1.24.2.41  2006/10/02 13:29:54  jsedwards
-- Changed to use a binary search for the indexes instead of the linear.
--
-- Revision 1.24.2.40  2006/10/02 12:33:44  jsedwards
-- Split out reference hash into a separate function and initialized next_ptr
-- so that gcc doesn't worry about it possibly being uninitialized.
--
-- Revision 1.24.2.39  2006/10/02 02:15:33  jsedwards
-- Added the capability of reading blocks from a compressed archive file
-- instead of the normal disk drive or partition.
--
-- Revision 1.24.2.38  2006/10/01 11:49:47  jsedwards
-- Increased the number of tries in finding a random id.  Made the code to
-- not access the hard drive for testing purposes ifdef'd so it can be enabled
-- or disabled with a -D option.
--
-- Revision 1.24.2.37  2006/09/29 04:12:39  jsedwards
-- Fixed "object_exists" and "block_used" functions again.  Added if'd out
-- code to write to memory instead of the disk (for testing).
--
-- Revision 1.24.2.36  2006/09/28 13:38:46  jsedwards
-- Re-arranged testing of reference in block to do object exists better.
-- Added printing of offset in lseek error statements.
--
-- Revision 1.24.2.35  2006/09/26 13:22:38  jsedwards
-- Changed to use a disk drive or partition for storage instead of a file.
-- This required hashing the id to fit in the disk or partition size.
--
-- Revision 1.24.2.34  2006/09/24 13:46:14  jsedwards
-- Previous change was a mistake, added a new function that checks to see if
-- the block is used and changed "object_exists" function back to the way it
-- was.
--
-- Revision 1.24.2.33  2006/09/24 13:38:06  jsedwards
-- Add test to "object_exists" function to verify it is not just a zero block.
--
-- Revision 1.24.2.32  2006/09/19 14:10:45  jsedwards
-- Fix bug where next block wasn't zeroed when reference list sidecar filled
-- up exactly.  Added printing of num_blocks in log.
--
-- Revision 1.24.2.31  2006/09/18 02:07:04  jsedwards
-- Add asserts to make sure the reference passed into routines is not void.
--
-- Revision 1.24.2.30  2006/09/18 01:32:52  jsedwards
-- Renamed read_file_and_decrypt and write_file with "nwos" prefix and removed
-- the static so that they can be accessed by other modules (like file.c).
--
-- Revision 1.24.2.29  2006/09/09 13:03:33  jsedwards
-- Moved "create_root_object" routine from security.c to objectify.c so that
-- the security module didn't have references to so many other parts of the
-- system.
--
-- Revision 1.24.2.28  2006/09/08 13:28:41  jsedwards
-- Changed to use a different sequence block for objects beyond the first for
-- objects larger than one block.
--
-- Revision 1.24.2.27  2006/09/08 11:07:34  jsedwards
-- Changed so that random_sequence table is a two dimentional array so we can
-- store different sequence tables for multiple block objects.
--
-- Revision 1.24.2.26  2006/09/08 11:01:23  jsedwards
-- Changed so that the sequence table is passed into "write_object_to_disk"
-- and "read_object_from_disk_and_decrypt" routines.
--
-- Revision 1.24.2.25  2006/09/07 12:59:46  jsedwards
-- Separated the sequence table generation code into a separate function.
--
-- Revision 1.24.2.24  2006/09/06 13:15:25  jsedwards
-- Removed nwos_seed_sequence function and instead pass pointers to the values
-- in the call to next_sequence function.
--
-- Revision 1.24.2.23  2006/09/05 12:59:34  jsedwards
-- Added checksums (crc32) to reference lists.
--
-- Revision 1.24.2.22  2006/09/05 01:26:41  jsedwards
-- Converted add_to_reference_list to use the new block structures.
--
-- Revision 1.24.2.21  2006/09/05 00:29:16  jsedwards
-- Converted the read_reference_list_from_disk routine to use the new block
-- structures.
--
-- Revision 1.24.2.20  2006/09/05 00:07:48  jsedwards
-- Converted the write_reference_list routine to use the new block structures.
--
-- Revision 1.24.2.19  2006/09/04 15:51:14  jsedwards
-- Change read_reference_list_into_cache routine to use new block structures.
--
-- Revision 1.24.2.18  2006/09/04 13:26:06  jsedwards
-- Added reference list block structures (for the blocks written to disk).
-- Changed cache to point to a reference list first block structure instead
-- of a reference list itself.  The code really doesn't use them yet.
--
-- Revision 1.24.2.17  2006/09/04 01:01:24  jsedwards
-- Split up the create_reference_list routine so it is possible to create a
-- reference list with a given id instead of generating a new one.  This is
-- helpful when doing the conversion from the old multiple file style (we
-- can just keep the old reference ids and not have to switch them around).
--
-- Revision 1.24.2.16  2006/09/03 13:56:35  jsedwards
-- Moved defintion of DEFAULT_FILE to objectify_private.h so the big_bang can
-- use it to create the file.
--
-- Revision 1.24.2.15  2006/09/03 13:14:55  jsedwards
-- Rearranged the order of functions in the file and added some comments to
-- mark the different sections.  NO code changes were made.
--
-- Revision 1.24.2.14  2006/09/03 12:20:40  jsedwards
-- Removed a couple more debugging print statements.
--
-- Revision 1.24.2.13  2006/09/03 12:13:57  jsedwards
-- Add code to check that all malloc'd blocks are freed during testing.
--
-- Revision 1.24.2.12  2006/09/03 11:09:14  jsedwards
-- Removed many of the debugging print statements, added a routine to allow
-- an object to be overwritted (instead of having to remove it and then write
-- it again.  Also rearranged the checksum checking code slightly.
--
-- Revision 1.24.2.11  2006/09/03 10:22:14  jsedwards
-- Initialize ivec in nwos_get_object_class function.
--
-- Revision 1.24.2.10  2006/09/02 15:08:16  jsedwards
-- Add reference list class reference to root object because now it has to
-- read the reference list to get it's size and we can't easily read it
-- without the read routine verifying it's class.
--
-- Fixed add reference routine to clean dirty flag before writing block.
--
-- Fixed remove object routine to do the seek before nuking the object.
--
-- Revision 1.24.2.9  2006/09/01 13:20:00  jsedwards
-- Fixed some bugs in new storing objects in one file.  Still NOT working!
--
-- Revision 1.24.2.8  2006/09/01 11:49:59  jsedwards
-- Merged version with file additions and version that writes objects into one
-- large file.  This vesion DOES NOT work correctly.
--
-- Revision 1.24.2.7  2006/08/25 13:06:54  jsedwards
-- Fixed daylight savings to work correctly when computing the end of time.
--
-- Revision 1.24.2.6  2006/08/25 12:43:37  jsedwards
-- Change to compute the end of time value in local time and save it.  This
-- way the time stamps on the file actually read 9/9/99...
--
-- Revision 1.24.2.5  2006/08/25 12:24:00  jsedwards
-- Added code to set the time stamps on all of the objects to 9/9/99 9:09:09.
--
-- Revision 1.24.2.4  2006/08/16 12:47:56  jsedwards
-- Changed the name of the static function "log" to "nwos_log" and added a
-- cast so that new versions of GCC don't complain.
-- doesn't complain about the type of a built in function being redefined
--
-- Revision 1.24.2.3  2006/08/13 16:36:43  jsedwards
-- Made a define for the block size of the file called FILE_BLOCK_SIZE, that
-- is the minimum size of the file for each object.  The amount of disk space
-- used for a one byte file.
--
-- Revision 1.24.2.2  2006/08/13 16:27:14  jsedwards
-- Change to compute the random sequence table only one time and save it.
--
-- Revision 1.24.2.1  2006/08/13 12:53:06  jsedwards
-- Added comments to encrypt and decrypt routines (NO code changes).
--
-- Revision 1.24  2006/01/12 02:57:28  jsedwards
-- Fixed "ask yes or no" routine to actually work.
--
-- Revision 1.23  2006/01/12 02:03:36  jsedwards
-- Added code to dump the object when the data checksum fails.
--
-- Revision 1.22  2006/01/08 15:27:28  jsedwards
-- Removed unused variable.
--
-- Revision 1.21  2006/01/03 03:21:38  jsedwards
-- Added "is compatible version" routine to determine if a particular version
-- of an object is still compatible with the current version.
--
-- Revision 1.20  2006/01/01 19:47:33  jsedwards
-- Added "is quit command" and "ask yes or no" routines.
--
-- Revision 1.19  2005/12/30 14:09:50  jsedwards
-- Commented out printing of blowfish key, was for debugging.
--
-- Revision 1.18  2005/12/29 19:08:00  jsedwards
-- Fixed some of the ifdef DISABLE_SECURITY_FEATURES that were accidentally
-- entered as #if DISABLE_SECURITY_FEATURES instead.  Also added a disable
-- security features to the unique id routine to make sure it finds a unique
-- one by incrementing (since we can punch the random number generator all
-- we want and it isn't going to fix that problem).
--
-- Revision 1.17  2005/12/29 18:30:42  jsedwards
-- Added ifdefs to disable/enable the security features.
--
-- Revision 1.16  2005/12/29 18:20:24  jsedwards
-- Split up routines, added read variable length objects and other changes to
-- make encryption work, HOWEVER, ENCRYPTION IS DISABLED IN THIS VERSION.
--
-- Revision 1.15  2005/12/28 12:53:06  jsedwards
-- First attempt at adding security.  Objects are encrypted before being
-- written to disk and decrypted when read from disk.  It is incomplete at
-- this point and doesn't work completely, for example, the read object
-- headers function won't work.  This version should only be used for testing.
--
-- Revision 1.14  2005/12/27 19:44:40  jsedwards
-- Added routine to set the root object.
--
-- Revision 1.13  2005/12/27 18:14:08  jsedwards
-- Changed ObjRefs to be 4 bytes instead of 8 bytes and made them random
-- instead of somehow based upon the contents.
--
-- Revision 1.12  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.11  2005/12/21 23:25:57  jsedwards
-- Added better error messages when the size of an object doesn't match the
-- number of bytes read.
--
-- Revision 1.10  2005/12/11 16:52:56  jsedwards
-- Added a log message when an object is added to a reference list and fixed
-- the error messages printed when adding to a reference list fails.
--
-- Revision 1.9  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.8  2005/12/05 19:03:06  jsedwards
-- Moved local read_object_from_disk (not nwos_read_object_from_disk) function
-- earlier in the file so that get_object_class function can call it instead
-- of nwos_read_object_from_disk which wasn't happy about it just reading the
-- common header.
--
-- Revision 1.7  2005/12/05 05:18:53  jsedwards
-- Ifdefd out logging each time a file is read, too much spam into log file.
--
-- Revision 1.6  2005/12/05 05:16:22  jsedwards
-- Made a separate routine 'read_reference_list_from_disk' to read reference
-- lists, that doesn't do checksums.  Changed 'read_object_from_disk' to
-- check both the header and data checksums.  Also added a log routine to
-- log each time a file is written to and read from disk.
--
-- Revision 1.5  2005/12/03 22:15:38  jsedwards
-- Added get_object_class and add_to_references functions.
--
-- Revision 1.4  2005/12/02 20:30:11  jsedwards
-- Changed so that reference list id is generated from the object that the
-- reference list is for's id.
--
-- Revision 1.3  2005/11/30 13:53:31  jsedwards
-- Added routines to get object size and read class definition.  Also added
-- 'nwos' prefix to generate_new_id routine and made it global.
--
-- Revision 1.2  2005/11/26 15:36:59  jsedwards
-- Added 'nwos' prefix to functions and added 'static' to local functions.
--
-- Revision 1.1.1.1  2005/11/25 12:44:27  jsedwards
-- Copied from 'lab'.
--
-- Revision 1.5  2005/11/24 15:47:47  jsedwards
-- Changed to match new object header layouts.
--
-- Revision 1.4  2005/11/22 13:57:04  jsedwards
-- Made a common subroutine to fill in the common header.
--
-- Revision 1.3  2005/11/22 13:09:58  jsedwards
-- Added read object from disk and create simple object routines.
--
-- Revision 1.2  2005/11/12 23:02:58  jsedwards
-- Rearranged headers.
--
-- Revision 1.1  2005/11/12 14:55:12  jsedwards
-- Routines for creating objects.
--
*/

/*---------------------------------------------------------------------------*/
/* The new plan for storing objects:                                         */
/*                                                                           */
/* Instead of storing objects in separate files (as was done previously),    */
/* the plan is to store them all in one sparse file.                         */
/*                                                                           */
/* One of the major problems with the previous plan was how to cope with     */
/* block sizes of all the different file systems.  I was using Resier4       */
/* because it's block size was 512 bytes and so if you created a file that   */
/* was only say 92 bytes you only wasted 430 bytes.  However when it was     */
/* used on an ext3 file system with 4096 blocks you wasted 4004 bytes.       */
/* I also suspsect there may have been more problems when there got to be    */
/* a large number of files with exceeding the number that were possible      */
/* in a directory and running out of inodes.                                 */
/*                                                                           */
/* So the new plan is to store all of the objects in a sparse file, indexed  */
/* by their ID number.  My current theory then is to use 256 byte blocks     */
/* for each object.  The first 8 bytes of each block is it's ID number.      */
/* If the first 8 bytes of the block are zeros then that block is unused.    */
/*                                                                           */
/* Variable sized objects that are larger than 256 bytes use multiple        */
/* blocks.  The last 4 bytes of the block point to the next block.           */
/*---------------------------------------------------------------------------*/


#define _LARGEFILE64_SOURCE

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <openssl/blowfish.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"
#include "time_stamp.h"


/* Global variables... BAD!... need to fix */

ObjRef nwos_class_definition_class_ref;
ObjRef nwos_reference_list_class_ref;
ObjRef nwos_next_public_ref;

static uint8  disk_header[32];
static uint32 linear_after_first;
static uint32 serial_after_first;
       uint8  nwos_random_sequence[NUM_STORED_SEQ][FILE_BLOCK_SIZE];     /* this will only work for blocks <= 256 bytes */
static BF_KEY blowfish_key;

static int    obj_file_desc;
static uint32 blocks_on_disk;
static uint32 reserved_public_blocks;
static uint32 private_blocks_on_disk;

static bool    read_only;
static bool    modified;
static char*   storage_path;static ObjRef* storage_index;
static int     storage_count;


static struct timeval end_of_time;   /* 9/9/99 9:09:09.999999 */


static bool read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE]);

static uint32 block_spacing;     /* maximum spacing between positions on the disk, to fit the file 0 = disabled */
static int block_direction;

/* static uint8* cache[MAX_BLOCKS_CACHED]; until we implement this don't use space */


#define BIT_MAP_CACHE_SIZE 256

typedef struct {
  uint32 chunk;
  int age;
  bool dirty;
  uint8* map;
} Bit_Map_Cache_Entry;

static Bit_Map_Cache_Entry bit_map_cache[BIT_MAP_CACHE_SIZE];

static int bit_map_tick = 0;

static void write_bit_map(Bit_Map_Cache_Entry* entry);


/**************************************************************************/
/* Special malloc and free routines that can help check for memory leaks. */
/**************************************************************************/

#ifdef CHECK_MEMORY_LEAKS
#define MAX_MEM_ADDR_SAVED 16384
void* save_malloc_addr[MAX_MEM_ADDR_SAVED];
int cur_malloc_index;
#endif

void* nwos_malloc(size_t size)
{
#ifdef CHECK_MEMORY_LEAKS
    int i;
#endif

    void* result = malloc(size);

    if (result == NULL)
    {
	char msg[64];
	sprintf(msg, "%zd bytes", size);
	perror(msg);
	assert(result);
	exit(1);
    }

#ifdef CHECK_MEMORY_LEAKS
    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	if (save_malloc_addr[cur_malloc_index] == NULL) break;
	cur_malloc_index++;
	if (cur_malloc_index >= MAX_MEM_ADDR_SAVED) cur_malloc_index = 0;  /* wrap if necessary */
    }

    assert(i < MAX_MEM_ADDR_SAVED);

    save_malloc_addr[cur_malloc_index] = result;
    cur_malloc_index++;
    if (cur_malloc_index >= MAX_MEM_ADDR_SAVED) cur_malloc_index = 0;  /* wrap if necessary */
#endif

    return result;
}

void nwos_free(void* ptr)
{
#ifdef CHECK_MEMORY_LEAKS
    int i;
#endif

    free(ptr);

#ifdef CHECK_MEMORY_LEAKS
    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	cur_malloc_index--;
	if (cur_malloc_index < 0) cur_malloc_index = MAX_MEM_ADDR_SAVED - 1;  /* wrap if necessary */
	if (save_malloc_addr[cur_malloc_index] == ptr) break;
    }

    assert(i < MAX_MEM_ADDR_SAVED);

    save_malloc_addr[cur_malloc_index] = NULL;    /* erase it */
#endif
}


/***********************************/
/* Variable Length Count functions */
/***********************************/

uint32 nwos_decode_variable_sized_count(uint8 count[4])
{
    uint32 result;

    result = count[0];   /* get the first byte */

    if (result > 127)   /* two bytes */
    {
	assert((result & 0xc0) == 0x80);  /* for now upper two bits must be 10 */

	result = (((result & 0x3f) << 8) | count[1]) + 128;
    }
    else
    {
	/* this byte should always be zero if first bytes is < 128.  the original */
	/* plan was to not store the extra bytes, but the compiler did it anyway */
	assert(count[1] == 0);
    }

    return result;
}

void nwos_encode_variable_sized_count(uint32 count, uint8 result[4])
{
    int adjusted;

    assert(count < (16384 + 128));   /* for now we can only handle two bytes */

    if (count > 127)   /* two bytes */
    {
	adjusted  = count - 128;
	result[0] = (0x80 | (adjusted >> 8));
	result[1] = adjusted & 0xff;
    }
    else
    {
	result[0] = count;
	result[1] = 0;
    }

    result[2] = 0;
    result[3] = 0;

    assert(nwos_decode_variable_sized_count(result) == count);
}



/******************************************/
/* Log file - THIS NEEDS TO GO AWAY SOON. */
/******************************************/

void nwos_log(char* str)
{
    struct timeval tv;
    FILE* fp;
    char time_str[32];

    gettimeofday(&tv, NULL);
    strncpy(time_str, ctime(&tv.tv_sec), 32);
    *strchr(time_str, '\n') = '\0';
    fp = fopen(LOG_FILE_PATH, "a");
    if (fp == NULL)
    {
	perror(LOG_FILE_PATH);
	exit(1);
    }
    fprintf(fp, "%s: %s\n", time_str, str);
    fclose(fp);
}



#ifdef DISABLE_SECURITY_FEATURES
uint32 randomxxx;
#endif


/**********************/
/* Sequence generator */
/**********************/

static void generate_sequence_table(uint32* linear, uint32* serial, uint8 table[FILE_BLOCK_SIZE])
{
    int i;
    uint8 j;
    uint8 filled[FILE_BLOCK_SIZE];    /* remember which slots are already filled */

    /* mark all slots as empty */
    for (i = 0; i < FILE_BLOCK_SIZE; i++) filled[i] = 0;

    /* then mark the first 8 as used */
    for (i = 0; i < 8; i++) filled[i] = 1;

    /* start at 0 because it is used already, so while loop will fail the first time. */
    j = 0;
    assert(filled[j] != 0);

    for (i = 8; i < FILE_BLOCK_SIZE; i++)
    {
	/* find where the next byte is stored */
	while(filled[j] != 0) j = nwos_next_sequence(linear, serial) % FILE_BLOCK_SIZE;
	table[i] = j;   /* store it in the table */
	filled[j] = 1;            /* mark it as used */
    }
}


/**********************************/
/* Initialization and Termination */
/**********************************/

void index_compressed_file()
{
    int i;
    int size_of_index = 262144;
    size_t bytes_read;
    uint8 buffer[FILE_BLOCK_SIZE];

    assert(obj_file_desc > 0);

    storage_index = malloc(size_of_index * sizeof(ObjRef));   /* allocate space for the first batch */

    assert(storage_index != NULL);

    printf("Indexing");
    fflush(stdout);

    /* read the first block, it contains the header */
    bytes_read = read(obj_file_desc, buffer, sizeof(buffer));

    assert(bytes_read == sizeof(buffer));

    i = 1;
    while (1)
    {
	if (i % 262144 == 1)
	{
	    printf(".");
	    fflush(stdout);
	}

	if (i == size_of_index)   /* need to allocate more space */
	{
	    size_of_index += 262144;
	    storage_index = realloc(storage_index, size_of_index * sizeof(ObjRef));
	    assert(storage_index != NULL);
	}

	assert(i < size_of_index);

	bytes_read = read(obj_file_desc, buffer, sizeof(buffer));

	if (bytes_read != sizeof(buffer)) break;

	memcpy(&storage_index[i], &buffer[4], sizeof(ObjRef));

	i++;
    }

    if (errno != 0)
    {
	perror(storage_path);
	exit(1);
    }

    assert(bytes_read == 0);

    storage_count = i;

    printf("\nblocks: %d\n", i);
}


void initialize_random_number_generator()
{
    struct timespec ts;

    clock_gettime(CLOCK_REALTIME, &ts);
    srandom((uint32)ts.tv_sec ^ (uint32)ts.tv_nsec);

#ifdef DISABLE_SECURITY_FEATURES
    randomxxx = 0x00000100;     /* start objects here */
#endif
}


void nwos_initialize_objectify(uint8 bf_key[16], uint32 linear_seed, uint32 serial_seed, StorageType type, char* path)
{
    struct tm bdt;
    int i;
    int j;
    int k;
    int tries;
    char log_msg[128];

    /* make sure the storage is something we can deal with */
    assert(type == Drive_Or_Partition_RO || type == Drive_Or_Partition_RW || type == Compressed_File_RO);

    assert(storage_path == NULL);   /* make sure this is only called once */
    assert(obj_file_desc == 0);

    storage_path = path;

    initialize_random_number_generator();
    nwos_crc32_initialize();

    BF_set_key(&blowfish_key, 16, bf_key);

    linear_after_first = linear_seed;
    serial_after_first = serial_seed;

    /* Create all of the sequence tables from the linear and serial parameters */
    for (i = 0; i < NUM_STORED_SEQ; i++)
    {
	tries = 0;

	do
	{
	    tries++;

	    assert(tries < 4);

	    generate_sequence_table(&linear_after_first, &serial_after_first, nwos_random_sequence[i]);

	    for (j = 0; j < i; j++)
	    {
		for (k = 8; k < FILE_BLOCK_SIZE; k++) if (nwos_random_sequence[i][k] != nwos_random_sequence[j][k]) break;

		if (k == FILE_BLOCK_SIZE) break;
	    }
	}
	while (j < i);    /* if j is less than i it means one of the tables matched and needs to be regenerated */
    }

    bdt.tm_sec = 9;
    bdt.tm_min = 9;
    bdt.tm_hour = 9;
    bdt.tm_mday = 9;
    bdt.tm_mon = 9 - 1;
    bdt.tm_year = 1999 - 1900;
    bdt.tm_isdst = -1;            /* hopefully mktime can figure this out! */

    end_of_time.tv_sec = mktime(&bdt);
    end_of_time.tv_usec = 999999;

#if 0
    {
    int i;
    printf("blowfish key: ");
    for (i = 0; i < 16; i++) printf("%02x", bf_key[i]);
    printf("\n");
    }
#endif

    read_only = ((type & 1) == 0);

    if (read_only)
    {
	obj_file_desc = open(storage_path, O_RDONLY | O_LARGEFILE);
    }
    else
    {
	obj_file_desc = open(storage_path, O_RDWR | O_LARGEFILE);
    }

    if (type == Compressed_File_RO)
    {
	index_compressed_file();   /* build an index into the compressed file */

	if (lseek64(obj_file_desc, 0LL, SEEK_SET) < 0)
	{
	    perror("rewinding after indexing");
	    exit(1);
	}
    }

    if (obj_file_desc < 0)
    {
	perror(storage_path);
	exit(1);
    }

    if (read(obj_file_desc, disk_header, sizeof(disk_header)) != sizeof(disk_header))
    {
	sprintf(log_msg, "reading disk header from: %s", storage_path);
	perror(log_msg);
	exit(1);
    }

    if (memcmp(&disk_header[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(&disk_header[4], VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    reserved_public_blocks = ((uint32)disk_header[8] << 24) | ((uint32)disk_header[9]<< 16) | ((uint32)disk_header[10] << 8) | ((uint32)disk_header[11]);

    blocks_on_disk = ((uint32)disk_header[12] << 24) | ((uint32)disk_header[13]<< 16) | ((uint32)disk_header[14] << 8) | ((uint32)disk_header[15]);

    copy_reference(&nwos_next_public_ref, (ObjRef*)&disk_header[16]);


    assert(BLOCKS_IN_CHUNK <= reserved_public_blocks && reserved_public_blocks <= NUM_PUBLIC_IDS);
    assert(reserved_public_blocks % BLOCKS_IN_CHUNK == 0);
    assert(blocks_on_disk > reserved_public_blocks);

    private_blocks_on_disk = blocks_on_disk - reserved_public_blocks;

    sprintf(log_msg, "nwos_initialize_objectify() - blocks_on_disk: %u", blocks_on_disk);
    nwos_log(log_msg);
}


/* returns true if the object was created, false if it already existed */

bool nwos_create_root(ObjRef* ref)
{
    C_struct_Root root_obj;
    ObjRef root_class_ref;
    bool result = false;

    /* first find out if we already have this root */


    if (!nwos_object_exists(ref))   /* it already exists */
    {
	assert(nwos_find_class_definition("ROOT", &root_class_ref));

	memset(&root_obj, 0, sizeof(root_obj));  /* zero it out */

	nwos_fill_in_common_header(&root_obj.header.common, ref, &root_class_ref);

	memcpy(&root_obj.class_definition_class, &nwos_class_definition_class_ref, sizeof(root_obj.class_definition_class));
	memcpy(&root_obj.reference_list_class, &nwos_reference_list_class_ref, sizeof(root_obj.reference_list_class));

	nwos_create_reference_list(ref, &root_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &root_obj.header.object, sizeof(ObjectHeader), root_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &root_obj.class_definition_class, sizeof(root_obj) - sizeof(EveryObject), root_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &root_obj, sizeof(root_obj));

	nwos_add_to_references(ref, &root_class_ref);

	result = true;
    }

    return result;
}


void nwos_set_root_object(ObjRef* ref)
{
    C_struct_Root root_obj;
    ObjRef verify_ref;

    nwos_read_object_from_disk(ref, &root_obj, sizeof(root_obj));
    memcpy(&nwos_class_definition_class_ref, &root_obj.class_definition_class, sizeof(nwos_class_definition_class_ref));
    memcpy(&nwos_reference_list_class_ref, &root_obj.reference_list_class, sizeof(nwos_reference_list_class_ref));

    assert(nwos_find_class_definition("REFERENCE LIST", &verify_ref));
    assert(is_same_object(&verify_ref, &nwos_reference_list_class_ref));
}


void nwos_terminate_objectify()
{
    int i;

    nwos_flush_dirty_ref_lists();  /* flush any reference lists that are dirty out to disk */

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].chunk != 0 && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);

	    bit_map_cache[i].chunk = 0;

	    free(bit_map_cache[i].map);
	    bit_map_cache[i].map = NULL;
	}
    }

    if (modified)
    {
	copy_reference((ObjRef*)&disk_header[16], &nwos_next_public_ref);

	nwos_get_time_stamp(&disk_header[24]);

	if (lseek64(obj_file_desc, (off64_t)0, SEEK_SET) < 0)
	{
	    perror(storage_path);
	    exit(1);
	}

	if (write(obj_file_desc, disk_header, sizeof(disk_header)) != sizeof(disk_header))
	{
	    perror(storage_path);
	    exit(1);
	}
    }

    if (close(obj_file_desc) < 0)
    {
	perror(storage_path);
	exit(1);
    }

    obj_file_desc = 0;
    storage_path = NULL;

#ifdef CHECK_MEMORY_LEAKS
    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	if (save_malloc_addr[i] != NULL) printf("WARNING - memory block not freed: %p\n", save_malloc_addr[i]);
    }
#endif
}

void nwos_flush_bit_maps()
{
    int i;

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].chunk != 0 && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);
	}
    }
}


/*******************************************************************/
/* Reference utility functions: conversion, testing and generation */
/*******************************************************************/

static inline uint32 hash_ref(ObjRef* ref)
{
    uint32 result;

    result = ((uint32)(ref->id[0]) << 24) | ((uint32)(ref->id[1]) << 16) | 
             ((uint32)(ref->id[2]) << 8) | ((uint32)(ref->id[3]));

    if ((ref->id[0] & 0xF0) != 0)  /* it's a private reference */
    {
	result = (result - NUM_PUBLIC_IDS) % private_blocks_on_disk + reserved_public_blocks;
    }

    return result;
}


static inline off64_t ref_to_offset(ObjRef* ref) 
{
    uint32 hash;

    assert(FILE_BLOCK_SIZE == 256);

    hash = hash_ref(ref);

    if ((ref->id[0] & 0xF0) != 0)  /* it's a private reference */
    {
	assert(BIT_MAP_BYTES == 8192);   /* if this fails need to fix the hack below */

	if ((hash & 0xffe0) == 0)     /* special hack to get around blocks that would overwrite the bit map */
	{
	    hash = (hash + 32) & 0xffffffe0;
	}
    }

    return (off64_t) hash << 8;
}


/* Doesn't return values where the uppermost nibble is zero */

static void generate_random_id(ObjRef* ref)
{
    int i = 0;

    ref->word = random();

    assert(NUM_PUBLIC_IDS == 0x10000000);   /* make sure that the upper nibble matches what is reserved */
                                            /* if this assert fails, need to fix the while loop below */

    while ((ref->id[0] & 0xF0) == 0)
    {
	ref->word = random();

	i++;    /* just in case random() is broken */

	assert(i < 10);
    }

//    printf("generate_random_id: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
#ifdef DISABLE_SECURITY_FEATURES  /* use an incremental value instead of random */
    randomxxx++;
    ref->id[0] = randomxxx >> 24;
    ref->id[1] = randomxxx >> 16;
    ref->id[2] = randomxxx >> 8;
    ref->id[3] = randomxxx;
#endif
}


bool block_contains_object(uint8 block[FILE_BLOCK_SIZE], ObjRef* ref)
{
    uint8 marker[8];

    marker[0] = 0;
    marker[1] = 0;
    marker[2] = 0;
    marker[3] = 0;
    marker[4] = ref->id[0];
    marker[5] = ref->id[1];
    marker[6] = ref->id[2];
    marker[7] = ref->id[3];

    return (memcmp(block, marker, sizeof(marker)) == 0);   /* return true if the object marker is there */
}


bool nwos_object_exists(ObjRef* ref)
{
    uint8 block[FILE_BLOCK_SIZE];

    return read_block(ref, block) && block_contains_object(block, ref);
}


void write_bit_map(Bit_Map_Cache_Entry* entry)
{
    char log_msg[80];

    sprintf(log_msg, "write bit_map: %08x", entry->chunk);

    nwos_log(log_msg);

    if (lseek64(obj_file_desc, (off64_t)entry->chunk << 8, SEEK_SET) < 0)
    {
	sprintf(log_msg, "write_bit_map lseek64 chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    if (write(obj_file_desc, entry->map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	sprintf(log_msg, "write_bit_map write chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    entry->dirty = false;

    modified = true;
}


void read_bit_map_into_cache(Bit_Map_Cache_Entry* entry)
{
    char log_msg[80];
#if 0
    sprintf(log_msg, "write bit_map: %08x\n", entry->chunk);

    nwos_log(log_msg);
#endif
    if (lseek64(obj_file_desc, (off64_t)entry->chunk << 8, SEEK_SET) < 0)
    {
	sprintf(log_msg, "read_bit_map lseek64 chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    if (entry->map == NULL)
    {
	entry->map = malloc(BIT_MAP_BYTES);
	assert(entry->map != NULL);
    }

    if (read(obj_file_desc, entry->map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	sprintf(log_msg, "read_bit_map write chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    entry->dirty = false;
}


static Bit_Map_Cache_Entry* find_bit_map_in_cache(uint32 block)
{
    int i;
    Bit_Map_Cache_Entry* result = NULL;

    assert((block % CHUNK_SIZE) != 0);

    /* see if it's already in the cache */
    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].chunk == (block & BIT_MAP_BLOCK_MASK)) break;
    }

    if (i < BIT_MAP_CACHE_SIZE)   /* found it */
    {
	result = &bit_map_cache[i];
    }
    else                           /* didn't find it */
    {
	/* find an empty one or find the oldest */

	result = bit_map_cache;   /* so we have an age to compare to */

	for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
	{
	    if (bit_map_cache[i].chunk == 0)
	    {
		result = &bit_map_cache[i];
		break;
	    }

	    if (bit_map_cache[i].age < result->age)
	    {
		result = &bit_map_cache[i];
	    }
	}

	if (i == BIT_MAP_CACHE_SIZE && result->dirty)    /* didn't find an empty one, write the oldest one out */
	{
	    write_bit_map(result);
	}

	result->chunk = block & BIT_MAP_BLOCK_MASK;

	read_bit_map_into_cache(result);
    }

    bit_map_tick++;
    result->age = bit_map_tick;

    assert(result != NULL);

    return result;
}


static void set_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    entry->map[byte_num] |= (0x80 >> bit_num);

/* printf("set_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

    entry->dirty = true;
}


bool nwos_block_used(ObjRef* ref)
{
    uint8 block[FILE_BLOCK_SIZE];

    assert(BIT_MAP_BYTES == 8192);   /* if this fails need to fix the hack below */

    if ((hash_ref(ref) & 0xffe0) == 0) return true;   /* the bit map block is always used */

    return  read_block(ref, block) && !is_void_reference((ObjRef*)&block[4]);
}


void nwos_set_sequential_blocks(uint32 approx_blocks)
{
    assert(approx_blocks == 0 || approx_blocks > 3);

    if (approx_blocks == 0)
    {
	block_spacing = 0;
    }
    else
    {
	block_spacing = blocks_on_disk / approx_blocks;
    }

    block_direction = 0;

    assert(block_spacing == 0 || block_spacing > 5);   /* let's say file shouldn't use 20% of disk */
}


static void nwos_generate_new_completely_random_id(ObjRef* ref)
{
    unsigned long dummy;
    int state = 1;
    int n;
    bool unable_to_find_unique_id = false;

    while (state > 0)
    {
	generate_random_id(ref);

	assert((ref->id[0] & 0xF0) != 0);

	if (!nwos_block_used(ref)) 
	{
	    state = 0;   /* exit loop */
	}
	else      /* found an unused id */
	{
#ifdef DISABLE_SECURITY_FEATURES
	  /* ignore the duplicate and just keep counting up until we find one */
#else
	    switch(state)
	    {
	      case 1:    /* try 9 times */
	      case 2:    /* just for fun */
	      case 3:
	      case 4:    /* try 3 times */
	      case 5:    /* just for fun */
	      case 6:
	      case 7:    /* try 3 times */
	      case 8:    /* just for fun */
	      case 9:
		state++;
		break;

	      case 10:    /* spin the dammed thing a few times */
		n = random() % 13;
		while ((random() % 17) != n) { }
		state++;
		break;

	      case 11:    /* reinitialize in case we've hit upon a sequence we've done before */	
		initialize_random_number_generator();
		state++;
		break;

	      case 12:    /* this is bad - wait a random amount of time and then try reinitializing again */
		dummy = random();
		usleep(dummy % 7919 + 2082);
		initialize_random_number_generator();
		state++;
		break;

	      case 13:    /* stop and figure out what is wrong */
		assert(unable_to_find_unique_id);
		break;
	    }
#endif
	}
    }
}


#define MAX_POSSIBLE_REFS ((int) (NUM_PRIVATE_IDS / blocks_on_disk + 1))  /* could be 1 too many if even */

static int nwos_generate_new_id_in_range(ObjRef* last_ref, uint32 spacing, int direction, ObjRef* new_ref)
{
    int new_direction = direction;
    uint32 old_block;
    uint32 new_block;
    int offset;
    int i;
    ObjRef possible_ref[MAX_POSSIBLE_REFS];

    assert((spacing & 0x8000000) == 0);     /* make sure it will convert to a positive integer */

    assert(spacing < (blocks_on_disk / 2));   /* and it isn't going to cause any strange behaviour by being too large */

    /* if this is the first time, choose a direction */

    old_block = hash_ref(last_ref);

    if (new_direction == 0)
      {
	if (old_block < reserved_public_blocks + spacing)
	{
	    new_direction = 1;
	}
	else if (old_block > blocks_on_disk - spacing)
	{
	    new_direction = -1;
	}
	else if ((random() & 1) == 0)
	{
	    new_direction = 1;
	}
	else
	{
	    new_direction = -1;
	}
    }


    /* Are we getting close to one end and need to reverse direction? */

    if (new_direction < 0)
    {
	if (old_block < (reserved_public_blocks + spacing)) new_direction = 1;    /* go forward now */
    }
    else 
    {
	if (old_block > (blocks_on_disk - spacing)) new_direction = -1;    /* go backwards now */
    }


    /* compute a new block number */

    do
    {
	offset = random() % (int)spacing;

	if (new_direction < 0)
	{
	    new_block = old_block - offset;    /* go backwards */
	}
	else 
	{
	    new_block = old_block + offset;    /* go forward */
	}

	assert(new_block > reserved_public_blocks);
	assert(new_block < blocks_on_disk);

	nwos_word_to_ref(new_block, &possible_ref[0]);
    }
    while (nwos_block_used(&possible_ref[0]));


    /* now convert the block into a random reference that will equate to that block */

    for (i = 1; i < MAX_POSSIBLE_REFS; i++)
    {
	if (((new_block - reserved_public_blocks) * i) > NUM_PRIVATE_IDS)
	{
	    assert(i + 1 == MAX_POSSIBLE_REFS);
	    break;
	}
	else
	{
	    nwos_word_to_ref((new_block - reserved_public_blocks) + private_blocks_on_disk * i + NUM_PUBLIC_IDS, &possible_ref[i]);
	}

	assert(hash_ref(&possible_ref[i]) == new_block);
    }

    i = random() % i;   /* pick on of the possible ids */

    assert(0 <= i && i < MAX_POSSIBLE_REFS);

    copy_reference(new_ref, &possible_ref[i]);

    return new_direction;
}


void nwos_generate_new_id(ObjRef* ref)
{
    static ObjRef last_ref;

    if (block_spacing == 0)
    {
	nwos_generate_new_completely_random_id(ref);
    }
    else
    {
	block_direction = nwos_generate_new_id_in_range(&last_ref, block_spacing, block_direction, ref);
    }

    copy_reference(&last_ref, ref);
}


void nwos_generate_new_public_id(ObjRef* ref)
{
    int i;

    copy_reference(ref, &nwos_next_public_ref);

    for (i = sizeof(nwos_next_public_ref)-1; i > -1; i++)
    {
	nwos_next_public_ref.id[i]++;
	if (nwos_next_public_ref.id[i] != 0) break;
    }
}


/*************************/
/* Verification routines */
/*************************/

#if 0
static bool is_compatible_version(char* header_version)
{
    static char* compatible[] = COMPATIBLE_HEADER_VERSIONS;
    const int num_compatible = sizeof(compatible) / sizeof(char*);
    int i;

    for (i = 0; i < num_compatible; i++) 
    {
	if (memcmp(header_version, compatible[i], strlen(HEADER_VERSION)) == 0) break;
    }

    return (i < num_compatible);    /* return true if found a match */

    return true;
}
#endif


static void check_common_header(ObjRef* ref, CommonHeader* common)
{
#if 0
#if 0
    printf("common header pointer: %p\n", common);
#endif

    if (memcmp(common->magic_number, MAGIC_NUMBER, sizeof(common->magic_number)))
    {
	printf("object: %s - missing magic number\n", path);
	exit(1);
    }

    if (!is_compatible_version((char*)common->version))
    {
	printf("object: %s - incompatible version: ", path);
	if (isprint(common->version[0]) && isprint(common->version[1]) &&
	    isprint(common->version[2]) && isprint(common->version[3]))
	{
	    printf("%c%c%c%c\n", common->version[0], common->version[1], common->version[2], common->version[3]);
	}
	else
	{
	    printf("%02x %02x %02x %02x\n", common->version[0], common->version[1], common->version[2], common->version[3]);
	}
	exit(1);
    }
#endif
}


/* ref is only needed for error messages */

static void check_object_header(ObjRef* ref, ObjectHeader* obj_header, uint8 stored[4])
{
    uint8 computed[4];
    bool checksum_ok;

    nwos_crc32_calculate((uint8*)obj_header, sizeof(*obj_header), computed);

    checksum_ok = (stored[0] == computed[0] || stored[1] == computed[1] || stored[2] == computed[2] || stored[3] == computed[3]);

    if (!checksum_ok)
    {
	printf("object: %02x%02x%02x%02x - bad header checksum, ", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	printf("size %zd - ", sizeof(*obj_header));
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);
	assert(checksum_ok);
    }
}


static void check_object_data(ObjRef* ref, uint8* object, size_t total_size, uint8 stored[4])
{
    uint8 computed[4];
    size_t data_size;

    data_size = total_size - sizeof(EveryObject);

#if 0
    printf("Total size: %d\n", total_size);
    printf("ObjectHeader: %d\n", sizeof(EveryObject));
    printf("object pointer: %p\n", object);
    {
      int i;
      uint8* ptr = (uint8*)object + sizeof(EveryObject);
      printf("data pointer: %p\n", ptr);
      printf("data: ");
      for (i = 0; i < data_size; i++) printf("%02x%c", *ptr++, i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    nwos_crc32_calculate((uint8*)object + sizeof(EveryObject), data_size, computed);

    if (stored[0] != computed[0] || stored[1] != computed[1] || stored[2] != computed[2] || stored[3] != computed[3])
    {
	printf("object: %02x%02x%02x%02x - bad data checksum, ", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	printf("size %zd - ", data_size);
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);

	printf("\n");
	{
	  int i;
	  for (i = 0; i < sizeof(EveryObject) + data_size; i++)
	  {
	      printf("%02x%c", object[i], (i % 16) == 15 ? '\n' : ' ');
	  }
	  printf("\n");
	}
	exit(1);
    }
}



/*****************************************/
/* Routines to read blocks, objects, etc */
/*****************************************/

#ifdef DEBUG_ENCRYPTION
uint8 save_unencrypted[FILE_BLOCK_SIZE];
uint8 save_add_random[FILE_BLOCK_SIZE];
uint8 save_mixed_up[FILE_BLOCK_SIZE];
uint8 save_encrypted[FILE_BLOCK_SIZE];
#endif

static bool read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char ref_str[64];
    off64_t offset;
    int i;
    int lower;
    int upper;
    uint32 ref_hash;
    uint32 mid_hash;

    assert(!is_void_reference(ref));

    /*    printf("Read block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]); */


#if NO_DISK_ACCESS

    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] != NULL)
    {
	memcpy(block, cache[i], FILE_BLOCK_SIZE);
    }

    return cache[i] != NULL;

#else

    if (storage_index == NULL)     /* normal storage */
    {
	offset = ref_to_offset(ref);
    }
    else
    {
	lower = 1;
	upper = storage_count;

	ref_hash = hash_ref(ref);

	while (lower <= upper)
	{
	    i = (lower + upper) / 2;

	    mid_hash = hash_ref(&storage_index[i-1]);

	    if (mid_hash > ref_hash)
	    {
		upper = i - 1;
	    }
	    else if (mid_hash < ref_hash)
	    {
		lower = i + 1;
	    }
	    else
	    {
		i = i - 1;
		break;
	    }
	}

	assert(lower <= upper);

	assert(is_same_object(&storage_index[i], ref));

	offset = (off64_t)i * FILE_BLOCK_SIZE;
    }

    if (lseek64(obj_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	sprintf(ref_str, "read_block lseek64 ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(ref_str);
	exit(1);
    }

    return read(obj_file_desc, block, FILE_BLOCK_SIZE) == FILE_BLOCK_SIZE;

#endif
}


#ifdef DEBUG_ENCRYPTION
void compare_bufs(uint8 buf1[FILE_BLOCK_SIZE], uint8 buf2[FILE_BLOCK_SIZE], char* descr)
{
  int i;
  for (i = 8; i < FILE_BLOCK_SIZE; i++) if (buf1[i] != buf2[i]) break;
  if (i == FILE_BLOCK_SIZE)
    {
      printf("%s - OK\n", descr);
    }
  else
    {
      printf("%s - bad\n", descr);
      for (i = 0; i < FILE_BLOCK_SIZE; i++)
	{
	  printf("%02x-%02x%c", buf1[i], buf2[i], i%16==15 ? '\n' : ' ');
	}
    }
}
#endif


void nwos_read_object_from_disk_and_decrypt(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], uint8 seq_table[FILE_BLOCK_SIZE])
{
    uint8 buffer1[FILE_BLOCK_SIZE];
    uint8 buffer2[FILE_BLOCK_SIZE];
    uint8* ptrobj = (uint8*)object;
    int i;
    int j;

    assert(!is_void_reference(ref));

    read_block(ref, buffer2);

    assert(block_contains_object(buffer2, ref));

#ifdef DEBUG_ENCRYPTION
    compare_bufs(buffer2, save_encrypted, "Encrypted");
#endif

#ifdef DISABLE_SECURITY_FEATURES
    if (true)
#else
    if ((buffer2[0] & UNENCRYPTED_FLAG) != 0)
#endif
    {
	memcpy(object, buffer2, size);    /* just copy the buffer into the object */
    }
    else
    {
	/* copy the first 8 bytes directly, not encrypted */
	memcpy(buffer1, buffer2, 8);

	/* decrypt the remaining bytes */
	BF_cbc_encrypt((buffer2 + 8), (buffer1 + 8), (FILE_BLOCK_SIZE - 8), &blowfish_key, ivec, BF_DECRYPT);

#if 0
	printf("ivec: ");
	for (i = 0; i < 8; i++) printf("%02x ", ivec[i]);
	printf("\n");
#endif

#ifdef DEBUG_ENCRYPTION
	compare_bufs(buffer1, save_mixed_up, "Mixed up");
#endif

	assert(size <= FILE_BLOCK_SIZE);

	/* copy the first 8 bytes directly accross */
	for (i = 0; i < 8; i++) ptrobj[i] = buffer1[i];

	for (i = 8; i < size; i++)
	{
	    j = seq_table[i];    /* find where the next byte is stored */
	    ptrobj[i] = buffer1[j];    /* get this byte from there */
	}

    }

#ifdef DEBUG_ENCRYPTION
    compare_bufs(ptrobj, save_unencrypted, "Unencrypted");
#endif
}


#define SIZE_EXTRA_BLOCK_DATA (FILE_BLOCK_SIZE - sizeof(uint32) - sizeof(ObjRef) - sizeof(ObjRef))

typedef struct
{
    uint8  flags[4];
    ObjRef id;
    uint8  data[SIZE_EXTRA_BLOCK_DATA];
    ObjRef next_block;
} Extra_Data_Block;


void nwos_read_object_from_disk(ObjRef* ref, void* object, size_t size)
{
    EveryObject* header;
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    ObjRef next_ref;
    size_t bytes_remaining;
    int i;
    int seq = 0;

    assert(!is_void_reference(ref));

#if 0
    char log_msg[128];

    sprintf(log_msg, "nwos_read_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    assert(size > sizeof(CommonHeader));  /* make sure this wasn't called to just get the header */

    memset(ivec, 0, sizeof(ivec));

    bytes_remaining = size;

    ptr_obj = object;

    nwos_read_object_from_disk_and_decrypt(ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);

    assert(sizeof(ObjRef) == 4);

    *ptr_obj++ = extra.flags[0];
    *ptr_obj++ = extra.flags[1];
    *ptr_obj++ = extra.flags[2];
    *ptr_obj++ = extra.flags[3];
    *ptr_obj++ = extra.id.id[0];
    *ptr_obj++ = extra.id.id[1];
    *ptr_obj++ = extra.id.id[2];
    *ptr_obj++ = extra.id.id[3];

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    *ptr_obj++ = extra.data[i];
	    bytes_remaining--;
	}

	/* save the reference because the extra block is going to get written over, we don't want it to change */
	copy_reference(&next_ref, &extra.next_block);

	seq++;

	nwos_read_object_from_disk_and_decrypt(&next_ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) *ptr_obj++ = extra.data[i];

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

    check_common_header(ref, &header->common);

    check_object_header(ref, &header->object, header->common.header_chksum);

    check_object_data(ref, (uint8*)object, size, header->common.data_chksum);
}


void nwos_read_variable_sized_object_from_disk(ObjRef* ref, void* object, size_t max_size, size_t (*size_function)(void*))
{
    EveryObject* header;
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    ObjRef next_ref;
    size_t bytes_remaining;
    int i;
    int seq = 0;

    assert(!is_void_reference(ref));

#if 0
    char log_msg[128];

    sprintf(log_msg, "nwos_read_variable_size_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    memset(ivec, 0, sizeof(ivec));

    ptr_obj = object;

    nwos_read_object_from_disk_and_decrypt(ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);

    assert(sizeof(ObjRef) == 4);

    *ptr_obj++ = extra.flags[0];
    *ptr_obj++ = extra.flags[1];
    *ptr_obj++ = extra.flags[2];
    *ptr_obj++ = extra.flags[3];
    *ptr_obj++ = extra.id.id[0];
    *ptr_obj++ = extra.id.id[1];
    *ptr_obj++ = extra.id.id[2];
    *ptr_obj++ = extra.id.id[3];

    bytes_remaining = (*size_function)(&extra);

    assert(bytes_remaining <= max_size);

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    *ptr_obj++ = extra.data[i];
	    bytes_remaining--;
	}

	/* save the reference because the extra block is going to get written over, we don't want it to change */
	copy_reference(&next_ref, &extra.next_block);

	seq++;

	nwos_read_object_from_disk_and_decrypt(&next_ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) *ptr_obj++ = extra.data[i];

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

    check_common_header(ref, &header->common);

    check_object_header(ref, &header->object, header->common.header_chksum);

    check_object_data(ref, (uint8*)object, (*size_function)(object), header->common.data_chksum);
}


void nwos_read_object_headers_from_disk(ObjRef* ref, EveryObject* header)
{
    uint8 ivec[IVEC_SIZE];

    assert(!is_void_reference(ref));

    memset(ivec, 0, sizeof(ivec));
    nwos_read_object_from_disk_and_decrypt(ref, header, sizeof(*header), ivec, nwos_random_sequence[0]);

#if 0
    printf("read header path: %s\n", path);
    printf("read header pointer: %p\n", header);
    {
      int i;
      for (i = 0; i < sizeof(EveryObject); i++) printf("%02x%c", *((uint8*)header+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    check_common_header(ref, &header->common);

    check_object_header(ref, &header->object, header->common.header_chksum);
}


/******************************************/
/* Routines to write blocks, objects, etc */
/******************************************/

static void write_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char log_msg[128];
    off64_t offset;

    assert(!is_void_reference(ref));
    assert(block[0] == 0 && block[1] == 0 && block[2] == 0 && block[3] == 0);
    assert(is_same_object((ObjRef*)block + 1, ref));

    sprintf(log_msg, "write_block - ref:%02x%02x%02x%02x block:%08x",
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3], hash_ref(ref));
    nwos_log(log_msg);

#if NO_DISK_ACCESS
{
    int i;
    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    printf("WARNING - rewritting the same block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] == NULL)
    {
	cache[i] = malloc(FILE_BLOCK_SIZE);
	assert(cache[i] != NULL);
    }

    memcpy(cache[i], block, FILE_BLOCK_SIZE);
}
#else
    assert(storage_index == NULL);   /* at this time if there is a storage index it is read only */

    offset = ref_to_offset(ref);
    if (lseek64(obj_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	sprintf(log_msg, "write_block lseek64 ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

    if (write(obj_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	sprintf(log_msg, "write_block write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

    modified = true;

    if ((ref->id[0] & 0xF0) != 0)  /* it's a private reference */
    {
	set_bit_in_map((uint32) (offset >> 8));
    }
#endif
}


void nwos_write_object_to_disk_and_encrypt(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], uint8 seq_table[FILE_BLOCK_SIZE])
{
    uint8 buffer1[FILE_BLOCK_SIZE];
    uint8 buffer2[FILE_BLOCK_SIZE];
    uint8* ptrobj = (uint8*)object;
    int i;
    int j;

    assert(!is_void_reference(ref));
    assert(size <= FILE_BLOCK_SIZE);

#ifdef DEBUG_ENCRYPTION
    memcpy(save_unencrypted, object, FILE_BLOCK_SIZE);
#endif

#ifdef DISABLE_SECURITY_FEATURES    /* always do the unencryped code */
    if (true)
#else
    if ((*ptrobj & UNENCRYPTED_FLAG) != 0)
#endif
    {
	memset(buffer2, 0, FILE_BLOCK_SIZE);    /* just copy the object into the buffer */
	memcpy(buffer2, object, size);
    }
    else
    {
	/* copy the first 8 bytes directly accross */
	for (i = 0; i < 8; i++) buffer1[i] = ptrobj[i];

	/* fill any empty slots with random junk */
	if (size < FILE_BLOCK_SIZE)
	{
	    for (i = 8; i < FILE_BLOCK_SIZE; i++) buffer1[i] = random();
	}

	for (i = 8; i < size; i++)
	{
	    /* find where the next byte is stored */
	    j = seq_table[i];    /* find where the next byte is stored */
	    buffer1[j] = ptrobj[i];    /* save this byte there */
	}

#ifdef DEBUG_ENCRYPTION
	memcpy(save_mixed_up, buffer1, FILE_BLOCK_SIZE);
#endif

	/* copy the first 8 bytes straight across */
	memcpy(buffer2, buffer1, 8);

	/* and encrypt the remainder 8 bytes at a time */
	BF_cbc_encrypt((buffer1 + 8), (buffer2 + 8), (FILE_BLOCK_SIZE - 8), &blowfish_key, ivec, BF_ENCRYPT);
    }

#ifdef DEBUG_ENCRYPTION
    memcpy(save_encrypted, buffer2, FILE_BLOCK_SIZE);
#endif

    write_block(ref, buffer2);
}



void nwos_write_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    size_t bytes_remaining;
    int i;
    int seq = 0;

    memset(ivec, 0, sizeof(ivec));


    assert(sizeof(extra) == FILE_BLOCK_SIZE);

    assert(!nwos_block_used(ref));

    ptr_obj = (uint8*) object;

    assert(ptr_obj[0] == 0 && ptr_obj[1] == 0 && ptr_obj[2] == 0 && ptr_obj[3] == 0);

    ptr_obj += 4;

    assert(is_same_object(ref, (ObjRef*)ptr_obj));

    ptr_obj += 4;

    bytes_remaining = size;

    extra.flags[0] = 0;
    extra.flags[1] = 0;
    extra.flags[2] = 0;
    extra.flags[3] = 0;

    copy_reference(&extra.id, ref);

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    extra.data[i] = *ptr_obj++;
	    bytes_remaining--;
	}

	nwos_generate_new_id(&extra.next_block);

	nwos_write_object_to_disk_and_encrypt(&extra.id, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);

	copy_reference(&extra.id, &extra.next_block);

	seq++;
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) extra.data[i] = *ptr_obj++;

    nwos_write_object_to_disk_and_encrypt(&extra.id, &extra, bytes_remaining, ivec, nwos_random_sequence[seq]);
}


void nwos_overwrite_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    uint8 ivec[IVEC_SIZE];

    assert(nwos_object_exists(ref));

    memset(ivec, 0, sizeof(ivec));

    nwos_write_object_to_disk_and_encrypt(ref, object, size, ivec, nwos_random_sequence[0]);
}


void nwos_write_public_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    uint8 block[FILE_BLOCK_SIZE];
//    int i;

    assert(size <= FILE_BLOCK_SIZE);
    assert((ref->id[0] & 0xF0) == 0);

    memset(block, 0, sizeof(block));

    memcpy(block, object, size);

    write_block(ref, object);

//    printf("write_public_object: %02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
}


/***************/
/* Class stuff */
/***************/

void nwos_get_object_class(ObjRef* obj, ObjRef* object_class)
{
    EveryObject header;
    uint8 ivec[IVEC_SIZE];

    assert(!is_void_reference(obj));

    memset(ivec, 0, sizeof(ivec));

    nwos_read_object_from_disk_and_decrypt(obj, &header, sizeof(header), ivec, nwos_random_sequence[0]);

    check_common_header(obj, &header.common);

    /* if it is not a reference list check the header checksum */

    if (!is_same_object(&header.common.class_definition, &nwos_reference_list_class_ref))
    {
	check_object_header(obj, &header.object, header.common.header_chksum);
    }

    memcpy(object_class, &header.common.class_definition, sizeof(object_class));
}


void nwos_read_class_definition(ObjRef* ref, C_struct_Class_Definition* class_def)
{
    assert(!is_void_reference(ref));

    nwos_read_object_from_disk(ref, class_def, sizeof(*class_def));
}


void nwos_fill_in_common_header(CommonHeader* common, ObjRef* ref, ObjRef* class_definition_ref)
{
    memset(common, 0, sizeof(CommonHeader));
    memcpy(&common->id, ref, sizeof(ObjRef));
    nwos_get_time_stamp(common->creation_time);
    memcpy(&common->class_definition, class_definition_ref, sizeof(common->class_definition));
}



/*************************************************/
/* Remove and object (by writing zero's over it) */
/*************************************************/

void nwos_remove_object(ObjRef* ref)
{
    char log_msg[128];
    uint8 block[FILE_BLOCK_SIZE];
    off64_t offset;

    assert(nwos_object_exists(ref));

    sprintf(log_msg, "nwos_remove_object - %02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
    nwos_log(log_msg);

    memset(block, 0, sizeof(block));

    assert(sizeof(block) == FILE_BLOCK_SIZE);

    offset = ref_to_offset(ref);
    if (lseek64(obj_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	sprintf(log_msg, "remove_object lseek64 ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

    if (write(obj_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	sprintf(log_msg, "nwos_remove_object write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

    modified = true;

    assert(!nwos_object_exists(ref));
}

#if 0
void create_object(uint64 class_def, EveryObject* header)
{
    nwos_fill_in_common_header(&header->common, 'l', class_def);

    memset(&header->object, 0, sizeof(ObjectHeader));  /* zero it out */

    nwos_create_reference_list(&header->object.references);

    nwos_crc32_calculate((uint8*) &header->object, sizeof(ObjectHeader), header->common.header_chksum);
}
#endif



bool nwos_is_quit_command(char* command)
{
    if (strcasecmp(command, "quit") == 0) return true;
    if (strcasecmp(command, "quit now") == 0) return true;
    if (strcasecmp(command, "exit") == 0) return true;
    if (strcasecmp(command, "stop") == 0) return true;
    if (strcasecmp(command, "stop now") == 0) return true;
    if (strcasecmp(command, "done") == 0) return true;
    if (strcasecmp(command, "done now") == 0) return true;
    if (strcasecmp(command, "finish") == 0) return true;
    if (strcasecmp(command, "finish now") == 0) return true;
    if (strcasecmp(command, "finished") == 0) return true;
    if (strcasecmp(command, "terminate") == 0) return true;

    return false;
}


bool nwos_ask_yes_or_no(char* statement, char* question)
{
    char buffer[16];

    if (statement != NULL)
    {
	printf("%s.  %s? (y/n): ", statement, question);
    }
    else
    {
	printf("%s? (y/n): ", question);
    }

    while (1)
    {
	fflush(stdout);
	fgets(buffer, sizeof(buffer), stdin);
	printf("\n");

	if (strchr(buffer, '\n') == NULL)  /* too much input */
	{
	    while(strchr(buffer, '\n') == NULL) fgets(buffer, sizeof(buffer), stdin);
	    buffer[0] = '\0';
	}

	if (strcasecmp(buffer, "y\n") == 0 || strcasecmp(buffer, "yes\n") == 0) break;

	if (strcasecmp(buffer, "n\n") == 0 || strcasecmp(buffer, "no\n") == 0) break;

	printf("Please answer the question: %s (yes or no)\n", question);
    }

    return (tolower(buffer[0]) == 'y');
}


/* since Linux or GNU don't seem to have this function */

#ifdef linux
extern size_t strlcpy(char* dst, char* src, size_t len)
{
    int i;

    for (i = 0; src[i] != '\0'; i++)
    {
	if (i < len)
	{
	    dst[i] = src[i];
	}
    }

    if (i < len)
    {
	dst[i] = src[i];
    }

    return i;
}

#endif
