/*
 * Copyright (C) 2006, 2009 Christophe Fergeau
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/*
 * This file was created from the libgpod-0.8.0/tests/test-cp.cc file and 
 * modified to obtain the tracks from an Objectify archive.
 *
 * Copyright (C) 2011  J. Scott Edwards
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <objectify.h>

#include <glib/gstdio.h>

#include <tag_c.h>

#include "itdb.h"

static Itdb_Track* track_from_file(const char *filename)
{
	TagLib_File* file;
	TagLib_Tag* tag;
	const TagLib_AudioProperties *audio_properties;
	Itdb_Track *track;
	struct stat st;

	file = taglib_file_new(filename);

	if (file == NULL)
	{
		g_print("Couldn't read tag file\n");
		return NULL;
	}

	tag = taglib_file_tag(file);

	if (tag == NULL)
	{
		taglib_file_free(file);
		g_print("Couldn't read tag\n");
		
		return NULL;
	}

	audio_properties = taglib_file_audioproperties(file);

/*	if (typeid(*tag) != typeid(TagLib::ID3v2::Tag))
	{
		g_print("Unknown tag type\n");
		return NULL;
		}*/

	/* FIXME: what happens if we try to open a non-MP3 file? */
#ifdef VERBOSE
	g_print("%s:\n", filename);
	g_print("\t%s\n", taglib_tag_artist(tag));
	g_print("\t%s\n", taglib_tag_album(tag));
	g_print("\t%s\n", taglib_tag_title(tag));
	g_print("\t%s\n", taglib_tag_genre(tag));
	g_print("\t%d\n", taglib_tag_year(tag));
	g_print("\t%d\n", taglib_tag_track(tag));
	g_print("\n");
#endif
	track = itdb_track_new();
	track->title = g_strdup(taglib_tag_title(tag));
	track->album = g_strdup(taglib_tag_album(tag));
	track->artist = g_strdup(taglib_tag_artist(tag));
	track->genre = g_strdup(taglib_tag_genre(tag));
	track->comment = g_strdup(taglib_tag_comment(tag));
	track->filetype = g_strdup("MP3-file");
	if (g_stat(filename, &st) == 0)
	{
		track->size = st.st_size;
	}
	track->track_nr = taglib_tag_track(tag);
	track->tracklen = taglib_audioproperties_length(audio_properties) * 1000;
	track->bitrate = taglib_audioproperties_bitrate(audio_properties);
	track->samplerate = taglib_audioproperties_samplerate(audio_properties);
	track->year = taglib_tag_year(tag);

	taglib_file_free(file);
	return track;
}


static gboolean copy_file(Itdb_iTunesDB *db, const char *filename, GError **error)
{
	Itdb_Track *track;
	gboolean copy_success;

	track = track_from_file(filename);
	itdb_track_add(db, track, -1);
	itdb_playlist_add_track(itdb_playlist_mpl(db), track, -1);
	copy_success = itdb_cp_track_to_ipod(track, filename, error);
	if (!copy_success)
	{
		itdb_track_free(track);
	}
	return copy_success;
}


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}




int main(int argc, char **argv)
{
	Itdb_iTunesDB *db;
	GError *error;
	ObjRef path_class_ref;
	ObjRef path_ref;
	ObjRef assoc_ref;
	ObjRef object_class;
	C_struct_Class_Definition class_def_obj;
	ReferenceList* ref_list;
	int num_refs;
	char tmp_path[20];
	uint8 kludge[MAX_PATH_OBJ_SIZE];
	C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    char name[256];
	int i;
	int j;
	int k;
	bool match;


	if (argc < 3)
	{
		g_print("Usage: %s <mountpoint> <filenames>\n", g_basename(argv[0]));
		exit(1);
	}

	db = itdb_parse(argv[1], NULL);

	if (db == NULL)
	{
		g_print("Error creating iPod database\n");
		exit(1);
	}

	nwos_initialize_objectify(READ_ONLY, NULL);

	if (!nwos_find_private_class_definition("FILE PATH", &path_class_ref))
	{
		printf("No files are stored in the system!\n");
		nwos_terminate_objectify();
		exit(1);
	}

	nwos_read_class_definition(&path_class_ref, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	printf("num_refs: %d\n", num_refs);

	for (i = 2; i < argc; i++)
	{
		match = false;

		for (j = 0; j < num_refs; j++)
		{
			nwos_get_object_class(&ref_list->references[j], &object_class);

			if (is_same_object(&object_class, &path_class_ref))
			{
				assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[j], kludge, sizeof(kludge), &get_path_object_size));

				/* remember ptr_path_obj points to the kludge buffer */

				for (k = 0; k < ptr_path_obj->count; k++) name[k] = ptr_path_obj->storage[k];
				name[k] = '\0';

				if (fnmatch(argv[i], name, 0) == 0)   /* found a match */
				{
					copy_reference(&path_ref, &ref_list->references[j]);

					printf("Path ref: %08x\n", nwos_ref_to_word(&path_ref));

					assert(nwos_file_path_to_path_and_file_association(&path_ref, 0, &assoc_ref));

					printf("Assoc ref: %08x\n", nwos_ref_to_word(&assoc_ref));

					snprintf(tmp_path, sizeof(tmp_path), "/tmp/%08x.mp3", nwos_ref_to_word(&assoc_ref));

					if (nwos_restore_file(&assoc_ref, tmp_path))
					{
						printf("Adding %s\n", name);

						error = NULL;
						if (!copy_file(db, tmp_path, &error))
						{
							g_print("Error copying music files\n");
							if (error && error->message)
							{
								g_print("%s\n", error->message);
								g_error_free(error);
							}
							break;
						}

						if (unlink(tmp_path) != 0)
						{
							g_print("Unable to remove temporary file: %s\n", tmp_path);
						}
					}

					match = true;
				}
			}
		}

		if (!match)
		{
			printf("No file(s) matching '%s' found.\n", argv[i]);
		}
	}

	itdb_write(db, &error);
	if (error)
	{
	    g_print("Error writing iPod database\n");
	    if (error->message)
	    {
		g_print("%s\n", error->message);
	    }
	    g_error_free(error);
	}

	itdb_free(db);

	nwos_terminate_objectify();

	return 0;
}
