/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-10-08 22:32:02 -0600 (Fri, 08 Oct 2010) $
--   $Revision: 4786 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <fnmatch.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "objectify.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


int main(int argc, char* argv[])
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Disc_Copy copy_obj;
    C_struct_Storage_Location location_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_File file_obj;
    C_struct_Person person_obj;
    uint8 cludge[MAX_PATH_OBJ_SIZE];
    C_struct_Disc_List* ptr_list_obj;
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)cludge;
    ObjRef list_class_ref;
    ObjRef copy_class_ref;
    ObjRef path_class_ref;
    ObjRef loc_class_ref;
    ObjRef person_class_ref;
    ObjRef name_ref;
    ObjRef object_class;
    ReferenceList* list_class_ref_list;
    ReferenceList* disc_list_ref_list;
    char name_str[128];
    char id_str[13];
    uint32 file_size;
    int list_class_num_refs;
    int disc_list_num_refs;
    int argn = 1;
    int i;
    int j;
    int count;
    int copies;
    bool do_contents = false;
    bool warning_printed = false;
    bool match;
    bool8* found_args = NULL;


    nwos_log_arguments(0, NULL);   /* disable logging */

    for (i = 1; i < argc; i++)
    {
	if (strcmp(argv[i], "--contents") == 0)
	{
	    if (i != 1)
	    {
		fprintf(stderr, "--content option must preceed any disc ids\n");
		exit(1);
	    }

	    do_contents = true;
	    argn = 2;
	}
	else if (*argv[i] == '-')
	{
	    fprintf(stderr, "unknown option: %s\n", argv[i]);
	    fprintf(stderr, "usage: %s [--contents] [disc id(s)]\n", argv[0]);
	    exit(1);
	}
	else if (found_args == NULL)    /* need to allocate array for flags */
	{
	    found_args = malloc(argc);

	    if (found_args == NULL)
	    {
		perror("allocating found_args array");
		exit(1);
	    }

	    memset(found_args, 0, argc);
	}
    }

    nwos_initialize_objectify(READ_ONLY, DEFAULT_FILE);

    fputc('\n', stdout);

    if (!nwos_find_private_class_definition("FILE PATH", &path_class_ref))
    {
	fprintf(stderr, "No private files found.\n");
	exit(1);
    }

    if (!nwos_find_private_class_definition("DISC LIST", &list_class_ref))
    {
	fprintf(stderr, "No private disc lists found.\n");
	exit(1);
    }

    if (!nwos_find_private_class_definition("DISC COPY", &copy_class_ref))
    {
	fprintf(stderr, "WARNING: no private disc copies found.\n");
	void_reference(&copy_class_ref);
	warning_printed = true;
    }

    if (!nwos_find_private_class_definition("STORAGE LOCATION", &loc_class_ref))
    {
	fprintf(stderr, "WARNING: no locations found.\n");
	void_reference(&loc_class_ref);
	warning_printed = true;
    }

    if (!nwos_find_private_class_definition("PERSON", &person_class_ref))
    {
	fprintf(stderr, "WARNING: no persons found.\n");
	void_reference(&person_class_ref);
	warning_printed = true;
    }

    if (warning_printed)     /* add some white space to make it easier to see */
    {
	fputc('\n', stderr);
    }

    nwos_read_class_definition(&list_class_ref, &class_def_obj);

    list_class_ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    list_class_num_refs = list_class_ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    ptr_list_obj = nwos_malloc(MAX_SIZE_DISC_LIST);

    for (i = 0; i < list_class_num_refs; i++)
    {
	nwos_get_object_class(&list_class_ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &list_class_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&list_class_ref_list->references[i], ptr_list_obj, MAX_SIZE_DISC_LIST, &nwos_get_disc_list_object_size));

	    memcpy(id_str, ptr_list_obj->id, 12);
	    id_str[12] = '\0';

	    match = true;   /* list all files if no ids were given on command line */

	    if (argn < argc)    /* disc ids were passed to the program on the command line */
	    {
		match = false;

		for (j = argn; j < argc; j++)
		{
		    if (fnmatch(argv[j], id_str, 0) == 0)   /* found a match */
		    {
			match = true;
			found_args[j] = true;
			break;
		    }
		}
	    }

	    if (match)   /* found a match */
	    {
		count = nwos_decode_variable_sized_count(ptr_list_obj->count);

		printf("disc: %s  files: %u\n", id_str, count);

		disc_list_ref_list = nwos_malloc_reference_list(&ptr_list_obj->header.object.references);

		disc_list_num_refs = disc_list_ref_list->common_header.num_refs;

		copies = 0;

		for (j = 0; j < disc_list_num_refs; j++)
		{
		    nwos_get_object_class(&disc_list_ref_list->references[j], &object_class);

		    if (!is_void_reference(&copy_class_ref) && is_same_object(&object_class, &copy_class_ref))
		    {
			copies++;

			assert(nwos_read_object_from_disk(&disc_list_ref_list->references[j], &copy_obj, sizeof(copy_obj)));

			nwos_get_object_class(&copy_obj.location, &object_class);

			if (is_void_reference(&copy_obj.header.object.next_version))
			{
			    if (is_same_object(&object_class, &loc_class_ref))
			    {
				assert(nwos_read_object_from_disk(&copy_obj.location, &location_obj, sizeof(location_obj)));
				copy_reference(&name_ref, &location_obj.name);
			    }
			    else if (is_same_object(&object_class, &person_class_ref))
			    {
				assert(nwos_read_object_from_disk(&copy_obj.location, &person_obj, sizeof(person_obj)));
				if (is_void_reference(&person_obj.goes_by))
				{
				    copy_reference(&name_ref, &person_obj.first_name);
				}
				else
				{
				    copy_reference(&name_ref, &person_obj.goes_by);
				}
			    }
			    else
			    {
				fprintf(stderr, "Error: unknown location class: %02x%02x%02x%02x\n",
					copy_obj.location.id[0],
					copy_obj.location.id[1],
					copy_obj.location.id[2],
					copy_obj.location.id[3]);
			    }

			    assert(nwos_name_to_string(&name_ref, name_str, sizeof(name_str)));

			    printf(" copy number: %d  location: %s\n", copy_obj.copy_number, name_str);
			}
		    }
		}

		nwos_free_reference_list(disc_list_ref_list);
		disc_list_ref_list = NULL;

		if (copies == 0)
		{
		    printf("  NO copies logged!\n");
		}

		if (do_contents)    /* list all the files as well */
		{
		    for (j = 0; j < count; j++)
		    {
			assert(nwos_read_object_from_disk(&ptr_list_obj->files[j], &assoc_obj, sizeof(assoc_obj)));

			assert(nwos_read_variable_sized_object_from_disk(&assoc_obj.path, cludge, sizeof(cludge), &get_path_object_size));

			memcpy(name_str, ptr_path_obj->storage, ptr_path_obj->count);
			name_str[ptr_path_obj->count] = '\0';

			assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));

			file_size = (file_obj.size[0] << 24) | (file_obj.size[1] << 16) | (file_obj.size[2] << 8) | (file_obj.size[3]);

			printf("   %s %u\n", name_str, file_size);
		    }
		}
	    }
	}
    }

    printf("\n");

    nwos_free(ptr_list_obj);
    ptr_list_obj = NULL;

    nwos_free_reference_list(list_class_ref_list);
    list_class_ref_list = NULL;

    nwos_terminate_objectify();

    if (found_args != NULL)
    {
	match = true;      /* see if they all matched */

	for (i = argn; i < argc; i++)
	{
	    if (!found_args[i])
	    {
		fprintf(stderr, "WARNING: no IDs found that match: %s\n", argv[i]);
		match = false;
	    }
	}

	if (!match)
	{
	    fprintf(stderr, "\n");
	}

	free(found_args);
	found_args = NULL;
    }

    return 0;
}


