/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-21 06:49:28 -0700 (Thu, 21 Jan 2010) $
--   $Revision: 4473 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "class_definition.h"


bool reference_list_has_reference(ObjRef* ref_list_ref, ObjRef* ref)
{
    ReferenceList* ref_list;
    int num_refs;
    int i;

    ref_list = nwos_malloc_reference_list(ref_list_ref);

    num_refs = ref_list->common_header.num_refs;

    for (i = 0; i < num_refs; i++)
    {
	if (is_same_object(&ref_list->references[i], ref)) break;
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_refs;    /* return true if found it */
}

void dump_reference_list(ObjRef* ref_list_ref)
{
    ReferenceList* ref_list;
    int num_refs;
    int i;

    ref_list = nwos_malloc_reference_list(ref_list_ref);

    num_refs = ref_list->common_header.num_refs;

    for (i = 0; i < num_refs; i++)
    {
	printf("   %08x\n", nwos_ref_to_word(&ref_list->references[i]));
    }
}


static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);

    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

void check_spelling(ObjRef* spelling_ref)
{
    int i;
    char buffer[64];
    uint8 kludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* spelling_obj_ptr = (C_struct_Spelling*)kludge;
#if 0
    size_t size;
    ObjRef ref;
    ReferenceList* ref_list;
    int num_refs;
#endif

    printf("Spelling: %08x ", nwos_ref_to_word(spelling_ref));
    fflush(stdout);

    assert(nwos_read_variable_sized_object_from_disk(spelling_ref, kludge, sizeof(kludge), &get_spelling_object_size));

    for (i = 0; i < spelling_obj_ptr->count; i++) buffer[i] = spelling_obj_ptr->storage[i];
    buffer[i] = '\0';

    printf(": %s\n", buffer);

    if (!is_void_reference(&spelling_obj_ptr->character_set))
    {
	printf("WARNING: character_set in Spelling object is non-void: %08x\n", nwos_ref_to_word(&spelling_obj_ptr->character_set));
    }
}


static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);

    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}

void check_name(ObjRef* name_ref)
{
    int i;
    uint8 kludge[MAX_NAME_OBJ_SIZE];
    C_struct_Name* name_obj_ptr = (C_struct_Name*)kludge;
    uint8 cludge[MAX_SPELLING_OBJ_SIZE];
    C_struct_Spelling* spelling_obj_ptr = (C_struct_Spelling*)cludge;
#if 0
    size_t size;
    ObjRef ref;
#endif

    printf("Name: %08x\n", nwos_ref_to_word(name_ref));
    fflush(stdout);

    assert(nwos_read_variable_sized_object_from_disk(name_ref, kludge, sizeof(kludge), &get_name_object_size));

    for (i = 0; i < name_obj_ptr->count; i++)
    {
	printf("  %08x\n", nwos_ref_to_word(&name_obj_ptr->spelling[i]));

	assert(nwos_read_variable_sized_object_from_disk(&name_obj_ptr->spelling[i], cludge, sizeof(cludge), &get_spelling_object_size));

	if (!reference_list_has_reference(&spelling_obj_ptr->header.object.references, name_ref))
	{
	    printf("ERROR: name object not in spelling reference list: %08x\n",
		   nwos_ref_to_word(&spelling_obj_ptr->header.object.references));
	    dump_reference_list(&spelling_obj_ptr->header.object.references);
	}
    }
}



void check_month(ObjRef* month_ref)
{
    C_struct_Month month_obj;
    char number[4];
#if 0
    int i;
    size_t size;
    ObjRef ref;
    ReferenceList* ref_list;
    int num_refs;
#endif

    printf("Month: %08x\n", nwos_ref_to_word(month_ref));

    assert(nwos_read_object_from_disk(month_ref, &month_obj, sizeof(month_obj)));

    nwos_ordinal_number_to_string(&month_obj.number, number, sizeof(number));

    printf("Number: %s\n", number);

    if (!is_void_reference(&month_obj.definition))
    {
	printf("WARNING: definition in Month object is non-void: %08x\n", nwos_ref_to_word(&month_obj.definition));
    }
}


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}

void check_file_path(ObjRef* file_path_ref)
{
    int i;
    char buffer[64];
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* file_path_obj_ptr = (C_struct_File_Path*)kludge;
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef assoc_class_ref;
    ObjRef object_class;
#if 0
    size_t size;
    ObjRef ref;
#endif
    ReferenceList* ref_list;
    int num_refs;

    printf("File Path: %08x ", nwos_ref_to_word(file_path_ref));
    fflush(stdout);

    assert(nwos_read_variable_sized_object_from_disk(file_path_ref, kludge, sizeof(kludge), &get_path_object_size));

    for (i = 0; i < file_path_obj_ptr->count; i++) buffer[i] = file_path_obj_ptr->storage[i];
    buffer[i] = '\0';

    printf(": %s\n", buffer);

    ref_list = nwos_malloc_reference_list(&file_path_obj_ptr->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("Number of refrences: %d\n", num_refs);

    assert(nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    if (!is_same_object(file_path_ref, &assoc_obj.path))
	    {
		printf("ERROR: file path object not in path and file assoc. object: %08x\n",
		       nwos_ref_to_word(&assoc_obj.path));
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}


	// Need this for something?
#if 0
    if (!reference_list_has_reference(&file_obj.header.object.references, file_path_ref))
    {
	printf("ERROR: file path object not in file reference list: %08x\n",
	       nwos_ref_to_word(&file_obj.header.object.references));
	dump_reference_list(&file_obj.header.object.references);
    }
#endif


void check_class(ObjRef* class_ref)
{
    int i;
#if 0
    size_t size;
    ObjRef ref;
#endif
    char buffer[64];
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*) kludge;
    ReferenceList* ref_list;
    int num_refs;

    printf("Reading class definition: %08x\n", nwos_ref_to_word(class_ref));

    assert(nwos_read_variable_sized_object_from_disk(class_ref, kludge, sizeof(kludge), &nwos_get_object_size));

    nwos_name_to_string(&class_def_obj_ptr->name, buffer, sizeof(buffer));

    printf("Class name: %s\n", buffer);

    if (!is_same_object(&class_def_obj_ptr->header.common.class_definition, &nwos_private_class_definition_class_ref))
    {
	printf("Class is wrong: %08x  should be: %08x\n",
	       nwos_ref_to_word(&class_def_obj_ptr->header.common.class_definition),
	       nwos_ref_to_word(&nwos_private_class_definition_class_ref));
    }
    else if (strcasecmp(buffer, "REFERENCE LIST") != 0)    /* reference lists don't have reference lists */
    {
	printf("Reading reference list: %08x\n", nwos_ref_to_word(&class_def_obj_ptr->header.object.references));
	fflush(stdout);

	ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

	num_refs = ref_list->common_header.num_refs;

	printf("Number of refrences: %d\n", num_refs);

	for (i = 0; i < num_refs; i++)
	{
	    if (strcasecmp(buffer, "SPELLING") == 0)
	    {
		check_spelling(&ref_list->references[i]);
	    }
	    else if (strcasecmp(buffer, "NAME") == 0)
	    {
		check_name(&ref_list->references[i]);
	    }
	    else if (strcasecmp(buffer, "MONTH") == 0)
	    {
		check_month(&ref_list->references[i]);
	    }
	    else if (strcasecmp(buffer, "FILE PATH") == 0)
	    {
		check_file_path(&ref_list->references[i]);
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }
}


int main(int argc, char* argv[])
{
    int i;
#if 0
    size_t size;
    ObjRef ref;
#endif
    char buffer[64];
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*) kludge;
    ReferenceList* ref_list;
    int num_refs;


    nwos_initialize_objectify(READ_ONLY, DEFAULT_FILE);

    printf("Reading class definition class: %08x\n", nwos_ref_to_word(&nwos_private_class_definition_class_ref));

    assert(nwos_read_variable_sized_object_from_disk(&nwos_private_class_definition_class_ref, kludge, sizeof(kludge), &nwos_get_object_size));

    nwos_name_to_string(&class_def_obj_ptr->name, buffer, sizeof(buffer));


    printf("Class name: %s\n", buffer);

    printf("Reading class definition class reference list: %08x\n", nwos_ref_to_word(&class_def_obj_ptr->header.object.references));

    ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("Number of refrences: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	check_class(&ref_list->references[i]);
    }

    nwos_free_reference_list(ref_list);

    return 0;
}

