/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-10-06 07:25:14 -0600 (Wed, 06 Oct 2010) $
--   $Revision: 4785 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <string.h>

#include "md5.h"
#include "sha1.h"
#include "sha256.h"
#include "sha512.h"

#include "crc32.h"
#include "objectify.h"


/*-------------------------------------------------------------------------------------------------------------------*/
/* MD5 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_md5(bool public, uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef md5_class_ref;
    ObjRef temp_ref;
    Sorted_Reference_List* sorted_list;
    C_struct_MD5sum* md5_object;
    int index;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if the md5sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("MD5SUM", &md5_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("MD5SUM", &md5_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("MD5SUM", &md5_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&md5_class_ref, &class_def_obj);

    md5_object = nwos_malloc_temporary_object(sizeof(*md5_object), &temp_ref);

    nwos_fill_in_common_header(&md5_object->header.common, &temp_ref, &md5_class_ref);

    memcpy(md5_object->md5sum, md5sum, sizeof(md5_object->md5sum));

    nwos_crc32_calculate((uint8*) &md5_object->header.object, sizeof(ObjectHeader), md5_object->header.common.header_chksum);
    nwos_crc32_calculate((uint8*) &md5_object->md5sum, sizeof(C_struct_MD5sum) - sizeof(EveryObject), md5_object->header.common.data_chksum);

    sorted_list = nwos_get_sorted_reference_list(&class_def_obj.header.object.references, &md5_class_ref);

    index = nwos_find_object_in_sorted_reference_list(sorted_list, &temp_ref);   /* returns -1 if not found */

    if (index > -1)
    {
	copy_reference(ref, &sorted_list->references[index]);
    }

    nwos_free_temporary_object(md5_object);

    return index > -1;
}


bool nwos_find_public_md5(uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    return find_md5(true, md5sum, ref);
}


/* find an md5 based upon what mode we are in */

bool nwos_find_md5(uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    return find_md5(nwos_in_public_mode(), md5sum, ref);
}


/* Find existing MD5sum or create new */

ObjCreateResult nwos_create_md5(uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef md5_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_MD5sum md5_object;
#ifdef VERIFY_WRITE
    C_struct_MD5sum* ptr_md5_obj;
#endif
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this md5 */

    if (!nwos_find_md5(md5sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("MD5SUM", &md5_class_ref));
#else
	nwos_find_or_create_private_class_definition("MD5SUM", &md5_class_ref);
#endif

	memset(&md5_object, 0, sizeof(md5_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&md5_object.header.common, ref, &md5_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_md5(md5sum, &public_ref))
	{
	    copy_reference(&md5_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "md5sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(md5_object.md5sum, md5sum, sizeof(md5_object.md5sum));

	nwos_create_reference_list(ref, &md5_object.header.object.references);

	nwos_crc32_calculate((uint8*) &md5_object.header.object, sizeof(ObjectHeader), md5_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &md5_object.md5sum, sizeof(C_struct_MD5sum) - sizeof(EveryObject), md5_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &md5_object, sizeof(md5_object));

	nwos_add_to_references(ref, &md5_class_ref);

#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	ptr_md5_obj = malloc(sizeof(C_struct_MD5sum));
	assert(nwos_read_object_from_disk(ref, ptr_md5_obj, sizeof(C_struct_MD5sum)));
	assert(memcmp(&md5_object, ptr_md5_obj, sizeof(C_struct_MD5sum)) == 0);
	free(ptr_md5_obj);
	ptr_md5_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}


bool nwos_md5_less_than(C_struct_MD5sum* current, C_struct_MD5sum* other)
{
    int three_way_compare;

    assert(!is_same_object(&current->header.common.id, &other->header.common.id));    // verify different objects

    three_way_compare = memcmp(current->md5sum, other->md5sum, sizeof(current->md5sum));

    // if using a temporary object it could be equal    assert(three_way_compare != 0);   // two different MD5 objects should never be identical

    return three_way_compare < 0;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* SHA1 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_sha1(bool public, uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef sha1_class_ref;
    ObjRef temp_ref;
    Sorted_Reference_List* sorted_list;
    C_struct_SHA1sum* sha1_object;
    int index;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if the sha1sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("SHA1SUM", &sha1_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("SHA1SUM", &sha1_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("SHA1SUM", &sha1_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&sha1_class_ref, &class_def_obj);

    sha1_object = nwos_malloc_temporary_object(sizeof(*sha1_object), &temp_ref);

    nwos_fill_in_common_header(&sha1_object->header.common, &temp_ref, &sha1_class_ref);

    memcpy(sha1_object->sha1sum, sha1sum, sizeof(sha1_object->sha1sum));

    nwos_crc32_calculate((uint8*) &sha1_object->header.object, sizeof(ObjectHeader), sha1_object->header.common.header_chksum);
    nwos_crc32_calculate((uint8*) &sha1_object->sha1sum, sizeof(C_struct_SHA1sum) - sizeof(EveryObject), sha1_object->header.common.data_chksum);

    sorted_list = nwos_get_sorted_reference_list(&class_def_obj.header.object.references, &sha1_class_ref);

    index = nwos_find_object_in_sorted_reference_list(sorted_list, &temp_ref);   /* returns -1 if not found */

    if (index > -1)
    {
	copy_reference(ref, &sorted_list->references[index]);
    }

    nwos_free_temporary_object(sha1_object);

    return index > -1;
}


bool nwos_find_public_sha1(uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha1(true, sha1sum, ref);
}


/* find an sha1 based upon what mode we are in */

bool nwos_find_sha1(uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha1(nwos_in_public_mode(), sha1sum, ref);
}



/* Find existing SHA1sum or create new */

ObjCreateResult nwos_create_sha1(uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef sha1_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_SHA1sum sha1_object;
#ifdef VERIFY_WRITE
    C_struct_SHA1sum* ptr_sha1_obj;
#endif
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this sha1 */

    if (!nwos_find_sha1(sha1sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("SHA1SUM", &sha1_class_ref));
#else
	nwos_find_or_create_private_class_definition("SHA1SUM", &sha1_class_ref);
#endif

	memset(&sha1_object, 0, sizeof(sha1_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&sha1_object.header.common, ref, &sha1_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_sha1(sha1sum, &public_ref))
	{
	    copy_reference(&sha1_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "sha1sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(sha1_object.sha1sum, sha1sum, sizeof(sha1_object.sha1sum));

	nwos_create_reference_list(ref, &sha1_object.header.object.references);

	nwos_crc32_calculate((uint8*) &sha1_object.header.object, sizeof(ObjectHeader), sha1_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &sha1_object.sha1sum, sizeof(C_struct_SHA1sum) - sizeof(EveryObject), sha1_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &sha1_object, sizeof(sha1_object));

	nwos_add_to_references(ref, &sha1_class_ref);

#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	ptr_sha1_obj = malloc(sizeof(C_struct_SHA1sum));
	assert(nwos_read_object_from_disk(ref, ptr_sha1_obj, sizeof(C_struct_SHA1sum)));
	assert(memcmp(&sha1_object, ptr_sha1_obj, sizeof(C_struct_SHA1sum)) == 0);
	free(ptr_sha1_obj);
	ptr_sha1_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}


bool nwos_sha1_less_than(C_struct_SHA1sum* current, C_struct_SHA1sum* other)
{
    int three_way_compare;

    assert(!is_same_object(&current->header.common.id, &other->header.common.id));    // verify different objects

    three_way_compare = memcmp(current->sha1sum, other->sha1sum, sizeof(current->sha1sum));

    // if using a temporary object it could be equal    assert(three_way_compare != 0);   // two different SHA1 objects should never be identical

    return three_way_compare < 0;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* SHA256 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_sha256(bool public, uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef sha256_class_ref;
    ObjRef temp_ref;
    Sorted_Reference_List* sorted_list;
    C_struct_SHA256sum* sha256_object;
    int index;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if the sha256sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("SHA256SUM", &sha256_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("SHA256SUM", &sha256_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("SHA256SUM", &sha256_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&sha256_class_ref, &class_def_obj);

    sha256_object = nwos_malloc_temporary_object(sizeof(*sha256_object), &temp_ref);

    nwos_fill_in_common_header(&sha256_object->header.common, &temp_ref, &sha256_class_ref);

    memcpy(sha256_object->sha256sum, sha256sum, sizeof(sha256_object->sha256sum));

    nwos_crc32_calculate((uint8*) &sha256_object->header.object, sizeof(ObjectHeader), sha256_object->header.common.header_chksum);
    nwos_crc32_calculate((uint8*) &sha256_object->sha256sum, sizeof(C_struct_SHA256sum) - sizeof(EveryObject), sha256_object->header.common.data_chksum);

    sorted_list = nwos_get_sorted_reference_list(&class_def_obj.header.object.references, &sha256_class_ref);

    index = nwos_find_object_in_sorted_reference_list(sorted_list, &temp_ref);   /* returns -1 if not found */

    if (index > -1)
    {
	copy_reference(ref, &sorted_list->references[index]);
    }

    nwos_free_temporary_object(sha256_object);

    return index > -1;
}


bool nwos_find_public_sha256(uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha256(true, sha256sum, ref);
}


/* find an sha256 based upon what mode we are in */

bool nwos_find_sha256(uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha256(nwos_in_public_mode(), sha256sum, ref);
}




/* Find existing SHA256sum or create new */

ObjCreateResult nwos_create_sha256(uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef sha256_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_SHA256sum sha256_object;
#ifdef VERIFY_WRITE
    C_struct_SHA256sum* ptr_sha256_obj;
#endif
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this sha256 */

    if (!nwos_find_sha256(sha256sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("SHA256SUM", &sha256_class_ref));
#else
	nwos_find_or_create_private_class_definition("SHA256SUM", &sha256_class_ref);
#endif

	memset(&sha256_object, 0, sizeof(sha256_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&sha256_object.header.common, ref, &sha256_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_sha256(sha256sum, &public_ref))
	{
	    copy_reference(&sha256_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "sha256sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(sha256_object.sha256sum, sha256sum, sizeof(sha256_object.sha256sum));

	nwos_create_reference_list(ref, &sha256_object.header.object.references);

	nwos_crc32_calculate((uint8*) &sha256_object.header.object, sizeof(ObjectHeader), sha256_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &sha256_object.sha256sum, sizeof(C_struct_SHA256sum) - sizeof(EveryObject), sha256_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &sha256_object, sizeof(sha256_object));

	nwos_add_to_references(ref, &sha256_class_ref);

#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	ptr_sha256_obj = malloc(sizeof(C_struct_SHA256sum));
	assert(nwos_read_object_from_disk(ref, ptr_sha256_obj, sizeof(C_struct_SHA256sum)));
	assert(memcmp(&sha256_object, ptr_sha256_obj, sizeof(C_struct_SHA256sum)) == 0);
	free(ptr_sha256_obj);
	ptr_sha256_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}


bool nwos_sha256_less_than(C_struct_SHA256sum* current, C_struct_SHA256sum* other)
{
    int three_way_compare;

    assert(!is_same_object(&current->header.common.id, &other->header.common.id));    // verify different objects

    three_way_compare = memcmp(current->sha256sum, other->sha256sum, sizeof(current->sha256sum));

    // if using a temporary object it could be equal    assert(three_way_compare != 0);   // two different SHA256 objects should never be identical

    return three_way_compare < 0;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* SHA512 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_sha512(bool public, uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef sha512_class_ref;
    ObjRef temp_ref;
    Sorted_Reference_List* sorted_list;
    C_struct_SHA512sum* sha512_object;
    int index;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if the sha512sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("SHA512SUM", &sha512_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("SHA512SUM", &sha512_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("SHA512SUM", &sha512_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&sha512_class_ref, &class_def_obj);

    sha512_object = nwos_malloc_temporary_object(sizeof(*sha512_object), &temp_ref);

    nwos_fill_in_common_header(&sha512_object->header.common, &temp_ref, &sha512_class_ref);

    memcpy(sha512_object->sha512sum, sha512sum, sizeof(sha512_object->sha512sum));

    nwos_crc32_calculate((uint8*) &sha512_object->header.object, sizeof(ObjectHeader), sha512_object->header.common.header_chksum);
    nwos_crc32_calculate((uint8*) &sha512_object->sha512sum, sizeof(C_struct_SHA512sum) - sizeof(EveryObject), sha512_object->header.common.data_chksum);

    sorted_list = nwos_get_sorted_reference_list(&class_def_obj.header.object.references, &sha512_class_ref);

    index = nwos_find_object_in_sorted_reference_list(sorted_list, &temp_ref);   /* returns -1 if not found */

    if (index > -1)
    {
	copy_reference(ref, &sorted_list->references[index]);
    }

    nwos_free_temporary_object(sha512_object);

    return index > -1;
}


bool nwos_find_public_sha512(uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha512(true, sha512sum, ref);
}


/* find an sha512 based upon what mode we are in */

bool nwos_find_sha512(uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha512(nwos_in_public_mode(), sha512sum, ref);
}




/* Find existing SHA512sum or create new */

ObjCreateResult nwos_create_sha512(uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef sha512_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_SHA512sum sha512_object;
#ifdef VERIFY_WRITE
    C_struct_SHA512sum* ptr_sha512_obj;
#endif
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this sha512 */

    if (!nwos_find_sha512(sha512sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("SHA512SUM", &sha512_class_ref));
#else
	nwos_find_or_create_private_class_definition("SHA512SUM", &sha512_class_ref);
#endif

	memset(&sha512_object, 0, sizeof(sha512_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&sha512_object.header.common, ref, &sha512_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_sha512(sha512sum, &public_ref))
	{
	    copy_reference(&sha512_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "sha512sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(sha512_object.sha512sum, sha512sum, sizeof(sha512_object.sha512sum));

	nwos_create_reference_list(ref, &sha512_object.header.object.references);

	nwos_crc32_calculate((uint8*) &sha512_object.header.object, sizeof(ObjectHeader), sha512_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &sha512_object.sha512sum, sizeof(C_struct_SHA512sum) - sizeof(EveryObject), sha512_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &sha512_object, sizeof(sha512_object));

	nwos_add_to_references(ref, &sha512_class_ref);

#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	ptr_sha512_obj = malloc(sizeof(C_struct_SHA512sum));
	assert(nwos_read_object_from_disk(ref, ptr_sha512_obj, sizeof(C_struct_SHA512sum)));
	assert(memcmp(&sha512_object, ptr_sha512_obj, sizeof(C_struct_SHA512sum)) == 0);
	free(ptr_sha512_obj);
	ptr_sha512_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}


bool nwos_sha512_less_than(C_struct_SHA512sum* current, C_struct_SHA512sum* other)
{
    int three_way_compare;

    assert(!is_same_object(&current->header.common.id, &other->header.common.id));    // verify different objects

    three_way_compare = memcmp(current->sha512sum, other->sha512sum, sizeof(current->sha512sum));

    // if using a temporary object it could be equal    assert(three_way_compare != 0);   // two different SHA512 objects should never be identical

    return three_way_compare < 0;
}


