/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-10-08 22:32:02 -0600 (Fri, 08 Oct 2010) $
--   $Revision: 4786 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <ctype.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "objectify.h"

#define MAX_STORAGE_LOCATIONS 48  /* for now */

static char* file_names[MAX_FILES_PER_DISC_LIST];
static ObjRef files[MAX_FILES_PER_DISC_LIST];


static void get_number(char* buffer, size_t bufsize)
{
    if (fgets(buffer, bufsize, stdin) == NULL)
    {
	fputc('\n', stderr);

	if (feof(stdin))
	{
	    fprintf(stderr, "ERROR: received end of input from standard input when reading number\n");
	    exit(1);
	}
	else
	{
	    perror("reading number");
	    exit(1);
	}
    }
}



int main(int argc, char* argv[])
{
    ObjRef ref;
    int i;
    char id[13];
    char copy_num[3] = "";
    char binder_num[4] = "";
    char name[24];
    char *p;
    bool ok = false;
    int binder = -2;
    uint32 num_files;
    ObjRef location_class_ref;
    ObjRef object_class;
    C_struct_Class_Definition class_def_obj;
    ReferenceList* ref_list;
    int num_refs;
    ObjRef location_ref[MAX_STORAGE_LOCATIONS];
    int num_locations = 0;
    C_struct_Storage_Location loc_obj;
    ObjRef disc_list_ref;
    ObjRef person_ref;
    bool missing_files = false;
    char msg[128];
#if 0
    bool bad_checksum = false;
    CheckFileResult check_file_result;
#endif
    C_struct_Disc_List* ptr_list_obj;
    uint32 error_mask = DISC_LIST_ERROR_MASK;
    int argi = 1;

    while (argi < argc - 1)
    {
	if (strcmp(argv[argi], "--ignore-empty-directories") == 0)
	{
	    error_mask &= ~DISC_LIST_EMPTY_DIR_FLAG;
	    argi++;
	}
	else if (strcmp(argv[argi], "--ignore-empty-files") == 0)
	{
	    error_mask &= ~DISC_LIST_EMPTY_FILE_FLAG;
	    argi++;
	}
    }

    if (argi != argc - 1)
    {
	fprintf(stderr, "usage: %s [--ignore-empty-directories] [--ignore-empty-files] directory\n", argv[0]);
	fprintf(stderr, "    --ignore-empty-directories  ignore any empty directories\n");
	fprintf(stderr, "    --ignore-empty-files        ignore any empty files\n");
	exit(1);
    }

    printf("\n");


    /******************************************************/
    /* First open the disc and read all of the file names */
    /******************************************************/

    num_files = nwos_read_files_disc_list(argv[argi], NULL, file_names, 0);    /* NULL for subdirectory indicates this is the root */

    if ((num_files & DISC_LIST_NUM_FILES_MASK) == 0)
    {
	fprintf(stderr, "\nERROR: Disc is empty!  Perhaps it is not mounted?\n\n");

	exit(16);  /* 16 means directory is empty */
    }

    if ((num_files & error_mask) != 0)
    {
	fprintf(stderr, "\n");

	exit(num_files >> DISC_LIST_ERROR_SHIFT);  /* exit code is error code returned by nwos_read_files_disc_list 1-15 */
    }

    num_files &= DISC_LIST_NUM_FILES_MASK;

    printf("\n");

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(READ_WRITE, DEFAULT_FILE);

    ptr_list_obj = nwos_malloc(MAX_SIZE_DISC_LIST);

    if (!nwos_check_blocks_available(14))   /* 14 if new storage location is created, normally just 2 */
    {
	fprintf(stderr, "Cannot create new storage location!\n");
	nwos_terminate_objectify();
	exit(1);
    }

    if (nwos_find_private_class_definition("STORAGE LOCATION", &location_class_ref))
    {
	nwos_read_class_definition(&location_class_ref, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &location_class_ref))
	    {
		copy_reference(&location_ref[num_locations], &ref_list->references[i]);

		num_locations++;

		if (num_locations == MAX_STORAGE_LOCATIONS)
		{
		    fprintf(stderr, "WARNING: more than 40 storage locations, remainder skipped\n");
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }
    else
    {
	fprintf(stderr, "Note: no storage locations are defined!\n");
    }

    /* first verify that all the listed files are already in the system */

    memset(files, 0, sizeof(files));

    for (i = 0; i < num_files; i++)
    {
	printf("%s: ", file_names[i]);
	fflush(stdout);

	if (!nwos_find_matching_path_and_file_association(argv[argi], file_names[i], &files[i], MatchAll))
	{
	    missing_files = true;
	    printf("  <--- NOT found");
	}

	printf("\n");
    }

    if (missing_files)
    {
	printf("The following files are not in the system:\n\n");

	for (i = 0; i < num_files; i++)
	{
	    if (is_void_reference(&files[i]))
	    {
		printf("   %s\n", file_names[i]);
	    }
	}

	printf("\n");
	exit(1);
    }


    /* find and verify the disc */

    if (!nwos_find_matching_disc_list(files, &disc_list_ref))
    {
	printf("No matching disc list found!\n");
	exit(1);
    }

    assert(nwos_read_variable_sized_object_from_disk(&disc_list_ref, ptr_list_obj, MAX_SIZE_DISC_LIST, &nwos_get_disc_list_object_size));

    memcpy(id, ptr_list_obj->id, sizeof(id) - 1);
    id[sizeof(id) - 1] = '\0';

    snprintf(msg, sizeof(msg), "disc list %02x%02x%02x%02x: %s",
	   disc_list_ref.id[0], disc_list_ref.id[1], disc_list_ref.id[2], disc_list_ref.id[3], id);
    nwos_log(msg);
    printf("%s\n", msg);

    /* create record of this disc */

    ok = false;
    while (!ok)
    {
	printf("Copy number: ");
	fflush(stdout);

	get_number(copy_num, sizeof(copy_num));

	p = strchr(copy_num, '\n');   /* find the newline char */

	if (p == NULL)    /* line was too long */
	{
	    while (p == NULL)   /* keep reading until we find the newline char */
	    {
		get_number(copy_num, sizeof(copy_num));
		p = strchr(copy_num, '\n');   /* find the newline char */
	    }
	    printf("copy number can only be 1 digit in this version\n");
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (strlen(copy_num) != 1)
	    {
		printf("copy number can only be 1 digit in this version\n");
	    }
	    else if (copy_num[0] < '1' || copy_num[0] > '9')
	    {
		printf("copy number must be a 1 digit number in this version\n");
	    }
	    else
	    {
		ok = true;   /* we should be good to go */
	    }
	}
    }

    ok = false;
    while (!ok)
    {
	printf("Available locations:\n\n");

	printf("   0=create new\n");

	for (i = 0; i < num_locations; i++)
	{
	    assert(nwos_read_object_from_disk(&location_ref[i], &loc_obj, sizeof(loc_obj)));
	    nwos_name_to_string(&loc_obj.name, name, sizeof(name));
	    printf("   %d=%s\n", i + 1, name);
	}

	printf("   P=person\n\n");

	printf("Selection: ");
	fflush(stdout);

	get_number(binder_num, sizeof(binder_num));

	p = strchr(binder_num, '\n');   /* find the newline char */

	if (p == NULL)    /* line was too long */
	{
	    while (p == NULL)    /* keep reading until we find the newline char */
	    {
		get_number(binder_num, sizeof(binder_num));
		p = strchr(binder_num, '\n');   /* find the newline char */
	    }
	    printf("storage binder can only be 1 or 2 digits in this version\n");
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (toupper(binder_num[0]) == 'P' && binder_num[1] == '\0')
	    {
		binder = -1;   /* flag it is a person */
		break;
	    }
	    else if (binder_num[0] == '0' && binder_num[1] == '\0')      /* create a new name */
	    {
		nwos_ask_user("name", name, sizeof(name));

		if (nwos_create_storage_location(name, &location_ref[num_locations]) == CREATED_NEW)
		{
		    binder = num_locations;
		    num_locations++;
		    break;
		}
		else
		{
		    fprintf(stderr, "Note: a storage location with that name already exists.\n");
		}
	    }
	    else if (isdigit(binder_num[0]) && binder_num[1] == '\0')    /* single non-zero digit entered */
	    {
		binder = binder_num[0] - '1';

		ok = true;   /* we should be good to go */
	    }
	    else if (isdigit(binder_num[0]) && binder_num[0] != '0' && isdigit(binder_num[1]) && binder_num[2] == '\0')   /* two digits entered */
	    {
		binder = (binder_num[0] - '0') * 10 + binder_num[1] - '0' - 1;

		if (binder < num_locations)
		{
		    ok = true;   /* we should be good to go */
		}
		else
		{
		    printf("Please enter a number less than or equal to %d\n", num_locations);
		}
	    }
	    else
	    {
		printf("Please enter a number between 0 and %d or 'P'.\n", num_locations);
	    }
	}
    }

    assert(-1 <= binder && binder < num_locations);

    if (binder == -1)
    {
	nwos_ask_user("name", name, sizeof(name));

	if (nwos_find_person(name, &person_ref))
	{
	    assert(nwos_create_disc_copy(&disc_list_ref, copy_num[0] - '0', &person_ref, &ref) == CREATED_NEW);
	}
	else
	{
	    fprintf(stderr, "%s not found, terminating without creating record\n", name);
	}
    }
    else
    {
	if (nwos_create_disc_copy(&disc_list_ref, copy_num[0] - '0', &location_ref[binder], &ref) == CREATED_NEW)
	{
	    printf("created new: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
	}
	else
	{
	    printf("updated: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
	}
    }

    nwos_free(ptr_list_obj);
    ptr_list_obj = NULL;

    for (i = 0; i < num_files; i++)
    {
	free(file_names[i]);
    }

    nwos_terminate_objectify();

    return 0;
}


